package com.flybits.commons.library.api;

import android.content.Context;
import android.content.SharedPreferences;
import android.support.annotation.NonNull;

import com.flybits.commons.library.FlyingConstants;
import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.deserializations.DeserializeLogin;
import com.flybits.commons.library.deserializations.DeserializeServerResult;
import com.flybits.commons.library.exceptions.FlybitsDisabledException;
import com.flybits.commons.library.exceptions.InvalidFlybitsRequestException;
import com.flybits.commons.library.exceptions.InvalidFlybitsResponseException;
import com.flybits.commons.library.http.DeleteRequest;
import com.flybits.commons.library.http.GetRequest;
import com.flybits.commons.library.http.PersistentCookieStore;
import com.flybits.commons.library.http.PostRequest;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.Device;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.commons.library.models.internal.ServerResult;
import com.flybits.commons.library.models.User;
import com.flybits.commons.library.utils.filters.LoginOptions;
import com.flybits.commons.library.utils.filters.RegisterOptions;

import java.io.IOException;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.util.HashMap;


/**
 * <p>This class is responsible for authentication include the Flybits core. Part of the login process
 * will automatically register the user to the Flybits core XMPP server using the retrieved XMPP
 * username, password, and XMPP server location. Logging into the XMPP server is an automatic
 * process so that developers do not need to concern themselves include XMPP registration.
 * </p>
 * <p>The methods of this call are completely static and handle all login/logout processes between
 * the application and the Flybits core.
 * </p>
 *
 * 	@author Petar Kramaric
 *	@version 2.0
 */
public class FlyAuthentication {

    public static Result addRememberMe(Context context) throws InvalidFlybitsResponseException, FlybitsDisabledException {

        String url = FlybitsAPIConstants.constructURL(FlybitsAPIConstants.API_REMEMBERME);

        try {
            ServerResult result = new GetRequest(context, url, null).getResponse();
            Result info = new DeserializeServerResult().fromJson(result.response);
            if (info == null) {
                info = new Result();
            }

            info.responseAsString = result.response;
            if (result.status >= 200 && result.status < 300) {
                info.responseStatus = RequestStatus.SUCCESS;
                info.exception      = null;
            } else if (result.status == 404){
                info.responseStatus = RequestStatus.NOT_FOUND;
            } else {
                info.responseStatus = RequestStatus.FAILED;
            }

            return info;
        } catch (IOException | NullPointerException e1) {
            Logger.instance(FlyingConstants.IS_DEBUG)._LOG_EXCEPTION(e1);
            clearUserInformation(context);
            throw new InvalidFlybitsResponseException("Error Connecting to Flybits Server: FlyException_Auth1");
        }
    }

    //DONE
    public static Result changePassword(Context context, String oldPassword, String newPassword) throws InvalidFlybitsResponseException, InvalidFlybitsRequestException, FlybitsDisabledException {

        String json = new ChangePassword(oldPassword, newPassword).toJson();
        String url = FlybitsAPIConstants.constructURL(FlybitsAPIConstants.API_CHANGE_PASSWORD);

        try {
            ServerResult result = new PostRequest(context, false, url, json, null).getResponse();
            Result info = new DeserializeServerResult().fromJson(result.response);
            if (info == null) {
                info = new Result();
            }

            if (result.status >= 200 && result.status < 300) {
                info.responseStatus = RequestStatus.SUCCESS;
                info.exception      = null;
            }else if (result.status == 404){
                info.responseStatus = RequestStatus.NOT_FOUND;
            } else {
                info.responseStatus = RequestStatus.FAILED;
            }

            info.responseAsString = result.response;
            return info;
        } catch (IOException | NullPointerException e) {
            Logger.instance(FlyingConstants.IS_DEBUG)._LOG_EXCEPTION(e);
            throw new InvalidFlybitsResponseException("Error Connecting to Flybits Server: FlyException_Auth2");
        }
    }

    public static Result disable(Context context) throws InvalidFlybitsResponseException, FlybitsDisabledException {

        String urlEndpoint = FlybitsAPIConstants.API_USER;
        String url = FlybitsAPIConstants.constructURL(urlEndpoint);

        try {
            ServerResult result = new DeleteRequest(context, false, url, null).getResponse();
            Result info = new DeserializeServerResult().fromJson(result.response);
            if (info == null) {
                info = new Result();
            }

            if (result.status >= 200 && result.status < 300) {
                info.responseStatus = RequestStatus.SUCCESS;
                info.exception      = null;
                clearUserInformation(context);
            }else if (result.status == 404){
                info.responseStatus = RequestStatus.NOT_FOUND;
            }  else {
                info.responseStatus = RequestStatus.FAILED;
            }

            info.responseAsString = result.response;
            return info;
        } catch (IOException | NullPointerException e) {
            Logger.instance(FlyingConstants.IS_DEBUG)._LOG_EXCEPTION(e);
            throw new InvalidFlybitsResponseException("Error Connecting to Flybits Server: FlyException_Auth3");
        }
    }

    //DONE
    public static Result forgotPasswordAPI(Context context, String email) throws InvalidFlybitsResponseException, InvalidFlybitsRequestException, FlybitsDisabledException {

        String url = FlybitsAPIConstants.constructURL(FlybitsAPIConstants.API_FORGOT_PASSWORD);
        String json = new Email(email).toJson();

        try {
            ServerResult result = new PostRequest(context, false, url, json, null).getResponse();
            Result info = new DeserializeServerResult().fromJson(result.response);
            if (info == null) {
                info = new Result();
            }
            info.responseAsString = result.response;

            if (result.status >= 200 && result.status < 300) {
                info.responseStatus = RequestStatus.SUCCESS;
                info.exception      = null;
            }else if (result.status == 404){
                info.responseStatus = RequestStatus.NOT_FOUND;
            }else{
                info.responseStatus = RequestStatus.FAILED;
            }

            return info;
        } catch (IOException | NullPointerException e) {
            Logger.instance(FlyingConstants.IS_DEBUG)._LOG_EXCEPTION(e);
            throw new InvalidFlybitsResponseException("Error Connecting to Flybits Server: FlyException_Auth4");
        }
    }

    //DONE
    public static Result<User> login(Context context, @NonNull LoginOptions filter) throws InvalidFlybitsResponseException, InvalidFlybitsRequestException, FlybitsDisabledException {

        if (filter == null){
            throw new InvalidFlybitsRequestException("It appears that your LoginOptions parameter is null. This is not allowed.");
        }

        if (filter.getType() == LoginOptions.FilterBy.LOGIN_ANONYMOUSLY){
            return loginAnnonymously(context, filter);
        }
        return loginWithPreRegisteredUser(context, filter);
    }

    //DONE
    public static Result logoutAPI(Context context) throws InvalidFlybitsResponseException, FlybitsDisabledException, InvalidFlybitsRequestException {

        String url = FlybitsAPIConstants.constructURL(FlybitsAPIConstants.API_LOGOUT);

        try {
            ServerResult result = new PostRequest(context, false, url, new HashMap<String, String>(), null).getResponse();
            Result info = new DeserializeServerResult().fromJson(result.response);
            if (info == null) {
                info  = new Result();
            }

            info.responseAsString = result.response;
            if (result.status >= 200 && result.status < 300) {
                info.responseStatus = RequestStatus.SUCCESS;
                info.exception      = null;
                clearUserInformation(context);

            } else if (result.status == 404){
                info.responseStatus = RequestStatus.NOT_FOUND;
            } else if (info.exception != null && info.exception.exceptionType != null
                    && info.exception.exceptionType.equalsIgnoreCase(FlyingConstants.SERVER_ERROR_ACCESS_DENIED)) {

                info.responseStatus = RequestStatus.SUCCESS;
                clearUserInformation(context);
            }

            return info;
        } catch (IOException | NullPointerException e1) {
            clearUserInformation(context);
            throw new InvalidFlybitsResponseException("Error Connecting to Flybits Server: FlyException_Auth6");
        }
    }

    //DONE
    public static Result<User> registerUser(Context context, RegisterOptions options) throws InvalidFlybitsResponseException, InvalidFlybitsRequestException, FlybitsDisabledException {

        String url = FlybitsAPIConstants.constructURL(FlybitsAPIConstants.API_REGISTER);

        try {
            ServerResult result = new PostRequest(context, false, url, options.getRequest(), null).getResponse();
            Result<User> info = new DeserializeLogin().fromJson(result.response);

            if (info == null){
                info    = new Result<>();
            }
            info.responseAsString   = result.response;

            if (info.responseStatus != null && info.responseStatus == RequestStatus.SUCCESS) {
                if (options.isRetrieveRememberMeToken()) {
                    addRememberMe(context);
                }
                setUserInformation(context, info.response);
            }else if (result.status == 404){
                info.responseStatus = RequestStatus.NOT_FOUND;
            }

            return info;
        } catch (IOException | NullPointerException e) {
            Logger.instance(FlyingConstants.IS_DEBUG)._LOG_EXCEPTION(e);
            throw new InvalidFlybitsResponseException("Error Connecting to Flybits Server: FlyException_Auth7");
        }
    }

    static boolean setUserInformation(Context context, @NonNull User user) {

        if (user != null && user.id != null) {
            SharedElements.setUserID(context, user.id);
            if (user.jwtToken != null) {
                SharedElements.setJWTToken(context, user.jwtToken);
            }

            try {
                Result<Device> responseResult = FlyDevice.addToDevices(context);
                if (responseResult.responseStatus == RequestStatus.SUCCESS
                        && responseResult.response != null
                            && responseResult.response.id != null) {
                    SharedElements.setDeviceID(context, responseResult.response.id);
                }else{
                    return false;
                }
            } catch (InvalidFlybitsResponseException | FlybitsDisabledException | InvalidFlybitsRequestException e) {
                Logger.instance(FlyingConstants.IS_DEBUG)._LOG_EXCEPTION(e);
                return false;
            }
            return true;

        }
        return false;
    }

    static void clearUserInformation(Context context) {

        SharedPreferences preferences = SharedElements.getPreferences(context);
        SharedPreferences.Editor editor = preferences.edit();
        editor.clear();
        editor.apply();

        PersistentCookieStore cookieStore = new PersistentCookieStore(context);
        cookieStore.removeAll();
    }

    //DONE
    private static Result<User> loginWithPreRegisteredUser(Context context, @NonNull LoginOptions filter) throws InvalidFlybitsResponseException, InvalidFlybitsRequestException, FlybitsDisabledException {

        try {
            String json = filter.getFilter();
            String url = FlybitsAPIConstants.constructURL(filter.getEndpoint());

            ServerResult result = new PostRequest(context, false, url, json, null).getResponse();

            Result<User> info = new DeserializeLogin().fromJson(result.response);
            info.responseAsString = result.response;

            if ((result.status >= 200 && result.status < 300) && info.responseStatus == RequestStatus.SUCCESS){

                setUserInformation(context, info.response);

            }else if (result.status >= 200 && result.status < 300){

                Result resultLogout = logoutAPI(context);
                if (resultLogout.responseStatus != RequestStatus.SUCCESS){
                    clearUserInformation(context);
                }

                info.responseStatus = RequestStatus.FAILED;
            }else if (result.status == 404){
                info.responseStatus = RequestStatus.NOT_FOUND;
            }

            return info;
        } catch (IOException | NullPointerException e) {
            Logger.instance(FlyingConstants.IS_DEBUG)._LOG_EXCEPTION(e);
            throw new InvalidFlybitsResponseException("Error Connecting to Flybits Server: FlyException_Auth5");
        }
    }

    //DONE
    private static Result<User> loginAnnonymously(Context context, @NonNull LoginOptions filter) throws InvalidFlybitsResponseException, InvalidFlybitsRequestException, FlybitsDisabledException {

        SecureRandom random = new SecureRandom();
        String email = new BigInteger(130, random).toString(32) + "@flybits.com";
        String password = new BigInteger(130, random).toString(32);

        RegisterOptions.Builder.Options options = new RegisterOptions.Builder()
                .setUserCredentials(email, password);

        if (!filter.isJWTSet()) {
            options.disableJWT();
        }
        if (!filter.isRememberMeSet()) {
            options.disableRememberMe();
        }

        return registerUser(context, options.build());
    }

    private static class Email {

        private String email;

        Email(String email) {
            this.email = email;
        }

        String toJson(){
            return "{\"email\" : \"" + email + "\"}";
        }
    }

    private static class ChangePassword {

        String currentPassword;
        String newPassword;

        ChangePassword(String currentPassword, String newPassword) {
            this.currentPassword = currentPassword;
            this.newPassword = newPassword;
        }

        String toJson(){
            return "{\"currentPassword\" : \"" + currentPassword + "\", \"newPassword\" : \"" + newPassword + "\"}";
        }
    }
}
