package com.flybits.internal

import android.content.Context
import androidx.annotation.VisibleForTesting
import com.flybits.commons.library.analytics.Analytics
import com.flybits.commons.library.api.results.BasicResult
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.internal.AnalyticsDebounceManager.interval_in_million_sec
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.FlowPreview
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.debounce
import kotlinx.coroutines.flow.emptyFlow
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.stateIn
import java.util.UUID
import java.util.concurrent.ConcurrentLinkedQueue

data class DebounceTask(val context: Context,
        val basicResult: BasicResult?,
        val uuid: String = UUID.randomUUID().toString())

@OptIn(FlowPreview::class)
/***
 * A Debouncer to flush Analytics and Context data to the backend. Any updated data within the same time window[interval_in_million_sec]
 * will only send one network request in batch.
 */
object AnalyticsDebounceManager {
    const val TAG = "AnalyticsDebounceManager"
    internal var interval_in_million_sec = 3000L

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    val debounceQueue: ConcurrentLinkedQueue<DebounceTask> = ConcurrentLinkedQueue()
    private val newValue = MutableStateFlow(false)
    val run = newValue.combine(newValue) { _, _ ->
        emptyFlow<Int>()
    }
            .debounce(interval_in_million_sec)
            .onEach {
                synchronized(debounceQueue) {
                    if (debounceQueue.isNotEmpty()) {
                        debounceQueue.let {
                            if (it.isNotEmpty()) {
                                var result = false
                                var error: Exception? = null

                                it.poll()?.let { debounceTask ->
                                    Analytics(debounceTask.context).flush({ success: Boolean, exception: Exception? ->
                                        debounceTask.basicResult?.apply {
                                            result = success
                                            error = exception
                                            if (success) {
                                                Logger.appendTag(TAG).d("Flush called successful.")
                                                setSuccess()
                                            } else {
                                                Logger.appendTag(TAG).d("Flush called not successful.")
                                                setFailed(FlybitsException(exception))
                                            }
                                        }

                                        while (debounceQueue.isNotEmpty()) {
                                            debounceQueue.poll()?.let { item ->
                                                item.basicResult.apply {
                                                    if (result) {
                                                        this?.setSuccess()
                                                    } else this?.setFailed(FlybitsException(error))
                                                }
                                            }
                                        }
                                    })
                                }
                            }
                        }
                    }
                }
            }.stateIn(CoroutineScope(Dispatchers.Default), SharingStarted.Eagerly, Any())

    /***
     * Push a [DebounceTask] to a queue
     */
    fun addTask(debounceTask: DebounceTask) {
        debounceQueue.add(debounceTask)
        Logger.appendTag(TAG).d("New task added, total task to " +
                "debounce: ${debounceQueue.size}")
        newValue.value = !newValue.value
    }
}