package com.flybits.commons.library.api.idps

import android.content.Context
import android.os.Parcel
import android.os.Parcelable
import com.flybits.commons.library.api.FlybitsConfiguration
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.exceptions.MissingProjectIDException
import com.flybits.commons.library.logging.Logger.appendTag
import com.flybits.commons.library.models.User
import org.json.JSONObject
import java.util.*

/**
 * The IDP class is the base class for all IDentity Providers (IDP) supported by Flybits. Through
 * this abstract class, developers can define their Identity Providers using the following pieces
 * of information:
 *
 *  * Provider - Which
 *  * URL Endpoint - The HTTP(S) route to login/register a user
 *  * HTTP Request Body - The POST body of the HTTP(S) request.
 *
 */
abstract class IDP : Parcelable {
    /**
     * Get the provider that indicates which IDP this class refers to. Examples can include
     * "Facebook", "Twitter", "Google", "Flybits", etc.
     *
     * @return The provider that indicates which IDP to connect to.
     */
    open val provider: String?

    /**
     * Constructor used to define the basic parameters including the `provider` and
     * `projectID`. These two pieces information are the base for all `IDP` classes.
     *
     * @param provider The provider that indicates which IDP this class refers to. Examples can
     * include "Facebook", "Twitter", "Google", "Flybits", etc.
     */
    constructor(provider: String?) {
        this.provider = provider
    }

    /**
     * Constructor used to define the basic parameters including the `provider` and
     * `projectID`. These two pieces information are the base for all `IDP` classes.
     *
     * @param in The parcel to un-marshall content associated to the IDP.
     */
    constructor(`in`: Parcel) {
        provider = `in`.readString()
    }

    /**
     * The URL Endpoint for the IDP to successfully login/register to.
     *
     * @return The URL Endpoint for login/register APIs.
     */
    abstract val authenticationEndPoint: String?

    /**
     * The Request Type for the IDP to connect to.
     *
     * @return The Request Type by default Post.
     */
    open fun getRequestType(): HttpMethod? {
        return HttpMethod.POST
    }

    /**
     * The custom headers needed for the [to perform the network operations.][empty]
     */
    open fun getHeaders(): HashMap<String, String>? {
        return HashMap()
    }

    /**
     * @return [JSONObject] body for network request
     */
    abstract fun getBody(): JSONObject?

    /**
     * Defines the operations once the IDP is connected
     *
     * @param context The current context of the application
     * @param user Represents user's details.
     *
     * Please Note: onAuthenticated() is not abstract, just to avoid any breaking change for the release 1.14.0.
     */
    open fun onAuthenticated(context: Context, user: User) {}

    override fun writeToParcel(out: Parcel, flags: Int) {
        out.writeString(provider)
    }

    companion object {
        private const val TAG_LOGGING = "IDP"

        /**
         * Get the unique identifier for your Flybits project.
         *
         * @return The unique identifier for your Flybits project.
         *
         * @throws MissingProjectIDException in case if it encounters Missing ID Exception
         */
        @JvmStatic
        @kotlin.jvm.Throws(MissingProjectIDException::class)
        fun getProjectID(): String {
            appendTag(TAG_LOGGING).i("Getting project ID")
            return FlybitsConfiguration.Builder.projectId
        }
    }
}