package com.flybits.commons.library.api.idps

import android.os.Parcel
import android.os.Parcelable
import com.flybits.commons.library.utils.Utilities
import org.json.JSONObject

/**
 * The `OAuthIDP` class is responsible for creating a specific IDentity Provider (IDP) that
 * is managed by any O-Auth based 3rd party integration such Facebook or Google. The
 * `OAuthIDP` creates a [com.flybits.commons.library.models.User] based on the
 * information retrieved from the O-Auth provider regardless of whether it is Facebook, Google, etc.
 */
class OAuthIDP : IDP {
    private var accessToken: String?
    private var clientID: String? = null
    override var provider: String?

    override val authenticationEndPoint: String
        get() = "/oauth"

    override fun getBody(): JSONObject {
        val bodyObject = JSONObject()

        Utilities.setValuesToBody("provider", provider, bodyObject)
        Utilities.setValuesToBody("accessToken", accessToken, bodyObject)
        if (clientID != null) {
            Utilities.setValuesToBody("clientID", clientID, bodyObject)
        }
        Utilities.setValuesToBody("projectId", getProjectID(), bodyObject)

        return bodyObject
    }

    private constructor(`in`: Parcel) : super(`in`) {
        accessToken = `in`.readString()
        clientID = `in`.readString()
        provider = `in`.readString()
    }

    /**
     * Constructor used to register an O-Auth identity provider that needs only the
     * `accessToken`. Currently, if you plan on supporting the Facebook or Google IDP, this
     * constructor should be used.
     *
     * @param provider The provider which indicates which IDP the `accessToken` and
     * `clientID` are associated to.
     * @param accessToken The `accessToken` that is retrieved from the 3rd party O-Auth
     * provider.
     */
    constructor(provider: String, accessToken: String) : super(provider) {
        this.provider = provider
        this.accessToken = accessToken
    }

    /**
     * Constructor used to register an O-Auth identity provider that needs only the
     * `accessToken`. Currently, if you plan on supporting the Facebook or Google IDP, this
     * constructor should be used.
     *
     * @param idp The [SupportedIDP] that indicates the provider that the `accessToken`
     * should be associated to.
     * @param accessToken The `accessToken` that is retrieved from the 3rd party O-Auth
     * provider.
     */
    constructor(idp: SupportedIDP, accessToken: String) : this(idp.key, accessToken) {}

    /**
     * Constructor used to register an O-Auth identity provider that needs both an
     * `accessToken`, as well as a `clientID` in order to successfully connect to the
     * IDP.
     *
     * @param provider The provider which indicates which IDP the `accessToken` and
     * `clientID` are associated to.
     * @param accessToken The `accessToken` that is retrieved from the 3rd party O-Auth
     * provider.
     * @param clientID The `clientID` refers to the project that is registered with the 3rd
     * party O-Auth provider for this specific application instance of the SDK.
     */
    constructor(provider: String, accessToken: String, clientID: String?) : this(provider, accessToken) {
        this.clientID = clientID
    }

    /**
     * Constructor used to register an O-Auth identity provider that needs only the
     * `accessToken`. Currently, if you plan on supporting the [SupportedIDP.FACEBOOK]
     * or [SupportedIDP.GOOGLE], this constructor should be used.
     *
     * @param idp The [SupportedIDP] that indicates the provider that the `accessToken`
     * should be associated to.
     * @param accessToken The `accessToken` that is retrieved from the 3rd party O-Auth
     * provider.
     * @param clientID The `clientID` refers to the project that is registered with the 3rd
     * party O-Auth provider for this specific application instance of the SDK.
     */
    constructor(idp: SupportedIDP, accessToken: String, clientID: String?) : this(idp.key, accessToken, clientID) {}

    /**
     * This enumerator indicates which O-Auth based Identity Providers are supported by Flybits. It
     * is possible for this enumerator to be outdated depending on how often you update this SDK. If
     * the IDP that you want to use is listed here you should use the
     * [OAuthIDP.OAuthIDP] otherwise if you believe that Flybits uses
     * accepts another IDP that is not listed here, you should use the more generic constructor
     * [OAuthIDP.OAuthIDP] or [OAuthIDP.OAuthIDP].
     */
    enum class SupportedIDP
    /**
     * Constructor that defines the key for each `SupportedIDP` option.
     *
     * @param key the string value representing each `SupportedIDP` option.
     */(
            /**
             * Get the string representation for the `SupportedIDP` option.
             *
             * @return string representation of the `SupportedIDP` option.
             */
            val key: String) {
        /**
         *
         */
        OKTA("okta"),

        /**
         *
         */
        FACEBOOK("facebook"),

        /**
         *
         */
        GOOGLE("gplus");

        companion object {
            /**
             * Get the `SupportedIDP` enum value corresponding to an String representation.
             *
             * @param key the string representation of the `SupportedIDP` enum.
             *
             * @return The `SupportedIDP` enum for the String representation.
             */
            fun fromKey(key: String?): SupportedIDP? {
                for (type in values()) {
                    if (type.key.equals(key, ignoreCase = true)) {
                        return type
                    }
                }
                return null
            }
        }
    }

    override fun describeContents(): Int {
        return 0
    }

    override fun writeToParcel(out: Parcel, flags: Int) {
        super.writeToParcel(out, flags)
        out.writeString(accessToken)
        out.writeString(clientID)
        out.writeString(provider)
    }

    companion object {
        @JvmField val CREATOR: Parcelable.Creator<OAuthIDP?> = object : Parcelable.Creator<OAuthIDP?> {
            override fun createFromParcel(`in`: Parcel): OAuthIDP {
                return OAuthIDP(`in`)
            }

            override fun newArray(size: Int): Array<OAuthIDP?> {
                return arrayOfNulls(size)
            }
        }
    }
}