package com.flybits.commons.library.api.idps

import android.content.Context
import android.os.Parcel
import android.os.Parcelable
import android.util.Base64
import com.flybits.commons.library.utils.Utilities
import org.json.JSONException
import org.json.JSONObject
import java.io.UnsupportedEncodingException

/**
 * The `SignedIDP` class is responsible for creating the specification for an IDentity Provider
 * (IDP) that can be logged into using the SAML protocol. The `SignedIDP` creates a
 * [com.flybits.commons.library.models.User] based on the information retrieved from the
 * SAML IDP regardless of where the IDP is hosted.
 */
class SignedIDP : IDP {
    private var accessToken: String?
    private var signature: String?

    private constructor(`in`: Parcel) : super(`in`) {
        accessToken = `in`.readString()
        signature = `in`.readString()
    }

    /**
     * Constructor used to register to an identity provider using a signed access token that
     * requires a confirm signature used to sign the `accessToken`.
     *
     * @param accessToken The access token obtained from the 3rd party source.
     * @param signature The access token signature used to validate the access token.
     */
    constructor(accessToken: String?, signature: String?) : super("signedLogin") {
        this.accessToken = accessToken
        this.signature = signature
    }

    override fun describeContents(): Int {
        return 0
    }

    override val authenticationEndPoint: String
        get() = "/signedLogin"

    override fun getBody(): JSONObject {
        val bodyObject = JSONObject()
        Utilities.setValuesToBody("accessToken", accessToken, bodyObject)
        Utilities.setValuesToBody("signature", signature, bodyObject)
        return bodyObject
    }

    override fun writeToParcel(out: Parcel, flags: Int) {
        super.writeToParcel(out, flags)
        out.writeString(accessToken)
        out.writeString(signature)
    }

    companion object {

        @JvmField val CREATOR: Parcelable.Creator<SignedIDP?> = object : Parcelable.Creator<SignedIDP?> {
            override fun createFromParcel(`in`: Parcel): SignedIDP? {
                return SignedIDP(`in`)
            }

            override fun newArray(size: Int): Array<SignedIDP?> {
                return arrayOfNulls(size)
            }
        }

        fun base64Encode(context: Context?, email: String?): String? {
            val bodyObject = JSONObject()
            try {
                bodyObject.put("projectId", getProjectID())
            } catch (e: JSONException) {
            }
            try {
                bodyObject.put("email", email)
            } catch (e: JSONException) {
            }
            return try {
                val body = bodyObject.toString()
                val data = body.toByteArray(charset("UTF-8"))
                Base64.encodeToString(data, Base64.NO_WRAP)
            } catch (ex: UnsupportedEncodingException) {
                null
            }
        }
    }
}