package com.flybits.commons.library.api

import android.content.Context
import androidx.annotation.VisibleForTesting
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.models.User
import com.flybits.commons.library.utils.getAuthenticationState
import java.util.*
import java.util.concurrent.ConcurrentHashMap
import kotlin.collections.set
import kotlin.properties.ReadWriteProperty
import kotlin.reflect.KProperty

/**
 *  Responsible for [FlybitsScope] Management for [FlybitsManager].
 */
internal object ScopeOfficer {

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal var authState by Synchronize(ScopeAuthState.INITIATED)
    private val lock = Any()

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal var firstRun: Boolean

    @JvmStatic
    var flybitsScopes: ConcurrentHashMap<String, FlybitsScope> = ConcurrentHashMap()

    init {
        // Since context cannot receive context, we make sure authState does not get set
        // multiple times by adding scopes, but only once.
        firstRun = true
    }

    /**
     * Add a [FlybitsScope] to the [FlybitsManager]. The [FlybitsScope]
     * defines the SDK and all the properties associated to it.
     *
     * @param scope The [FlybitsScope] that is associated to this application.
     * @return A [Builder] object where additional [FlybitsScope] attributes can
     * be set.
     **/
    @JvmStatic
    fun addScope(scope: FlybitsScope, context: Context) {
        Logger.i("Adding Scope ${scope.scopeName}")
        if (firstRun) {
            Logger.i("ScopeOfficer detected an addScope with Auth State $authState")
            authState = getAuthenticationState(context)
            firstRun = false
        }
        flybitsScopes[scope.scopeName] = scope
        Logger.i("ScopeOfficer detected an onStart")
        scope.onStart(authState)
        Logger.i("... calling onStart of ${scope.scopeName}")
    }

    /**
     * Remove a [FlybitsScope] to the [FlybitsManager]. The [FlybitsScope]
     * defines the SDK and all the properties associated to it.
     *
     * @param scope The [FlybitsScope] that is associated to this application.
     * @return A [Builder] object where additional  [FlybitsManager] attributes can
     * be set.
     **/
    @JvmStatic
    fun removeScope(scope: FlybitsScope) {
        Logger.i("Removing Scope ${scope.scopeName}...")
        val it = flybitsScopes.entries.iterator()
        while (it.hasNext()) {
            val item = it.next()
            if (item.key == scope.scopeName) {
                it.remove()
                Logger.i("ScopeOfficer detected an onStop")
                scope.onStop()
                Logger.i("Removing Scope ${scope.scopeName} ... Success")
            }
        }
    }

    /**
     * Clear all scopes [FlybitsScope] added to [FlybitsManager]. The [FlybitsScope]
     * defines the SDK and all the properties associated to it.
     *
     **/
    @JvmStatic
    internal fun clearScopes() {
        Logger.i("Clearing all scopes")
        flybitsScopes.clear()
        authState = ScopeAuthState.INITIATED
        firstRun = true
    }

    /**
     * Indicates that the login was successfully made and that the appropriate actions for an SDK
     * should be performed.
     *
     * @param context The current Context of the application
     * @param user The logged in [User].
     */
    fun onConnected(context: Context, user: User) {
        Logger.i("ScopeOfficer detected an onConnected")
        authState = getAuthenticationState(context)
        val it = flybitsScopes.keys.iterator()
        while (it.hasNext()) {
            val item = it.next()
            Logger.i("... calling onConnected of $item")
            flybitsScopes[item]?.onConnected(context, user)
        }
    }

    /**
     * Indicates that the logout was successful. The SDK will need to make the appropriate actions.
     *
     * @param context The current Context of the application.
     */
    fun onDisconnected(context: Context) {
        Logger.i("ScopeOfficer detected an onDisconnected")
        authState = getAuthenticationState(context)
        val it = flybitsScopes.keys.iterator()
        while (it.hasNext()) {
            val item = it.next()
            Logger.i("... calling onDisconnected of $item")
            flybitsScopes[item]?.onDisconnected(context)
        }
    }

    /**
     * Indicates [FlybitsManager.connect] gets initiated.
     */
    fun onConnecting() {
        Logger.i("ScopeOfficer detected an onConnecting")
        authState = ScopeAuthState.CONNECTING
        val it = flybitsScopes.keys.iterator()
        while (it.hasNext()) {
            val item = it.next()
            Logger.i("... calling onConnecting of $item")
            flybitsScopes[item]?.onConnecting()
        }
    }

    /**
     * Indicates the logged in [User] has changed the opted state.
     *
     * @param context The current Context of the application
     * @param optedState True if the user opted in, false if the user opted out.
     */
    fun onOptedStateChange(context: Context, optedState: Boolean) {
        Logger.i("ScopeOfficer detected an onOptedStateChange")
        authState = getAuthenticationState(context)
        val it = flybitsScopes.keys.iterator()
        while (it.hasNext()) {
            val item = it.next()
            Logger.i("... calling onOptedStateChange of $item")
            flybitsScopes[item]?.onOptedStateChange(context, optedState)
        }
    }

    fun onFailedToConnect(context: Context) {
        Logger.i("ScopeOfficer detected an onOptedStateChange")
        authState = getAuthenticationState(context)
        val it = flybitsScopes.keys.iterator()
        while (it.hasNext()) {
            val item = it.next()
            Logger.i("... calling onFailedToConnect of $item")
            flybitsScopes[item]?.onFailedToConnect()
        }
    }

    /**
     * Use delegates to add synchronization on the [ScopeAuthState] property getter/setter with the [lock].
     */
    private class Synchronize<ScopeAuthState>(defaultValue: ScopeAuthState) :
            ReadWriteProperty<Any, ScopeAuthState> {
        private var backField = defaultValue
        override fun getValue(thisRef: Any, property: KProperty<*>): ScopeAuthState {
            return synchronized(lock) {
                backField
            }
        }

        override fun setValue(thisRef: Any, property: KProperty<*>,
                value: ScopeAuthState) {
            return synchronized(lock) {
                backField = value
            }
        }
    }
}