package com.flybits.commons.library

import android.content.Context
import android.content.SharedPreferences
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.models.PushProviderType
import com.flybits.commons.library.models.User
import com.flybits.commons.library.utils.Utilities
import com.flybits.internal.db.UserDAO
import com.google.gson.Gson
import kotlinx.coroutines.*
import kotlinx.coroutines.Dispatchers.IO

/**
 * This class is used to get and set all variables that can be shared across the various Flybits'
 * Android SDKs. Currently the items that can be shared between the SDKs are as follows;
 *
 *  * Language Information
 *  * Device Identifier
 *  * User Identifier
 *  * JWT
 *  * Project ID
 *  * Connected IDP
 *
 *  @param sharedPreferences [SharedPreferences] that data will be stored and retrieved from.
 *  @param userDAO [UserDAO] that user data will be retrieved from.
 *
 */
abstract class SharedElements internal constructor(
        protected val sharedPreferences: SharedPreferences, private val userDAO: UserDAO,
        dispatcher: CoroutineDispatcher = newSingleThreadContext("SharedElements$IO")) {

    private var jwtToken: String? = null
    private var languageCodes: String? = null
    private var idp: String? = null
    private var notificationChannelID: String? = null
    private var isPushNotificationEnabled: Boolean = false
    private var userId: String? = null
    private var locationPermission: String? = null
    private var user: User? = null
    private var mPushProviderType: PushProviderType? = null

    internal val job = CoroutineScope(SupervisorJob() + dispatcher)

    private var gateway: String = ""

    companion object {
        internal val TAG = SharedElements::class.java.simpleName

        const val PREF_LANGUAGE_CODES = "com.flybits.language.codes"
        const val PREF_JWT_TOKEN = "com.flybits.jwt.token"
        const val PREF_IDP_CONNECTED = "com.flybits.idp.connected"
        const val PREF_NOTIFICATION_CHANNEL_ID = "com.flybits.idp.channel_id"
        const val PREF_IS_NOTIFICATION_ENABLED = "com.flybits.idp.isPushNotificationEnabled"
        const val PREF_GATEWAY_URL = "com.flybits.project.url"
        const val PREF_UNIQUE_ID = "com.flybits.device.unique_id"
        const val PREF_USER_ID = "com.flybits.user.id"
        const val PREF_LOCATION_PERMISSION = "com.flybits.location.permission"
        const val PREF_USER = "com.flybits.user"
        const val PREF_IS_TOKEN_SENT = "com.flybits.push.isTokenSent"
        const val KEY_IS_TOKEN_SENT_UNKNOWN = "unknown"
        const val KEY_IS_TOKEN_SENT_SENT = "sent"
        const val KEY_IS_TOKEN_SENT_DELETED = "deleted"
        const val PREF_PUSH_PROVIDER_TYPE = "com.flybits.push.provider.type"

        const val PREF_SET_PUSH_PERMISSION_FIRST_TIME = "com.flybits.push.permissionForFirstTime"

        const val FLYBITS_STORAGE_UNENCRYPTED_V0 = "FLYBITS_PREF"        // used before encryption existed
        const val FLYBITS_STORAGE_UNENCRYPTED_V1 = "FLYBITS_PREF_BACKUP" // previously used as a backup from encryption
        const val FLYBITS_STORAGE_UNENCRYPTED_V2 = "FLYBITS_PREF"        // now used as backup from encryption
        const val FLYBITS_STORAGE_ENC_V1 = "flybits_con_storage"
        const val FLYBITS_STORAGE_ENC_V2 = "flybits_secure_storage_v2"
        const val FLYBITS_STORAGE_ENC_V3 = "flybits_secure_storage_v3"   // sets encryption based sharedpref file name.
        const val ARG_PROJECT_ID = "flybits_arg_project_id"
    }

    init {
        idp = getStringVariable(PREF_IDP_CONNECTED, "")
        languageCodes = getStringVariable(PREF_LANGUAGE_CODES, "")
        gateway = getStringVariable(PREF_GATEWAY_URL, "")
        locationPermission = getStringVariable(PREF_LOCATION_PERMISSION, "")
        jwtToken = getStringVariable(PREF_JWT_TOKEN, "")
    }

    /**
     * Migrate data from previous versions of [SharedElements] to the latest version.
     *
     * @param context Context associated with the Application.
     *
     * @return how many values were migrated successfully.
     */
    @JvmName("migrateData")
    internal fun migrateData(context: Context): Int {
        return performMigration(context)
    }

    /**
     * Migrate data from previous versions of [SharedElements] to the latest version.
     *
     * @param context Context associated with the Application.
     *
     * @return how many values were migrated successfully.
     */
    protected abstract fun performMigration(context: Context): Int

    /**
     * Get the IDP that the application used to connect to Flybits with. If the application is not
     * connected to Flybits then "" empty string will be returned.
     *
     * @return The saved string representation of the IDP used to connect to Flybits with, or "" if
     * the user is not connected to Flybits.
     */
    fun getConnectedIDP(): String {
        if (idp == null) {
            idp = getStringVariable(PREF_IDP_CONNECTED, "")
        }
        return idp ?: ""
    }

    /**
     * Get the previously saved [com.flybits.commons.library.models.User.deviceID].
     *
     * @return The saved [com.flybits.commons.library.models.User.deviceID] or "" if no
     * [com.flybits.commons.library.models.User.deviceID] is saved.
     */
    fun getDeviceID(): String {
        val user = userDAO.activeUser
        return if (user != null) {
            user.deviceID
        } else ""
    }

    /**
     * Get the ArrayList representation of the language codes set for this application.
     *
     * @return The ArrayList representation of the language codes set for this application.
     */
    fun getEnabledLanguagesAsArray(): ArrayList<String> {
        val languageCodes = getEnabledLanguagesAsString()
        return Utilities.convertLocalizationStringToList(languageCodes)
    }

    /**
     * Get the String representation of the language codes set for this application.
     *
     * @return The String representation of the language codes set for this application or "" if no
     * language is set.
     */
    fun getEnabledLanguagesAsString(): String {
        if (languageCodes == null) {
            languageCodes = getStringVariable(PREF_LANGUAGE_CODES, "")
        }
        return languageCodes ?: ""
    }

    /**
     * Get the gateway URL to be used for communicating with the Flybits servers.
     *
     * @return The root URL that all requests should point to.
     */
    fun getGatewayURL(): String {
        if (gateway == "") {
            gateway = getStringVariable(PREF_GATEWAY_URL, "")
        }
        return gateway
    }

    /**
     * Get the Notification Channel ID
     *
     * @return Push Notification channel ID
     */
    fun getNotificationChannel(): String {
        return if (notificationChannelID?.isNotBlank() == true) {
            notificationChannelID ?: ""
        } else getStringVariable(PREF_NOTIFICATION_CHANNEL_ID, "")
    }

    /**
     * Get the Notification authentication status
     *@return Is the Notification been authorized
     */

    fun getIsPushNotificationEnabled() = getBooleanVariable(PREF_IS_NOTIFICATION_ENABLED, isPushNotificationEnabled)

    /**
     * Get the device current location permission as string
     *
     * @return location permission, one of always/inUse/never
     */
    fun getLocationPermission(): String {
        if (locationPermission == null) {
            locationPermission = getStringVariable(PREF_LOCATION_PERMISSION, "")
        }
        return locationPermission ?: ""
    }

    /**
     * Get the previously saved `JWT` which is obtained when calling
     * [FlybitsManager.connect] the first time.
     *
     * @return The saved `JWT` which is obtained once the application logs into Flybits  or ""
     * if the application has not successfully received a `JWT` token.
     */
    fun getSavedJWTToken(): String {
        if (jwtToken == null) {
            jwtToken = getStringVariable(PREF_JWT_TOKEN, "")
        }
        return jwtToken ?: ""
    }

    private fun getPushPermissionForFirstTime() = getBooleanVariable(PREF_SET_PUSH_PERMISSION_FIRST_TIME, true)

    private fun clearMemory() {
        jwtToken = null
        idp = null
        languageCodes = null
        gateway = ""
        userId = null
        notificationChannelID = null
        isPushNotificationEnabled = false
        locationPermission = null
        user = null
    }

    /**
     * Sets the gateway URL to be used for communicating with the Flybits servers.
     *
     * @param url Gateway URL to be set.
     */
    fun setGatewayURL(url: String) {
        gateway = url
        setStringVariable(PREF_GATEWAY_URL, url)
    }

    /**
     * Sets the IDP that was used to connect to Flybits with such as Flybits, Anonymous, OAuth, etc.
     *
     * @param idp The string representation of the IDP used to connect to Flybits with.
     */
    fun setConnectedIDP(idp: String) {
        this.idp = idp
        setStringVariable(PREF_IDP_CONNECTED, idp)
    }

    /**
     * Sets current device location permission status, it's one of always/inUse/never
     *
     * @param locationPermission Device location permission status.
     */
    fun setLocationPermission(locationPermission: String): Boolean {
        return if (this.locationPermission == locationPermission) {
            false
        } else {
            this.locationPermission = locationPermission
            setStringVariable(PREF_LOCATION_PERMISSION, locationPermission)
            true
        }
    }

    /**
     * Sets the Notification Channel ID that to use to display the Notification for API >=26
     *
     * @param channelID Notification Channel ID
     * @return if the value need to be updated
     */
    fun setNotificationChannel(channelID: String) {
        setStringVariable(PREF_NOTIFICATION_CHANNEL_ID, channelID)
    }

    /**
     * Sets the push permission depending on previously saved permission
     * @return true if specified push permission is different from previously saved permission
     *
     * * if OS < 26, listen to App notification value only
     * if OS>=26 and push channel ID== null, listen to App notification value only
     * if OS>=26 and push channel ID is valued, listen to App Notification value && Channel value
     * @param isPushEnabled if the push notification get enabled
     */
    fun setIsPushEnabled(isPushEnabled: Boolean): Boolean {
        if (getIsPushNotificationEnabled() == isPushEnabled) {
            // return true if its running for very first time in order to send the analytics.
            if (getPushPermissionForFirstTime()) {
                setPushPermissionForFirstTime(false)
                return true
            }
            return false
        }
        setPushPermissionForFirstTime(false)
        setBooleanVariable(PREF_IS_NOTIFICATION_ENABLED, isPushEnabled)
        isPushNotificationEnabled = isPushEnabled
        return true
    }

    internal fun setPushPermissionForFirstTime(isForFirstTime: Boolean) {
        setBooleanVariable(PREF_SET_PUSH_PERMISSION_FIRST_TIME, isForFirstTime)
    }

    /**
     * Set custom Unique Device Id generator here otherwise a default one will be used
     * @param uniqueDeviceId String representing unique device ID
     */
    fun setUniqueDevice(uniqueDeviceId: String): Boolean {
        return if (uniqueDeviceId.equals(getUniqueDeviceId())) {
            false
        } else {
            setStringVariable(PREF_UNIQUE_ID, uniqueDeviceId)
            Logger.appendTag(TAG)
                    .d("Firebase Id Added.")
            true
        }
    }

    /**
     * @throws FlybitsException if no Unique Device ID stored and could not generate unique device id using uniqueDeviceIDGenerator
     *
     * @return unique device Id set by the user
     */
    fun getUniqueDeviceId(): String {
        return getStringVariable(PREF_UNIQUE_ID, "")
    }

    /**
     *
     * @return push provider type set in FlybitsManager's Builder.
     */
    fun getPushProviderType(): String {
        return getStringVariable(PREF_PUSH_PROVIDER_TYPE, "")
    }

    /**
     * Sets the Push Type for Flybits server.
     *
     * @param pushProviderType The push provider type to be stored to be sent on Flybits Server.
     */
    fun storePushProviderType(pushProviderType: PushProviderType?) {
        pushProviderType?.let {
            this.mPushProviderType = pushProviderType
            setStringVariable(PREF_PUSH_PROVIDER_TYPE, it.name)
        } ?: run {
            setStringVariable(PREF_PUSH_PROVIDER_TYPE, "")
        }
    }

    /**
     * Sets the unique JWT Token obtained from the Flybits Core for the user/device combination.
     *
     * @param jwtToken The unique JWT Token that is used by Flybits to identify a user/device
     * combination.
     */
    fun setJWTToken(jwtToken: String) {
        this.jwtToken = jwtToken
        setStringVariable(PREF_JWT_TOKEN, jwtToken)
    }

    /**
     * Sets the localization values of the device.
     *
     * @param listOfLanguages The array of languages that should be used for this device.
     */
    fun setLocalization(listOfLanguages: ArrayList<String>) {
        val languages = Utilities.convertLocalizationCodeToString(listOfLanguages)
        this.languageCodes = languages
        setStringVariable(PREF_LANGUAGE_CODES, languages)
    }

    /**
     * Sets the User Id which can be used by other components/sdks within the
     * Flybits ecosystem.
     *
     * @param userId The unique Identifier that represents current user
     */
    fun setUserId(userId: String) {
        this.userId = userId
        setStringVariable(PREF_USER_ID, userId)
    }

    /**
     * Sets the User once Flybits connected.
     *
     * @param user The unique object that represents current user
     */
    fun setUser(user: User?) {
        user?.let {
            this.user = user
            val gson = Gson()
            try {
                val json: String = gson.toJson(user)
                setStringVariable(PREF_USER, json)
            } catch (e: Exception) {
                setStringVariable(PREF_USER, "")
            }
        } ?: run {
            setStringVariable(PREF_USER, "")
        }
    }

    /**
     * @return unique active {@link User} saved on connect.
     */
    fun getUser(): User? {
        val gson = Gson()
        val json: String = getStringVariable(PREF_USER, "")
        if (json.isNotEmpty()) {
            return try {
                gson.fromJson(json, User::class.java)
            } catch (e: Exception) {
                null
            }
        }
        return null
    }

    /**
     * Sets the value that indicate whether the Push token has been sent successfully
     * @param isSent Whether the Push token has been sent successfully
     */
    fun setPushTokenStatus(isSent: String) {
        setStringVariable(PREF_IS_TOKEN_SENT, isSent)
    }

    /**
     * @return Whether the Push token has been sent
     */
    fun getPushTokenStatus(): String {
        return getStringVariable(PREF_IS_TOKEN_SENT, KEY_IS_TOKEN_SENT_UNKNOWN)
    }

    abstract fun setStringVariable(key: String, value: String)
    abstract fun setBooleanVariable(key: String, value: Boolean)

    private fun getStringVariable(key: String, default: String) = sharedPreferences.getString(key, default) ?: ""
    private fun getBooleanVariable(key: String, default: Boolean) =
            sharedPreferences.getBoolean(key, default)
}
