/*
 * Decompiled with CFR 0.152.
 */
package com.fizzed.jne;

import com.fizzed.jne.HardwareArchitecture;
import com.fizzed.jne.LinuxLibC;
import com.fizzed.jne.MemoizedInitializer;
import com.fizzed.jne.OperatingSystem;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PlatformInfo {
    private static final Logger log = LoggerFactory.getLogger(PlatformInfo.class);
    private static final MemoizedInitializer<OperatingSystem> operatingSystemRef = new MemoizedInitializer();
    private static final MemoizedInitializer<HardwareArchitecture> hardwareArchitectureRef = new MemoizedInitializer();
    private static final MemoizedInitializer<LinuxLibC> linuxLibCRef = new MemoizedInitializer();
    private static final MemoizedInitializer<LinuxDetectedFilesResult> libFilesResultRef = new MemoizedInitializer();
    private static final MemoizedInitializer<LinuxDetectedFilesResult> mappedFilesResultRef = new MemoizedInitializer();

    public static OperatingSystem detectOperatingSystem() {
        return operatingSystemRef.once(new MemoizedInitializer.Initializer<OperatingSystem>(){

            @Override
            public OperatingSystem init() {
                return PlatformInfo.doDetectOperatingSystem();
            }
        });
    }

    static OperatingSystem doDetectOperatingSystem() {
        String osName = System.getProperty("os.name");
        long now = System.currentTimeMillis();
        log.trace("Trying to detect operating system via system property [{}]", (Object)osName);
        OperatingSystem operatingSystem = PlatformInfo.detectOperatingSystemFromValues(osName);
        if (operatingSystem != OperatingSystem.UNKNOWN) {
            log.debug("Detected operating system {} (in {} ms)", (Object)operatingSystem, (Object)(System.currentTimeMillis() - now));
        } else {
            log.warn("Unable to detect operating system (in {} ms)", (Object)(System.currentTimeMillis() - now));
        }
        return operatingSystem;
    }

    static OperatingSystem detectOperatingSystemFromValues(String osName) {
        if (osName != null) {
            if ((osName = osName.toLowerCase()).contains("windows")) {
                return OperatingSystem.WINDOWS;
            }
            if (osName.contains("mac") || osName.contains("darwin")) {
                return OperatingSystem.MACOS;
            }
            if (osName.contains("linux")) {
                return OperatingSystem.LINUX;
            }
            if (osName.contains("sun") || osName.contains("solaris")) {
                return OperatingSystem.SOLARIS;
            }
            if (osName.contains("freebsd")) {
                return OperatingSystem.FREEBSD;
            }
            if (osName.contains("openbsd")) {
                return OperatingSystem.OPENBSD;
            }
        }
        return OperatingSystem.UNKNOWN;
    }

    public static HardwareArchitecture detectHardwareArchitecture() {
        return hardwareArchitectureRef.once(new MemoizedInitializer.Initializer<HardwareArchitecture>(){

            @Override
            public HardwareArchitecture init() {
                return PlatformInfo.doDetectHardwareArchitecture();
            }
        });
    }

    static HardwareArchitecture doDetectHardwareArchitecture() {
        String osArch = System.getProperty("os.arch");
        String abiType = System.getProperty("sun.arch.abi");
        String bootLibPath = System.getProperty("sun.boot.library.path", "").toLowerCase();
        LinuxDetectedFilesResult linuxMappedFilesResult = PlatformInfo.detectLinuxMappedFiles();
        long now = System.currentTimeMillis();
        HardwareArchitecture hardwareArchitecture = PlatformInfo.detectHardwareArchitectureFromValues(osArch, abiType, bootLibPath, linuxMappedFilesResult);
        if (hardwareArchitecture != HardwareArchitecture.UNKNOWN) {
            log.debug("Detected hardware architecture {} (in {} ms)", (Object)hardwareArchitecture, (Object)(System.currentTimeMillis() - now));
        } else {
            log.warn("Unable to detect hardware architecture (in {} ms)", (Object)(System.currentTimeMillis() - now));
        }
        return hardwareArchitecture;
    }

    public static HardwareArchitecture detectHardwareArchitectureFromValues(String osArch, String abiType, String bootLibPath, LinuxDetectedFilesResult linuxMappedFilesResult) {
        log.trace("Trying to detect hardware architecture via sysprops arch={}, abi={}, bootpath={}", new Object[]{osArch, abiType, bootLibPath});
        if (osArch != null) {
            osArch = osArch != null ? osArch.toLowerCase() : "none";
            abiType = abiType != null ? abiType.toLowerCase() : "none";
            String string = bootLibPath = bootLibPath != null ? bootLibPath.toLowerCase() : "none";
            if (osArch.contains("amd64") || osArch.contains("x86_64")) {
                return HardwareArchitecture.X64;
            }
            if (osArch.contains("i386") || osArch.contains("i686") || osArch.contains("x86")) {
                return HardwareArchitecture.X32;
            }
            if (osArch.contains("aarch64")) {
                return HardwareArchitecture.ARM64;
            }
            if (osArch.contains("armv7l")) {
                return HardwareArchitecture.ARMHF;
            }
            if (osArch.contains("arm") || osArch.contains("aarch32")) {
                log.trace("System property arch [{}] is ambiguous, will try a few workarounds", (Object)osArch);
                if ("gnueabihf".equals(abiType)) {
                    return HardwareArchitecture.ARMHF;
                }
                if (bootLibPath.contains("armhf") || bootLibPath.contains("aarch32hf")) {
                    return HardwareArchitecture.ARMHF;
                }
                if ("gnueabi".equals(abiType)) {
                    return HardwareArchitecture.ARMEL;
                }
                if (bootLibPath.contains("armsf") || bootLibPath.contains("aarch32sf")) {
                    return HardwareArchitecture.ARMEL;
                }
                if (linuxMappedFilesResult != null && linuxMappedFilesResult.getArch() != null && linuxMappedFilesResult.getArch() != HardwareArchitecture.UNKNOWN) {
                    return linuxMappedFilesResult.getArch();
                }
            } else {
                if (osArch.contains("riscv64")) {
                    return HardwareArchitecture.RISCV64;
                }
                if (osArch.contains("s390x")) {
                    return HardwareArchitecture.S390X;
                }
                if (osArch.contains("ppc64le")) {
                    return HardwareArchitecture.PPC64LE;
                }
                if (osArch.contains("mips64el") || osArch.contains("mips64le")) {
                    return HardwareArchitecture.MIPS64LE;
                }
            }
        }
        return HardwareArchitecture.UNKNOWN;
    }

    public static LinuxLibC detectLinuxLibC() {
        return linuxLibCRef.once(new MemoizedInitializer.Initializer<LinuxLibC>(){

            @Override
            public LinuxLibC init() {
                LinuxDetectedFilesResult detectedFilesResult = PlatformInfo.detectLinuxMappedFiles();
                if (detectedFilesResult.getLibc() != null && detectedFilesResult.getLibc() != LinuxLibC.UNKNOWN) {
                    return detectedFilesResult.getLibc();
                }
                detectedFilesResult = PlatformInfo.detectLinuxLibFiles();
                if (detectedFilesResult.getLibc() != null && detectedFilesResult.getLibc() != LinuxLibC.UNKNOWN) {
                    return detectedFilesResult.getLibc();
                }
                log.debug("Will assume we are running on GLIBC");
                return LinuxLibC.GLIBC;
            }
        });
    }

    public static LinuxDetectedFilesResult detectLinuxLibFiles() {
        return libFilesResultRef.once(new MemoizedInitializer.Initializer<LinuxDetectedFilesResult>(){

            @Override
            public LinuxDetectedFilesResult init() {
                return PlatformInfo.doDetectLinuxLibFiles();
            }
        });
    }

    static LinuxDetectedFilesResult doDetectLinuxLibFiles() {
        long now = System.currentTimeMillis();
        Path libDir = Paths.get("/lib/", new String[0]);
        log.trace("Trying to detect libc/arch via [{}]", (Object)libDir);
        LinuxDetectedFilesResult result = new LinuxDetectedFilesResult();
        try {
            File[] mapFiles;
            if (Files.exists(libDir, new LinkOption[0]) && (mapFiles = libDir.toFile().listFiles()) != null) {
                for (File mapFile : mapFiles) {
                    try {
                        log.trace("Analyzing file {}", (Object)mapFile);
                        String name = mapFile.getName().toLowerCase();
                        if (name.contains("musl") && result.getLibc() == null) {
                            log.debug("Detected libc MUSL via /lib dir strategy (in {} ms)", (Object)(System.currentTimeMillis() - now));
                            result.setLibc(LinuxLibC.MUSL);
                        }
                        if (name.contains("armhf") || name.contains("arm-linux-gnueabihf")) {
                            if (result.getArch() == HardwareArchitecture.ARMHF) continue;
                            log.debug("Detected hardware architecture ARMHF via /lib dir strategy (in {} ms)", (Object)(System.currentTimeMillis() - now));
                            result.setArch(HardwareArchitecture.ARMHF);
                            continue;
                        }
                        if (!name.contains("armel") && !name.contains("arm-linux-gnueabi") || result.getArch() == HardwareArchitecture.ARMEL) continue;
                        log.debug("Detected hardware architecture ARMEL via /lib dir strategy (in {} ms)", (Object)(System.currentTimeMillis() - now));
                        result.setArch(HardwareArchitecture.ARMEL);
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                }
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return result;
    }

    public static LinuxDetectedFilesResult detectLinuxMappedFiles() {
        return mappedFilesResultRef.once(new MemoizedInitializer.Initializer<LinuxDetectedFilesResult>(){

            @Override
            public LinuxDetectedFilesResult init() {
                return PlatformInfo.doDetectLinuxMappedFiles();
            }
        });
    }

    static LinuxDetectedFilesResult doDetectLinuxMappedFiles() {
        long now = System.currentTimeMillis();
        Path mapFilesDir = Paths.get("/proc/self/map_files", new String[0]);
        log.trace("Trying to detect libc/arch via [{}]", (Object)mapFilesDir);
        LinuxDetectedFilesResult result = new LinuxDetectedFilesResult();
        boolean possiblyFoundGlibc = false;
        try {
            File[] mapFiles;
            if (Files.exists(mapFilesDir, new LinkOption[0]) && (mapFiles = mapFilesDir.toFile().listFiles()) != null) {
                for (File mapFile : mapFiles) {
                    try {
                        String _realMapFilePath = PlatformInfo.realpath(mapFile.toPath());
                        log.trace("Analyzing file {}", (Object)_realMapFilePath);
                        String realMapFilePath = _realMapFilePath.toLowerCase();
                        if (realMapFilePath.contains("musl")) {
                            if (result.getLibc() == null) {
                                log.debug("Detected libc MUSL via mapped files strategy (in {} ms)", (Object)(System.currentTimeMillis() - now));
                                result.setLibc(LinuxLibC.MUSL);
                            }
                        } else if (realMapFilePath.contains("/libc")) {
                            possiblyFoundGlibc = true;
                        }
                        if (realMapFilePath.contains("armhf") || realMapFilePath.contains("arm-linux-gnueabihf")) {
                            if (result.getArch() == HardwareArchitecture.ARMHF) continue;
                            log.debug("Detected hardware architecture ARMHF via mapped files strategy (in {} ms)", (Object)(System.currentTimeMillis() - now));
                            result.setArch(HardwareArchitecture.ARMHF);
                            continue;
                        }
                        if (!realMapFilePath.contains("armel") && !realMapFilePath.contains("arm-linux-gnueabi") || result.getArch() == HardwareArchitecture.ARMEL) continue;
                        log.debug("Detected hardware architecture ARMEL via mapped files strategy (in {} ms)", (Object)(System.currentTimeMillis() - now));
                        result.setArch(HardwareArchitecture.ARMEL);
                    }
                    catch (IOException iOException) {
                        // empty catch block
                    }
                }
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        if (possiblyFoundGlibc) {
            log.debug("Detected libc GLIBC via mapped files strategy (in {} ms)", (Object)(System.currentTimeMillis() - now));
            result.setLibc(LinuxLibC.GLIBC);
        }
        if (result.getLibc() == null) {
            log.warn("Unable to detect libc via mapped files strategy (in {} ms)", (Object)(System.currentTimeMillis() - now));
            result.setLibc(LinuxLibC.UNKNOWN);
        }
        return result;
    }

    private static String realpath(Path file) throws IOException {
        try {
            return file.toRealPath(new LinkOption[0]).toString();
        }
        catch (IOException ioe1) {
            try {
                return Files.readSymbolicLink(file).toString();
            }
            catch (IOException iOException) {
                throw ioe1;
            }
        }
    }

    public static class LinuxDetectedFilesResult {
        private LinuxLibC libc;
        private HardwareArchitecture arch;

        public LinuxLibC getLibc() {
            return this.libc;
        }

        public void setLibc(LinuxLibC libc) {
            this.libc = libc;
        }

        public HardwareArchitecture getArch() {
            return this.arch;
        }

        public void setArch(HardwareArchitecture arch) {
            this.arch = arch;
        }
    }
}

