/*
 * Decompiled with CFR 0.152.
 */
package com.fizzed.jne;

import com.fizzed.jne.ExtractException;
import com.fizzed.jne.HardwareArchitecture;
import com.fizzed.jne.JarUtil;
import com.fizzed.jne.OperatingSystem;
import com.fizzed.jne.Options;
import com.fizzed.jne.ResourceNotFoundException;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JNE {
    private static final Logger log = LoggerFactory.getLogger(JNE.class);
    public static Options DEFAULT_OPTIONS = new Options();
    private static File TEMP_DIRECTORY;
    private static final ConcurrentHashMap<File, String> JAR_VERSION_HASHES;

    public static synchronized File findExecutable(String name) throws IOException {
        return JNE.findExecutable(name, null, null);
    }

    public static synchronized File findExecutable(String name, String targetName) throws IOException {
        return JNE.findExecutable(name, targetName, null);
    }

    public static synchronized File findExecutable(String name, Options options) throws IOException {
        return JNE.findExecutable(name, null, options);
    }

    public static synchronized File findExecutable(String name, String targetName, Options options) throws IOException {
        File file;
        if (options == null) {
            options = DEFAULT_OPTIONS;
        }
        String fileName = options.createExecutableName(name, options.getOperatingSystem());
        String targetFileName = null;
        if (targetName != null) {
            targetFileName = options.createExecutableName(targetName, options.getOperatingSystem());
        }
        if ((file = JNE.find(fileName, targetFileName, options, options.getOperatingSystem(), options.getHardwareArchitecture())) == null && options.isX32ExecutableFallback() && options.getHardwareArchitecture() == HardwareArchitecture.X64) {
            file = JNE.find(fileName, targetFileName, options, options.getOperatingSystem(), HardwareArchitecture.X32);
        }
        return file;
    }

    public static synchronized File requireExecutable(String name) throws IOException {
        return JNE.requireExecutable(name, null, null);
    }

    public static synchronized File requireExecutable(String name, Options options) throws IOException {
        return JNE.requireExecutable(name, null, options);
    }

    public static synchronized File requireExecutable(String name, String targetName, Options options) throws IOException {
        File file = JNE.findExecutable(name, targetName, options);
        if (file == null) {
            throw new ResourceNotFoundException("Resource executable " + name + " not found");
        }
        return file;
    }

    public static synchronized File findLibrary(String name) {
        return JNE.findLibrary(name, null, null);
    }

    public static synchronized File findLibrary(String name, Integer majorVersion) {
        return JNE.findLibrary(name, null, majorVersion);
    }

    public static synchronized File findLibrary(String name, Options options, Integer majorVersion) {
        if (options == null) {
            options = DEFAULT_OPTIONS;
        }
        String fileName = options.createLibraryName(name, options.getOperatingSystem(), majorVersion, null, null);
        try {
            return JNE.find(fileName, null, options, options.getOperatingSystem(), options.getHardwareArchitecture());
        }
        catch (IOException e) {
            throw new UnsatisfiedLinkError(e.getMessage());
        }
    }

    public static synchronized void loadLibrary(String name) {
        JNE.loadLibrary(name, null, null);
    }

    public static synchronized void loadLibrary(String name, Integer majorVersion) {
        JNE.loadLibrary(name, null, majorVersion);
    }

    public static synchronized void loadLibrary(String name, Options options, Integer majorVersion) {
        File f = null;
        try {
            f = JNE.findLibrary(name, options, majorVersion);
        }
        catch (Exception e) {
            log.debug("Exception while finding library: " + e.getMessage());
            throw new UnsatisfiedLinkError("Unable to cleanly find (or extract) library [" + name + "] as resource");
        }
        if (f != null) {
            log.trace("System.load(" + f.getAbsolutePath() + ")");
            System.load(f.getAbsolutePath());
        } else {
            log.trace("Falling back to System.loadLibrary(" + name + ")");
            System.loadLibrary(name);
        }
        log.debug("Library [" + name + "] loaded!");
    }

    public static synchronized File findFile(String name) throws IOException {
        return JNE.findFile(name, null);
    }

    public static synchronized File findFile(String name, Options options) throws IOException {
        File file;
        if (options == null) {
            options = DEFAULT_OPTIONS;
        }
        if ((file = JNE.find(name, name, options, options.getOperatingSystem(), options.getHardwareArchitecture())) == null) {
            file = JNE.find(name, name, options, options.getOperatingSystem(), HardwareArchitecture.ANY);
        }
        if (file == null) {
            file = JNE.find(name, name, options, OperatingSystem.ANY, HardwareArchitecture.ANY);
        }
        return file;
    }

    public static synchronized File requireFile(String name) throws IOException {
        return JNE.requireFile(name, null);
    }

    public static synchronized File requireFile(String name, Options options) throws IOException {
        File file = JNE.findFile(name, options);
        if (file == null) {
            throw new ResourceNotFoundException("Resource file " + name + " not found");
        }
        return file;
    }

    public static synchronized File find(String fileName, String targetFileName, Options options, OperatingSystem os, HardwareArchitecture arch) throws IOException {
        if (options == null) {
            options = DEFAULT_OPTIONS;
        }
        if (os == null || os == OperatingSystem.UNKNOWN) {
            throw new ExtractException("Unable to detect operating system (e.g. Windows)");
        }
        if (arch == null || arch == HardwareArchitecture.UNKNOWN) {
            throw new ExtractException("Unable to detect hardware architecture (e.g. x86)");
        }
        if (targetFileName == null) {
            targetFileName = fileName;
        }
        log.trace("Finding fileName [" + fileName + "] targetFileName [" + targetFileName + "] os [" + (Object)((Object)os) + "] arch [" + (Object)((Object)arch) + "]...");
        List<String> resourcePaths = options.createResourcePaths(os, arch, options.getLinuxLibC(), fileName);
        URL url = null;
        for (String resourcePath : resourcePaths) {
            log.trace("Finding resource [" + resourcePath + "]");
            url = JNE.class.getResource(resourcePath);
            if (url == null) continue;
            break;
        }
        if (url == null) {
            log.debug("Unable to locate any resource of {}", resourcePaths);
            return null;
        }
        log.trace("Resource found @ " + url);
        if (url.getProtocol().equals("jar")) {
            log.trace("Resource in jar; extracting file if necessary...");
            String versionHash = JNE.getJarVersionHashForResource(url);
            log.trace("Version hash [" + versionHash + "]");
            File d = options.getExtractDir();
            if (d == null) {
                d = JNE.getOrCreateTempDirectory(options.isCleanupExtracted());
            } else {
                if (!d.exists()) {
                    d.mkdirs();
                }
                if (!d.isDirectory()) {
                    throw new ExtractException("Extract dir [" + d + "] is not a directory");
                }
            }
            log.trace("Using dir [" + d + "]");
            File exeFile = new File(d, targetFileName);
            File exeHashFile = new File(exeFile.getAbsolutePath() + ".hash");
            if (exeFile.exists()) {
                log.trace("File already exists; verifying if hash matches");
                if (!exeHashFile.exists()) {
                    exeFile.delete();
                } else {
                    String existingHash = JNE.readFileToString(exeHashFile);
                    if (existingHash == null || !existingHash.equals(versionHash)) {
                        log.trace("Hash mismatch; deleting files; will freshly extract file");
                        exeFile.delete();
                        exeHashFile.delete();
                    } else {
                        log.trace("Hash matches; will use existing file");
                        return exeFile;
                    }
                }
            }
            if (!exeFile.exists()) {
                try {
                    log.trace("Extracting [" + url + "] to [" + exeFile + "]...");
                    JNE.extractTo(url, exeFile);
                    log.trace("Setting to executable");
                    exeFile.setExecutable(true);
                    log.trace("Writing hash file");
                    JNE.writeStringToFile(exeHashFile, versionHash);
                    if (options.isCleanupExtracted()) {
                        log.trace("Scheduling file and hash for delete on exit");
                        exeFile.deleteOnExit();
                        exeHashFile.deleteOnExit();
                    }
                }
                catch (IOException e) {
                    log.debug("Failed to extract file: {}", (Object)e.getMessage());
                    throw new ExtractException("Unable to cleanly extract executable from jar", e);
                }
            }
            log.trace("Returning [" + exeFile + "]");
            return exeFile;
        }
        if (url.getProtocol().equals("file")) {
            log.trace("Resource in file");
            try {
                File exeFile = new File(url.toURI());
                if (!exeFile.canExecute()) {
                    log.trace("Setting file to executable");
                    if (!exeFile.setExecutable(true)) {
                        log.debug("Unable to cleanly set file to executable");
                        throw new ExtractException("Executable was found but it cannot be set to execute [" + exeFile.getAbsolutePath() + "]");
                    }
                }
                log.trace("Returning [" + exeFile + "]");
                return exeFile;
            }
            catch (URISyntaxException e) {
                log.debug("URL syntax error");
                throw new ExtractException("Unable to create executable file from uri", e);
            }
        }
        throw new ExtractException("Unsupported executable resource protocol [" + url.getProtocol() + "]");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void extractTo(URL url, File file) throws IOException {
        try (InputStream in = url.openStream();
             BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(file, false));){
            int len;
            byte[] buffer = new byte[8192];
            while ((len = in.read(buffer)) > -1) {
                ((OutputStream)out).write(buffer, 0, len);
            }
            ((OutputStream)out).flush();
        }
    }

    private static String readFileToString(File file) throws IOException {
        StringBuilder result = new StringBuilder();
        try (BufferedInputStream is = new BufferedInputStream(new FileInputStream(file));){
            int len;
            byte[] buf = new byte[1024];
            while ((len = is.read(buf)) > -1) {
                result.append(new String(buf, 0, len, "UTF-8"));
            }
        }
        return result.toString();
    }

    private static void writeStringToFile(File file, String s) throws IOException {
        try (FileOutputStream os = new FileOutputStream(file, false);){
            os.write(s.getBytes("UTF-8"));
            os.flush();
        }
    }

    private static String getJarVersionHashForResource(URL resource) throws IOException {
        File jarFile = JarUtil.getJarFileForResource(resource);
        if (JAR_VERSION_HASHES.containsKey(jarFile)) {
            return JAR_VERSION_HASHES.get(jarFile);
        }
        String manifestVersion = JarUtil.getManifestVersionNumber(jarFile);
        StringBuilder hashBuilder = new StringBuilder();
        hashBuilder.append("file:");
        hashBuilder.append(jarFile.getAbsolutePath());
        hashBuilder.append("|last_modified:");
        hashBuilder.append(jarFile.lastModified());
        hashBuilder.append("|version:");
        hashBuilder.append(manifestVersion);
        String hash = hashBuilder.toString();
        JAR_VERSION_HASHES.put(jarFile, hash);
        return hash;
    }

    private static File getOrCreateTempDirectory(boolean deleteOnExit) throws ExtractException {
        if (TEMP_DIRECTORY != null && TEMP_DIRECTORY.exists()) {
            return TEMP_DIRECTORY;
        }
        try {
            Path baseDir = Paths.get(System.getProperty("java.io.tmpdir"), new String[0]);
            Path tempDirectory = baseDir.resolve("jne." + UUID.randomUUID().toString());
            Files.createDirectories(tempDirectory, new FileAttribute[0]);
            File tempDirectoryAsFile = tempDirectory.toFile();
            if (deleteOnExit) {
                tempDirectoryAsFile.deleteOnExit();
            }
            TEMP_DIRECTORY = tempDirectoryAsFile;
            return TEMP_DIRECTORY;
        }
        catch (IOException e) {
            throw new ExtractException("Unable to create temporary dir", e);
        }
    }

    static {
        JAR_VERSION_HASHES = new ConcurrentHashMap();
    }
}

