/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.verifier;

import com.facebook.presto.spi.ErrorCode;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.verifier.QueryPair;
import com.facebook.presto.verifier.QueryResult;
import com.facebook.presto.verifier.Validator;
import com.facebook.presto.verifier.VerifierConfig;
import com.facebook.presto.verifier.VerifierQueryEvent;
import com.google.common.base.Joiner;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableSet;
import io.airlift.event.client.EventClient;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import java.io.Closeable;
import java.io.IOException;
import java.io.PrintStream;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletionService;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

public class Verifier {
    private static final Logger log = Logger.get(Verifier.class);
    private static final Set<ErrorCode> EXPECTED_ERRORS = ImmutableSet.builder().add((Object)StandardErrorCode.REMOTE_TASK_MISMATCH.toErrorCode()).add((Object)StandardErrorCode.TOO_MANY_REQUESTS_FAILED.toErrorCode()).add((Object)StandardErrorCode.PAGE_TRANSPORT_TIMEOUT.toErrorCode()).build();
    private final VerifierConfig config;
    private final Set<EventClient> eventClients;
    private final int threadCount;
    private final Set<String> whitelist;
    private final Set<String> blacklist;
    private final int precision;

    public Verifier(PrintStream out, VerifierConfig config, Set<EventClient> eventClients) {
        Objects.requireNonNull(out, "out is null");
        this.config = Objects.requireNonNull(config, "config is null");
        this.eventClients = Objects.requireNonNull(eventClients, "eventClients is null");
        this.whitelist = Objects.requireNonNull(config.getWhitelist(), "whitelist is null");
        this.blacklist = Objects.requireNonNull(config.getBlacklist(), "blacklist is null");
        this.threadCount = config.getThreadCount();
        this.precision = config.getDoublePrecision();
    }

    public int run(List<QueryPair> queries) throws InterruptedException {
        ExecutorService executor = Executors.newFixedThreadPool(this.threadCount);
        ExecutorCompletionService<Validator> completionService = new ExecutorCompletionService<Validator>(executor);
        int totalQueries = queries.size() * this.config.getSuiteRepetitions();
        log.info("Total Queries:     %d", new Object[]{totalQueries});
        log.info("Whitelisted Queries: %s", new Object[]{Joiner.on((char)',').join(this.whitelist)});
        int queriesSubmitted = 0;
        for (int i = 0; i < this.config.getSuiteRepetitions(); ++i) {
            for (QueryPair query : queries) {
                for (int j = 0; j < this.config.getQueryRepetitions(); ++j) {
                    if (!this.whitelist.isEmpty() && !this.whitelist.contains(query.getName())) {
                        log.debug("Query %s is not whitelisted", new Object[]{query.getName()});
                        continue;
                    }
                    if (this.blacklist.contains(query.getName())) {
                        log.debug("Query %s is blacklisted", new Object[]{query.getName()});
                        continue;
                    }
                    Validator validator = new Validator(this.config.getControlGateway(), this.config.getTestGateway(), this.config.getControlTimeout(), this.config.getTestTimeout(), this.config.getMaxRowCount(), this.config.isExplainOnly(), this.config.getDoublePrecision(), this.isCheckCorrectness(query), true, this.config.isVerboseResultsComparison(), this.config.getControlTeardownRetries(), this.config.getTestTeardownRetries(), query);
                    completionService.submit(validator::valid, validator);
                    ++queriesSubmitted;
                }
            }
        }
        log.info("Allowed Queries:     %d", new Object[]{queriesSubmitted});
        log.info("Skipped Queries:     %d", new Object[]{totalQueries - queriesSubmitted});
        log.info("---------------------");
        executor.shutdown();
        int total = 0;
        int valid = 0;
        int failed = 0;
        int skipped = 0;
        double lastProgress = 0.0;
        while (total < queriesSubmitted) {
            ++total;
            Validator validator = (Validator)Verifier.takeUnchecked(completionService);
            if (validator.isSkipped()) {
                if (!this.config.isQuiet()) {
                    log.warn("%s", new Object[]{validator.getSkippedMessage()});
                }
                ++skipped;
                continue;
            }
            if (validator.valid()) {
                ++valid;
            } else {
                ++failed;
            }
            for (EventClient eventClient : this.eventClients) {
                eventClient.post((Object[])new VerifierQueryEvent[]{this.buildEvent(validator)});
            }
            double progress = (double)total / (double)totalQueries * 100.0;
            if (this.config.isQuiet() && !(progress - lastProgress > 1.0)) continue;
            log.info("Progress: %s valid, %s failed, %s skipped, %.2f%% done", new Object[]{valid, failed, skipped, progress});
            lastProgress = progress;
        }
        log.info("Results: %s / %s (%s skipped)", new Object[]{valid, failed, skipped});
        log.info("");
        for (EventClient eventClient : this.eventClients) {
            if (!(eventClient instanceof Closeable)) continue;
            try {
                ((Closeable)eventClient).close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
            log.info("");
        }
        return failed;
    }

    private boolean isCheckCorrectness(QueryPair query) {
        if (Pattern.matches(this.config.getSkipCorrectnessRegex(), query.getTest().getQuery()) || Pattern.matches(this.config.getSkipCorrectnessRegex(), query.getControl().getQuery())) {
            return false;
        }
        return this.config.isCheckCorrectnessEnabled();
    }

    private VerifierQueryEvent buildEvent(Validator validator) {
        String errorMessage = null;
        QueryPair queryPair = validator.getQueryPair();
        QueryResult control = validator.getControlResult();
        QueryResult test = validator.getTestResult();
        if (!validator.valid()) {
            errorMessage = String.format("Test state %s, Control state %s\n", new Object[]{test.getState(), control.getState()});
            Exception e = test.getException();
            if (e != null && Verifier.shouldAddStackTrace(e)) {
                errorMessage = errorMessage + Throwables.getStackTraceAsString((Throwable)e);
            }
            if (control.getState() == QueryResult.State.SUCCESS && test.getState() == QueryResult.State.SUCCESS) {
                errorMessage = errorMessage + validator.getResultsComparison(this.precision).trim();
            }
        }
        return new VerifierQueryEvent(queryPair.getSuite(), this.config.getRunId(), this.config.getSource(), queryPair.getName(), !validator.valid(), queryPair.getTest().getCatalog(), queryPair.getTest().getSchema(), queryPair.getTest().getPreQueries(), queryPair.getTest().getQuery(), queryPair.getTest().getPostQueries(), test.getQueryId(), Verifier.optionalDurationToSeconds(test.getCpuTime()), Verifier.optionalDurationToSeconds(test.getWallTime()), queryPair.getControl().getCatalog(), queryPair.getControl().getSchema(), queryPair.getControl().getPreQueries(), queryPair.getControl().getQuery(), queryPair.getControl().getPostQueries(), control.getQueryId(), Verifier.optionalDurationToSeconds(control.getCpuTime()), Verifier.optionalDurationToSeconds(control.getWallTime()), errorMessage);
    }

    private static Double optionalDurationToSeconds(Duration duration) {
        return duration != null ? Double.valueOf(duration.convertTo(TimeUnit.SECONDS).getValue()) : null;
    }

    private static <T> T takeUnchecked(CompletionService<T> completionService) throws InterruptedException {
        try {
            return completionService.take().get();
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    private static boolean shouldAddStackTrace(Exception e) {
        ErrorCode errorCode;
        return !(e instanceof PrestoException) || !EXPECTED_ERRORS.contains(errorCode = ((PrestoException)e).getErrorCode());
    }
}

