/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.verifier;

import com.facebook.presto.verifier.QueryPair;
import com.facebook.presto.verifier.QueryResult;
import com.facebook.presto.verifier.Validator;
import com.facebook.presto.verifier.VerifierConfig;
import com.facebook.presto.verifier.VerifierQueryEvent;
import com.google.common.base.Joiner;
import com.google.common.base.Throwables;
import io.airlift.event.client.EventClient;
import io.airlift.log.Logger;
import java.io.PrintStream;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletionService;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

public class Verifier {
    private static final Logger log = Logger.get(Verifier.class);
    private final VerifierConfig config;
    private final Set<EventClient> eventClients;
    private final int threadCount;
    private final Set<String> whitelist;
    private final Set<String> blacklist;
    private final int precision;

    public Verifier(PrintStream out, VerifierConfig config, Set<EventClient> eventClients) {
        Objects.requireNonNull(out, "out is null");
        this.config = Objects.requireNonNull(config, "config is null");
        this.eventClients = Objects.requireNonNull(eventClients, "eventClients is null");
        this.whitelist = Objects.requireNonNull(config.getWhitelist(), "whitelist is null");
        this.blacklist = Objects.requireNonNull(config.getBlacklist(), "blacklist is null");
        this.threadCount = config.getThreadCount();
        this.precision = config.getDoublePrecision();
    }

    public int run(List<QueryPair> queries) throws InterruptedException {
        ExecutorService executor = Executors.newFixedThreadPool(this.threadCount);
        ExecutorCompletionService<Validator> completionService = new ExecutorCompletionService<Validator>(executor);
        int totalQueries = queries.size() * this.config.getSuiteRepetitions();
        log.info("Total Queries:     %d", new Object[]{totalQueries});
        log.info("Whitelisted Queries: %s", new Object[]{Joiner.on((char)',').join(this.whitelist)});
        int queriesSubmitted = 0;
        for (int i = 0; i < this.config.getSuiteRepetitions(); ++i) {
            for (QueryPair query : queries) {
                for (int j = 0; j < this.config.getQueryRepetitions(); ++j) {
                    if (!this.whitelist.isEmpty() && !this.whitelist.contains(query.getName())) {
                        log.debug("Query %s is not whitelisted", new Object[]{query.getName()});
                        continue;
                    }
                    if (this.blacklist.contains(query.getName())) {
                        log.debug("Query %s is blacklisted", new Object[]{query.getName()});
                        continue;
                    }
                    Validator validator = new Validator(this.config, query);
                    completionService.submit(Verifier.validateTask(validator), validator);
                    ++queriesSubmitted;
                }
            }
        }
        log.info("Allowed Queries:     %d", new Object[]{queriesSubmitted});
        log.info("Skipped Queries:     %d", new Object[]{totalQueries - queriesSubmitted});
        log.info("---------------------");
        executor.shutdown();
        int total = 0;
        int valid = 0;
        int failed = 0;
        int skipped = 0;
        double lastProgress = 0.0;
        while (total < queriesSubmitted) {
            ++total;
            Validator validator = (Validator)Verifier.takeUnchecked(completionService);
            if (validator.isSkipped()) {
                if (!this.config.isQuiet()) {
                    log.warn("%s", new Object[]{validator.getSkippedMessage()});
                }
                ++skipped;
                continue;
            }
            if (validator.valid()) {
                ++valid;
            } else {
                ++failed;
            }
            for (EventClient eventClient : this.eventClients) {
                eventClient.post((Object[])new VerifierQueryEvent[]{this.buildEvent(validator)});
            }
            double progress = (double)total / (double)totalQueries * 100.0;
            if (this.config.isQuiet() && !(progress - lastProgress > 1.0)) continue;
            log.info("Progress: %s valid, %s failed, %s skipped, %.2f%% done", new Object[]{valid, failed, skipped, progress});
            lastProgress = progress;
        }
        log.info("Results: %s / %s (%s skipped)", new Object[]{valid, failed, skipped});
        return failed;
    }

    private VerifierQueryEvent buildEvent(Validator validator) {
        String errorMessage = null;
        QueryPair queryPair = validator.getQueryPair();
        QueryResult control = validator.getControlResult();
        QueryResult test = validator.getTestResult();
        if (!validator.valid()) {
            errorMessage = String.format("Test state %s, Control state %s\n", new Object[]{test.getState(), control.getState()});
            errorMessage = test.getException() != null ? errorMessage + Throwables.getStackTraceAsString((Throwable)test.getException()) : errorMessage + validator.getResultsComparison(this.precision).trim();
        }
        return new VerifierQueryEvent(queryPair.getSuite(), this.config.getRunId(), this.config.getSource(), queryPair.getName(), !validator.valid(), queryPair.getTest().getCatalog(), queryPair.getTest().getSchema(), queryPair.getTest().getQuery(), null, Verifier.optionalDurationToSeconds(test), queryPair.getControl().getCatalog(), queryPair.getControl().getSchema(), queryPair.getControl().getQuery(), null, Verifier.optionalDurationToSeconds(control), errorMessage);
    }

    private static Double optionalDurationToSeconds(QueryResult test) {
        return test.getDuration() != null ? Double.valueOf(test.getDuration().convertTo(TimeUnit.SECONDS).getValue()) : null;
    }

    private static <T> T takeUnchecked(CompletionService<T> completionService) throws InterruptedException {
        try {
            return completionService.take().get();
        }
        catch (ExecutionException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    private static Runnable validateTask(final Validator validator) {
        return new Runnable(){

            @Override
            public void run() {
                validator.valid();
            }
        };
    }
}

