/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.orc;

import com.facebook.presto.orc.ChunkedSliceOutput;
import com.facebook.presto.orc.ColumnWriterOptions;
import com.facebook.presto.orc.DwrfDataEncryptor;
import com.facebook.presto.orc.OrcEncryptionException;
import com.facebook.presto.orc.checkpoint.InputStreamCheckpoint;
import com.facebook.presto.orc.metadata.CompressionKind;
import com.facebook.presto.orc.writer.CompressionBufferPool;
import com.facebook.presto.orc.zlib.DeflateCompressor;
import com.facebook.presto.orc.zstd.ZstdJniCompressor;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.airlift.compress.Compressor;
import io.airlift.compress.lz4.Lz4Compressor;
import io.airlift.compress.snappy.SnappyCompressor;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceOutput;
import io.airlift.slice.Slices;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Nullable;
import org.openjdk.jol.info.ClassLayout;
import sun.misc.Unsafe;

public class OrcOutputBuffer
extends SliceOutput {
    private static final int INSTANCE_SIZE = ClassLayout.parseClass(OrcOutputBuffer.class).instanceSize();
    private static final int PAGE_HEADER_SIZE = 3;
    private static final int INITIAL_BUFFER_SIZE = 256;
    private final int maxBufferSize;
    private final int minOutputBufferChunkSize;
    private final int maxOutputBufferChunkSize;
    private final int minCompressibleSize;
    private final CompressionBufferPool compressionBufferPool;
    private final Optional<DwrfDataEncryptor> dwrfEncryptor;
    @Nullable
    private final Compressor compressor;
    private ChunkedSliceOutput compressedOutputStream;
    private Slice slice;
    private byte[] buffer;
    private long bufferOffset;
    private int bufferPosition;

    public OrcOutputBuffer(ColumnWriterOptions columnWriterOptions, Optional<DwrfDataEncryptor> dwrfEncryptor) {
        Objects.requireNonNull(columnWriterOptions, "columnWriterOptions is null");
        Objects.requireNonNull(dwrfEncryptor, "dwrfEncryptor is null");
        int maxBufferSize = columnWriterOptions.getCompressionMaxBufferSize();
        Preconditions.checkArgument((maxBufferSize > 3 ? 1 : 0) != 0, (Object)"maximum buffer size should be greater than page header size");
        CompressionKind compressionKind = columnWriterOptions.getCompressionKind();
        this.maxBufferSize = compressionKind == CompressionKind.NONE ? maxBufferSize : maxBufferSize - 3;
        this.minOutputBufferChunkSize = columnWriterOptions.getMinOutputBufferChunkSize();
        this.maxOutputBufferChunkSize = columnWriterOptions.getMaxOutputBufferChunkSize();
        this.minCompressibleSize = compressionKind.getMinCompressibleSize();
        this.buffer = new byte[256];
        this.slice = Slices.wrappedBuffer((byte[])this.buffer);
        this.compressionBufferPool = columnWriterOptions.getCompressionBufferPool();
        this.dwrfEncryptor = Objects.requireNonNull(dwrfEncryptor, "dwrfEncryptor is null");
        if (compressionKind == CompressionKind.NONE) {
            this.compressor = null;
        } else if (compressionKind == CompressionKind.SNAPPY) {
            this.compressor = new SnappyCompressor();
        } else if (compressionKind == CompressionKind.ZLIB) {
            this.compressor = new DeflateCompressor(columnWriterOptions.getCompressionLevel());
        } else if (compressionKind == CompressionKind.LZ4) {
            this.compressor = new Lz4Compressor();
        } else if (compressionKind == CompressionKind.ZSTD) {
            this.compressor = new ZstdJniCompressor(columnWriterOptions.getCompressionLevel());
        } else {
            throw new IllegalArgumentException("Unsupported compression " + (Object)((Object)compressionKind));
        }
    }

    public long getOutputDataSize() {
        Preconditions.checkState((this.bufferPosition == 0 ? 1 : 0) != 0, (Object)"Buffer must be flushed before getOutputDataSize can be called");
        return this.getCompressedOutputSize();
    }

    private int getCompressedOutputSize() {
        return this.compressedOutputStream != null ? this.compressedOutputStream.size() : 0;
    }

    public long estimateOutputDataSize() {
        return this.getCompressedOutputSize() + this.bufferPosition;
    }

    public int writeDataTo(SliceOutput outputStream) {
        Preconditions.checkState((this.bufferPosition == 0 ? 1 : 0) != 0, (Object)"Buffer must be closed before writeDataTo can be called");
        if (this.compressedOutputStream == null) {
            return 0;
        }
        this.compressedOutputStream.writeTo(outputStream);
        return this.compressedOutputStream.size();
    }

    public long getCheckpoint() {
        if (this.compressor == null && !this.dwrfEncryptor.isPresent()) {
            return this.size();
        }
        return InputStreamCheckpoint.createInputStreamCheckpoint(this.getCompressedOutputSize(), this.bufferPosition);
    }

    public void flush() {
        this.flushBufferToOutputStream();
    }

    public void close() {
        this.flushBufferToOutputStream();
    }

    public void reset() {
        if (this.compressedOutputStream != null) {
            this.compressedOutputStream.reset();
        }
        this.bufferOffset = 0L;
        this.bufferPosition = 0;
    }

    public void reset(int position) {
        throw new UnsupportedOperationException();
    }

    public int size() {
        return Math.toIntExact(this.bufferOffset + (long)this.bufferPosition);
    }

    public long getRetainedSize() {
        return (long)INSTANCE_SIZE + (this.compressedOutputStream != null ? this.compressedOutputStream.getRetainedSize() : 0L) + this.slice.getRetainedSize();
    }

    public int writableBytes() {
        return Integer.MAX_VALUE;
    }

    public boolean isWritable() {
        return true;
    }

    public void writeByte(int value) {
        this.ensureWritableBytes(1);
        this.slice.setByte(this.bufferPosition, value);
        ++this.bufferPosition;
    }

    public void writeShort(int value) {
        this.ensureWritableBytes(2);
        this.slice.setShort(this.bufferPosition, value);
        this.bufferPosition += 2;
    }

    public void writeInt(int value) {
        this.ensureWritableBytes(4);
        this.slice.setInt(this.bufferPosition, value);
        this.bufferPosition += 4;
    }

    public void writeLong(long value) {
        this.ensureWritableBytes(8);
        this.slice.setLong(this.bufferPosition, value);
        this.bufferPosition += 8;
    }

    public void writeFloat(float value) {
        this.writeInt(Float.floatToIntBits(value));
    }

    public void writeDouble(double value) {
        this.writeLong(Double.doubleToLongBits(value));
    }

    public void writeBytes(Slice source) {
        this.writeBytes(source, 0, source.length());
    }

    public void writeBytes(Slice source, int sourceOffset, int length) {
        byte[] bytes = (byte[])source.getBase();
        int bytesOffset = (int)(source.getAddress() - (long)Unsafe.ARRAY_BYTE_BASE_OFFSET);
        this.writeBytes(bytes, sourceOffset + bytesOffset, length);
    }

    public void writeBytes(byte[] source) {
        this.writeBytes(source, 0, source.length);
    }

    public void writeBytes(byte[] bytes, int bytesOffset, int length) {
        if (length == 0) {
            return;
        }
        if (this.bufferPosition != 0) {
            int chunkSize = Math.min(length, this.maxBufferSize - this.bufferPosition);
            this.ensureWritableBytes(chunkSize);
            this.slice.setBytes(this.bufferPosition, bytes, bytesOffset, chunkSize);
            this.bufferPosition += chunkSize;
            length -= chunkSize;
            bytesOffset += chunkSize;
        }
        if (length >= this.maxBufferSize) {
            this.flushBufferToOutputStream();
            int bytesOffsetBefore = bytesOffset;
            while (length >= this.maxBufferSize) {
                this.writeChunkToOutputStream(bytes, bytesOffset, this.maxBufferSize);
                length -= this.maxBufferSize;
                bytesOffset += this.maxBufferSize;
            }
            this.bufferOffset += (long)(bytesOffset - bytesOffsetBefore);
        }
        if (length > 0) {
            this.ensureWritableBytes(length);
            this.slice.setBytes(this.bufferPosition, bytes, bytesOffset, length);
            this.bufferPosition += length;
        }
    }

    public void writeBytes(InputStream in, int length) throws IOException {
        while (length > 0) {
            int batch = this.ensureBatchSize(length);
            this.slice.setBytes(this.bufferPosition, in, batch);
            this.bufferPosition += batch;
            length -= batch;
        }
    }

    public void writeZero(int length) {
        Preconditions.checkArgument((length >= 0 ? 1 : 0) != 0, (Object)"length must be 0 or greater than 0.");
        while (length > 0) {
            int batch = this.ensureBatchSize(length);
            Arrays.fill(this.buffer, this.bufferPosition, this.bufferPosition + batch, (byte)0);
            this.bufferPosition += batch;
            length -= batch;
        }
    }

    private int ensureBatchSize(int length) {
        this.ensureWritableBytes(Math.min(length, this.maxBufferSize - this.bufferPosition));
        if (this.availableInBuffer() == 0) {
            this.flushBufferToOutputStream();
        }
        return Math.min(length, this.availableInBuffer());
    }

    private int availableInBuffer() {
        return this.slice.length() - this.bufferPosition;
    }

    public SliceOutput appendLong(long value) {
        this.writeLong(value);
        return this;
    }

    public SliceOutput appendDouble(double value) {
        this.writeDouble(value);
        return this;
    }

    public SliceOutput appendInt(int value) {
        this.writeInt(value);
        return this;
    }

    public SliceOutput appendShort(int value) {
        this.writeShort(value);
        return this;
    }

    public SliceOutput appendByte(int value) {
        this.writeByte(value);
        return this;
    }

    public SliceOutput appendBytes(byte[] source, int sourceIndex, int length) {
        this.writeBytes(source, sourceIndex, length);
        return this;
    }

    public SliceOutput appendBytes(byte[] source) {
        this.writeBytes(source);
        return this;
    }

    public SliceOutput appendBytes(Slice slice) {
        this.writeBytes(slice);
        return this;
    }

    public Slice slice() {
        throw new UnsupportedOperationException();
    }

    public Slice getUnderlyingSlice() {
        throw new UnsupportedOperationException();
    }

    public String toString(Charset charset) {
        throw new UnsupportedOperationException();
    }

    public String toString() {
        StringBuilder builder = new StringBuilder("OrcOutputBuffer{");
        builder.append("outputStream=").append((Object)this.compressedOutputStream);
        builder.append(", bufferSize=").append(this.slice.length());
        builder.append('}');
        return builder.toString();
    }

    private void ensureWritableBytes(int minWritableBytes) {
        Preconditions.checkArgument((minWritableBytes <= this.maxBufferSize ? 1 : 0) != 0, (Object)"Min writable bytes must not exceed max buffer size");
        int neededBufferSize = this.bufferPosition + minWritableBytes;
        if (neededBufferSize <= this.slice.length()) {
            return;
        }
        if (this.slice.length() >= this.maxBufferSize) {
            this.flushBufferToOutputStream();
            return;
        }
        int newBufferSize = Math.min(Math.max(this.slice.length() * 2, neededBufferSize), this.maxBufferSize);
        if (newBufferSize >= neededBufferSize) {
            byte[] previousBuffer = this.buffer;
            this.buffer = new byte[newBufferSize];
            this.slice = Slices.wrappedBuffer((byte[])this.buffer);
            System.arraycopy(previousBuffer, 0, this.buffer, 0, this.bufferPosition);
        } else {
            this.flushBufferToOutputStream();
            this.buffer = new byte[newBufferSize];
            this.slice = Slices.wrappedBuffer((byte[])this.buffer);
        }
    }

    private void flushBufferToOutputStream() {
        if (this.bufferPosition > 0) {
            this.writeChunkToOutputStream(this.buffer, 0, this.bufferPosition);
            this.bufferOffset += (long)this.bufferPosition;
            this.bufferPosition = 0;
        }
    }

    private void initCompressedOutputStream() {
        Preconditions.checkState((this.compressedOutputStream == null ? 1 : 0) != 0, (Object)"compressedOutputStream is already initialized");
        this.compressedOutputStream = new ChunkedSliceOutput(this.minOutputBufferChunkSize, this.maxOutputBufferChunkSize);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeChunkToOutputStream(byte[] chunk, int offset, int length) {
        if (this.compressedOutputStream == null) {
            this.initCompressedOutputStream();
        }
        if (this.compressor == null && !this.dwrfEncryptor.isPresent()) {
            this.compressedOutputStream.write(chunk, offset, length);
            return;
        }
        Preconditions.checkArgument((length <= this.maxBufferSize ? 1 : 0) != 0, (Object)"Write chunk length must be less than max compression buffer size");
        boolean isCompressed = false;
        byte[] compressionBuffer = null;
        try {
            int minCompressionBufferSize;
            int compressedSize;
            if (this.compressor != null && length >= this.minCompressibleSize && (compressedSize = this.compressor.compress(chunk, offset, length, compressionBuffer = this.compressionBufferPool.checkOut(minCompressionBufferSize = this.compressor.maxCompressedLength(length)), 0, compressionBuffer.length)) < length) {
                isCompressed = true;
                chunk = compressionBuffer;
                length = compressedSize;
                offset = 0;
            }
            if (this.dwrfEncryptor.isPresent()) {
                chunk = this.dwrfEncryptor.get().encrypt(chunk, offset, length);
                length = chunk.length;
                offset = 0;
                if (length > 0x800000) {
                    throw new OrcEncryptionException("Encrypted data size %s exceeds limit of 2^23", length);
                }
            }
            int header = isCompressed ? length << 1 : (length << 1) + 1;
            this.writeChunkedOutput(chunk, offset, length, header);
            if (compressionBuffer != null) {
                this.compressionBufferPool.checkIn(compressionBuffer);
            }
        }
        catch (Throwable throwable) {
            if (compressionBuffer != null) {
                this.compressionBufferPool.checkIn(compressionBuffer);
            }
            throw throwable;
        }
    }

    private void writeChunkedOutput(byte[] chunk, int offset, int length, int header) {
        this.compressedOutputStream.write(header & 0xFF);
        this.compressedOutputStream.write((header & 0xFF00) >> 8);
        this.compressedOutputStream.write((header & 0xFF0000) >> 16);
        this.compressedOutputStream.writeBytes(chunk, offset, length);
    }

    @VisibleForTesting
    int getBufferCapacity() {
        return this.slice.length();
    }
}

