/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.orc;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.primitives.Longs;
import io.airlift.units.DataSize;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

public class DictionaryCompressionOptimizer {
    private static final double DICTIONARY_MIN_COMPRESSION_RATIO = 1.25;
    private static final double DICTIONARY_ALWAYS_KEEP_COMPRESSION_RATIO = 3.0;
    static final DataSize DICTIONARY_MEMORY_MAX_RANGE = new DataSize(4.0, DataSize.Unit.MEGABYTE);
    private final Set<DictionaryColumnManager> allWriters;
    private final Set<DictionaryColumnManager> dictionaryWriters = new HashSet<DictionaryColumnManager>();
    private final int stripeMaxBytes;
    private final int stripeMinRowCount;
    private final int stripeMaxRowCount;
    private final int dictionaryMemoryMaxBytesLow;
    private final int dictionaryMemoryMaxBytesHigh;
    private int dictionaryMemoryBytes;
    private int stripeRowCount;

    public DictionaryCompressionOptimizer(Set<? extends DictionaryColumn> writers, int stripeMaxBytes, int stripeMinRowCount, int stripeMaxRowCount, int dictionaryMemoryMaxBytes) {
        Objects.requireNonNull(writers, "writers is null");
        this.allWriters = ImmutableSet.copyOf((Collection)writers.stream().map(DictionaryColumnManager::new).collect(Collectors.toSet()));
        Preconditions.checkArgument((stripeMaxBytes >= 0 ? 1 : 0) != 0, (Object)"stripeMaxBytes is negative");
        this.stripeMaxBytes = stripeMaxBytes;
        Preconditions.checkArgument((stripeMinRowCount >= 0 ? 1 : 0) != 0, (Object)"stripeMinRowCount is negative");
        this.stripeMinRowCount = stripeMinRowCount;
        Preconditions.checkArgument((stripeMaxRowCount >= stripeMinRowCount ? 1 : 0) != 0, (Object)"stripeMaxRowCount is less than stripeMinRowCount");
        this.stripeMaxRowCount = stripeMaxRowCount;
        Preconditions.checkArgument((dictionaryMemoryMaxBytes >= 0 ? 1 : 0) != 0, (Object)"dictionaryMemoryMaxBytes is negative");
        this.dictionaryMemoryMaxBytesHigh = dictionaryMemoryMaxBytes;
        this.dictionaryMemoryMaxBytesLow = (int)Math.max((long)dictionaryMemoryMaxBytes - DICTIONARY_MEMORY_MAX_RANGE.toBytes(), 0L);
        this.dictionaryWriters.addAll(this.allWriters);
    }

    public boolean isFull() {
        if (this.stripeRowCount > this.stripeMinRowCount) {
            return this.dictionaryMemoryBytes > this.dictionaryMemoryMaxBytesLow;
        }
        return this.dictionaryMemoryBytes > this.dictionaryMemoryMaxBytesHigh;
    }

    public void reset() {
        this.dictionaryWriters.clear();
        this.dictionaryWriters.addAll(this.allWriters);
        this.dictionaryMemoryBytes = 0;
        this.allWriters.forEach(DictionaryColumnManager::reset);
    }

    public void finalOptimize() {
        this.convertLowCompressionStreams();
    }

    public void optimize(int bufferedBytes, int stripeRowCount) {
        this.dictionaryMemoryBytes = this.dictionaryWriters.stream().mapToInt(DictionaryColumnManager::getDictionaryBytes).sum();
        this.dictionaryWriters.forEach(column -> column.updateHistory(stripeRowCount));
        this.stripeRowCount = stripeRowCount;
        if (this.dictionaryMemoryBytes <= this.dictionaryMemoryMaxBytesLow) {
            return;
        }
        this.convertLowCompressionStreams();
        if (this.dictionaryMemoryBytes <= this.dictionaryMemoryMaxBytesLow) {
            return;
        }
        int nonDictionaryBufferedBytes = bufferedBytes;
        for (DictionaryColumnManager dictionaryWriter : this.dictionaryWriters) {
            nonDictionaryBufferedBytes = (int)((long)nonDictionaryBufferedBytes - dictionaryWriter.getBufferedBytes());
        }
        while (this.dictionaryMemoryBytes > this.dictionaryMemoryMaxBytesHigh) {
            DictionaryCompressionProjection projection = this.selectDictionaryColumnToConvert(nonDictionaryBufferedBytes, stripeRowCount);
            if (projection.getColumnToConvert().getCompressionRatio() >= 3.0) {
                return;
            }
            this.convertToDirect(projection.getColumnToConvert());
        }
        if (stripeRowCount >= this.stripeMinRowCount) {
            double currentCompressionRatio = this.currentCompressionRatio(nonDictionaryBufferedBytes);
            while (!this.dictionaryWriters.isEmpty()) {
                DictionaryCompressionProjection projection = this.selectDictionaryColumnToConvert(nonDictionaryBufferedBytes, stripeRowCount);
                if (projection.getPredictedFileCompressionRatio() < currentCompressionRatio) {
                    return;
                }
                this.convertToDirect(projection.getColumnToConvert());
            }
        }
    }

    private void convertLowCompressionStreams() {
        ImmutableList.copyOf(this.dictionaryWriters).stream().filter(dictionaryWriter -> dictionaryWriter.getCompressionRatio() < 1.25).forEach(this::convertToDirect);
    }

    private void convertToDirect(DictionaryColumnManager dictionaryWriter) {
        this.dictionaryMemoryBytes -= dictionaryWriter.getDictionaryBytes();
        dictionaryWriter.convertToDirect();
        this.dictionaryWriters.remove(dictionaryWriter);
    }

    private double currentCompressionRatio(int allOtherBytes) {
        long uncompressedBytes = allOtherBytes;
        long compressedBytes = allOtherBytes;
        for (DictionaryColumnManager column : this.dictionaryWriters) {
            uncompressedBytes += column.getRawBytes();
            compressedBytes += (long)column.getDictionaryBytes();
        }
        return 1.0 * (double)uncompressedBytes / (double)compressedBytes;
    }

    private DictionaryCompressionProjection selectDictionaryColumnToConvert(int totalNonDictionaryBytes, int stripeRowCount) {
        Preconditions.checkState((!this.dictionaryWriters.isEmpty() ? 1 : 0) != 0);
        int totalNonDictionaryBytesPerRow = totalNonDictionaryBytes / stripeRowCount;
        long totalDictionaryRawBytes = 0L;
        long totalDictionaryBytes = 0L;
        long totalDictionaryIndexBytes = 0L;
        long totalDictionaryRawBytesPerRow = 0L;
        long totalDictionaryBytesPerNewRow = 0L;
        long totalDictionaryIndexBytesPerRow = 0L;
        for (DictionaryColumnManager column : this.dictionaryWriters) {
            totalDictionaryRawBytes += column.getRawBytes();
            totalDictionaryBytes += (long)column.getDictionaryBytes();
            totalDictionaryIndexBytes += (long)column.getIndexBytes();
            totalDictionaryRawBytesPerRow = (long)((double)totalDictionaryRawBytesPerRow + column.getRawBytesPerRow());
            totalDictionaryBytesPerNewRow = (long)((double)totalDictionaryBytesPerNewRow + column.getDictionaryBytesPerFutureRow());
            totalDictionaryIndexBytesPerRow = (long)((double)totalDictionaryIndexBytesPerRow + column.getIndexBytesPerRow());
        }
        long totalUncompressedBytesPerRow = (long)totalNonDictionaryBytesPerRow + totalDictionaryRawBytesPerRow;
        DictionaryCompressionProjection maxProjectedCompression = null;
        for (DictionaryColumnManager column : this.dictionaryWriters) {
            long currentRawBytes = (long)totalNonDictionaryBytes + column.getRawBytes();
            long currentDictionaryBytes = totalDictionaryBytes - (long)column.getDictionaryBytes();
            long currentIndexBytes = totalDictionaryIndexBytes - (long)column.getIndexBytes();
            long currentTotalBytes = currentRawBytes + currentDictionaryBytes + currentIndexBytes;
            double rawBytesPerFutureRow = (double)totalNonDictionaryBytesPerRow + column.getRawBytesPerRow();
            double dictionaryBytesPerFutureRow = (double)totalDictionaryBytesPerNewRow - column.getDictionaryBytesPerFutureRow();
            double indexBytesPerFutureRow = (double)totalDictionaryIndexBytesPerRow - column.getIndexBytesPerRow();
            double totalBytesPerFutureRow = rawBytesPerFutureRow + dictionaryBytesPerFutureRow + indexBytesPerFutureRow;
            long rowsToDictionaryMemoryLimit = (long)((double)((long)this.dictionaryMemoryMaxBytesLow - currentDictionaryBytes) / dictionaryBytesPerFutureRow);
            long rowsToStripeMemoryLimit = (long)((double)((long)this.stripeMaxBytes - currentTotalBytes) / totalBytesPerFutureRow);
            long rowsToStripeRowLimit = this.stripeMaxRowCount - stripeRowCount;
            long rowsToLimit = Longs.min((long[])new long[]{rowsToDictionaryMemoryLimit, rowsToStripeMemoryLimit, rowsToStripeRowLimit});
            long predictedUncompressedSizeAtLimit = (long)totalNonDictionaryBytes + totalDictionaryRawBytes + totalUncompressedBytesPerRow * rowsToLimit;
            long predictedCompressedSizeAtLimit = (long)((double)currentTotalBytes + totalBytesPerFutureRow * (double)rowsToLimit);
            double predictedCompressionRatioAtLimit = 1.0 * (double)predictedUncompressedSizeAtLimit / (double)predictedCompressedSizeAtLimit;
            if (maxProjectedCompression != null && !(maxProjectedCompression.getPredictedFileCompressionRatio() < predictedCompressionRatioAtLimit)) continue;
            maxProjectedCompression = new DictionaryCompressionProjection(column, predictedCompressionRatioAtLimit);
        }
        return maxProjectedCompression;
    }

    public static int estimateIndexBytesPerValue(int dictionaryEntries) {
        if (dictionaryEntries <= 256) {
            return 1;
        }
        if (dictionaryEntries <= 65536) {
            return 2;
        }
        if (dictionaryEntries <= 0x1000000) {
            return 3;
        }
        return 4;
    }

    private static class DictionaryCompressionProjection {
        private final DictionaryColumnManager columnToConvert;
        private final double predictedFileCompressionRatio;

        public DictionaryCompressionProjection(DictionaryColumnManager columnToConvert, double predictedFileCompressionRatio) {
            this.columnToConvert = Objects.requireNonNull(columnToConvert, "columnToConvert is null");
            this.predictedFileCompressionRatio = predictedFileCompressionRatio;
        }

        public DictionaryColumnManager getColumnToConvert() {
            return this.columnToConvert;
        }

        public double getPredictedFileCompressionRatio() {
            return this.predictedFileCompressionRatio;
        }
    }

    private static class DictionaryColumnManager {
        private final DictionaryColumn dictionaryColumn;
        private boolean directEncoded;
        private int rowCount;
        private int pastValueCount;
        private int pastDictionaryEntries;
        private int pendingPastValueCount;
        private int pendingPastDictionaryEntries;

        public DictionaryColumnManager(DictionaryColumn dictionaryColumn) {
            this.dictionaryColumn = dictionaryColumn;
        }

        void convertToDirect() {
            this.directEncoded = true;
            this.dictionaryColumn.convertToDirect();
        }

        void reset() {
            this.directEncoded = false;
            this.pastValueCount = 0;
            this.pastDictionaryEntries = 0;
            this.pendingPastValueCount = 0;
            this.pendingPastDictionaryEntries = 0;
        }

        public void updateHistory(int rowCount) {
            this.rowCount = rowCount;
            int currentValueCount = this.dictionaryColumn.getValueCount();
            if (currentValueCount - this.pendingPastValueCount >= 1024) {
                this.pastValueCount = this.pendingPastValueCount;
                this.pastDictionaryEntries = this.pendingPastDictionaryEntries;
                this.pendingPastValueCount = currentValueCount;
                this.pendingPastDictionaryEntries = this.dictionaryColumn.getDictionaryEntries();
            }
        }

        public long getRawBytes() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            return this.dictionaryColumn.getRawBytes();
        }

        public double getRawBytesPerRow() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            return 1.0 * (double)this.getRawBytes() / (double)this.rowCount;
        }

        public int getDictionaryBytes() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            return this.dictionaryColumn.getDictionaryBytes();
        }

        public double getDictionaryBytesPerFutureRow() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            int currentDictionaryEntries = this.dictionaryColumn.getDictionaryEntries();
            int currentValueCount = this.dictionaryColumn.getValueCount();
            double dictionaryBytesPerEntry = 1.0 * (double)this.dictionaryColumn.getDictionaryBytes() / (double)currentDictionaryEntries;
            double dictionaryEntriesPerFutureValue = 1.0 * (double)(currentDictionaryEntries - this.pastDictionaryEntries) / (double)(currentValueCount - this.pastValueCount);
            return dictionaryBytesPerEntry * dictionaryEntriesPerFutureValue;
        }

        public int getIndexBytes() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            return DictionaryCompressionOptimizer.estimateIndexBytesPerValue(this.dictionaryColumn.getDictionaryEntries()) * this.dictionaryColumn.getNonNullValueCount();
        }

        public double getIndexBytesPerRow() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            return 1.0 * (double)this.getIndexBytes() / (double)this.rowCount;
        }

        public int getNullsBytes() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            return (this.dictionaryColumn.getValueCount() - this.dictionaryColumn.getNonNullValueCount() + 7) / 8;
        }

        public int getCompressedBytes() {
            return this.getDictionaryBytes() + this.getIndexBytes() + this.getNullsBytes();
        }

        public double getCompressionRatio() {
            Preconditions.checkState((!this.directEncoded ? 1 : 0) != 0);
            return 1.0 * (double)this.getRawBytes() / (double)this.getCompressedBytes();
        }

        public long getBufferedBytes() {
            return this.getIndexBytes() + this.getDictionaryBytes();
        }
    }

    public static interface DictionaryColumn {
        public int getValueCount();

        public int getNonNullValueCount();

        public long getRawBytes();

        public int getDictionaryEntries();

        public int getDictionaryBytes();

        public void convertToDirect();
    }
}

