/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.mongodb;

import com.facebook.presto.mongodb.MongoClientConfig;
import com.facebook.presto.mongodb.MongoColumnHandle;
import com.facebook.presto.mongodb.MongoIndex;
import com.facebook.presto.mongodb.MongoSplit;
import com.facebook.presto.mongodb.MongoTable;
import com.facebook.presto.mongodb.MongoTableHandle;
import com.facebook.presto.mongodb.ObjectIdType;
import com.facebook.presto.spi.ColumnHandle;
import com.facebook.presto.spi.SchemaNotFoundException;
import com.facebook.presto.spi.SchemaTableName;
import com.facebook.presto.spi.TableNotFoundException;
import com.facebook.presto.spi.predicate.Domain;
import com.facebook.presto.spi.predicate.Range;
import com.facebook.presto.spi.predicate.TupleDomain;
import com.facebook.presto.spi.type.BigintType;
import com.facebook.presto.spi.type.BooleanType;
import com.facebook.presto.spi.type.DoubleType;
import com.facebook.presto.spi.type.NamedTypeSignature;
import com.facebook.presto.spi.type.TimestampType;
import com.facebook.presto.spi.type.Type;
import com.facebook.presto.spi.type.TypeManager;
import com.facebook.presto.spi.type.TypeSignature;
import com.facebook.presto.spi.type.TypeSignatureParameter;
import com.facebook.presto.spi.type.VarcharType;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.Verify;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.util.concurrent.UncheckedExecutionException;
import com.mongodb.MongoClient;
import com.mongodb.client.FindIterable;
import com.mongodb.client.ListIndexesIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.IndexOptions;
import com.mongodb.client.result.DeleteResult;
import io.airlift.log.Logger;
import io.airlift.slice.Slice;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.bson.types.ObjectId;

public class MongoSession {
    private static final Logger log = Logger.get(MongoSession.class);
    private static final List<String> SYSTEM_TABLES = Arrays.asList("system.indexes", "system.users", "system.version");
    private static final String TABLE_NAME_KEY = "table";
    private static final String FIELDS_KEY = "fields";
    private static final String FIELDS_NAME_KEY = "name";
    private static final String FIELDS_TYPE_KEY = "type";
    private static final String FIELDS_HIDDEN_KEY = "hidden";
    private static final String OR_OP = "$or";
    private static final String AND_OP = "$and";
    private static final String NOT_OP = "$not";
    private static final String NOR_OP = "$nor";
    private static final String EQ_OP = "$eq";
    private static final String NOT_EQ_OP = "$ne";
    private static final String EXISTS_OP = "$exists";
    private static final String GTE_OP = "$gte";
    private static final String GT_OP = "$gt";
    private static final String LT_OP = "$lt";
    private static final String LTE_OP = "$lte";
    private static final String IN_OP = "$in";
    private static final String NOTIN_OP = "$nin";
    private final TypeManager typeManager;
    private final MongoClient client;
    private final String schemaCollection;
    private final int cursorBatchSize;
    private final LoadingCache<SchemaTableName, MongoTable> tableCache;
    private final String implicitPrefix;

    public MongoSession(TypeManager typeManager, MongoClient client, MongoClientConfig config) {
        this.typeManager = Objects.requireNonNull(typeManager, "typeManager is null");
        this.client = Objects.requireNonNull(client, "client is null");
        this.schemaCollection = config.getSchemaCollection();
        this.cursorBatchSize = config.getCursorBatchSize();
        this.implicitPrefix = config.getImplicitRowFieldPrefix();
        this.tableCache = CacheBuilder.newBuilder().expireAfterWrite(1L, TimeUnit.HOURS).refreshAfterWrite(1L, TimeUnit.MINUTES).build((CacheLoader)new CacheLoader<SchemaTableName, MongoTable>(){

            public MongoTable load(SchemaTableName key) throws TableNotFoundException {
                return MongoSession.this.loadTableSchema(key);
            }
        });
    }

    public void shutdown() {
        this.client.close();
    }

    public List<String> getAllSchemas() {
        return ImmutableList.copyOf((Iterable)this.client.listDatabaseNames());
    }

    public Set<String> getAllTables(String schema) throws SchemaNotFoundException {
        ImmutableSet.Builder builder = ImmutableSet.builder();
        builder.addAll((Iterable)ImmutableList.copyOf((Iterable)this.client.getDatabase(schema).listCollectionNames()).stream().filter(name -> !name.equals(this.schemaCollection)).filter(name -> !SYSTEM_TABLES.contains(name)).collect(Collectors.toSet()));
        builder.addAll(this.getTableMetadataNames(schema));
        return builder.build();
    }

    public MongoTable getTable(SchemaTableName tableName) throws TableNotFoundException {
        return MongoSession.getCacheValue(this.tableCache, tableName, TableNotFoundException.class);
    }

    public void createTable(SchemaTableName name, List<MongoColumnHandle> columns) {
        this.createTableMetadata(name, columns);
    }

    public void dropTable(SchemaTableName tableName) {
        this.deleteTableMetadata(tableName);
        this.getCollection(tableName).drop();
        this.tableCache.invalidate((Object)tableName);
    }

    private MongoTable loadTableSchema(SchemaTableName tableName) throws TableNotFoundException {
        Document tableMeta = this.getTableMetadata(tableName);
        ImmutableList.Builder columnHandles = ImmutableList.builder();
        for (Document columnMetadata : this.getColumnMetadata(tableMeta)) {
            MongoColumnHandle columnHandle = this.buildColumnHandle(columnMetadata);
            columnHandles.add((Object)columnHandle);
        }
        MongoTableHandle tableHandle = new MongoTableHandle(tableName);
        return new MongoTable(tableHandle, (List<MongoColumnHandle>)columnHandles.build(), this.getIndexes(tableName));
    }

    private MongoColumnHandle buildColumnHandle(Document columnMeta) {
        String name = columnMeta.getString((Object)FIELDS_NAME_KEY);
        String typeString = columnMeta.getString((Object)FIELDS_TYPE_KEY);
        boolean hidden = columnMeta.getBoolean((Object)FIELDS_HIDDEN_KEY, false);
        Type type = this.typeManager.getType(TypeSignature.parseTypeSignature((String)typeString));
        return new MongoColumnHandle(name, type, hidden);
    }

    private List<Document> getColumnMetadata(Document doc) {
        if (!doc.containsKey((Object)FIELDS_KEY)) {
            return ImmutableList.of();
        }
        return (List)doc.get((Object)FIELDS_KEY);
    }

    public MongoCollection<Document> getCollection(SchemaTableName tableName) {
        return this.getCollection(tableName.getSchemaName(), tableName.getTableName());
    }

    private MongoCollection<Document> getCollection(String schema, String table) {
        return this.client.getDatabase(schema).getCollection(table);
    }

    public List<MongoIndex> getIndexes(SchemaTableName tableName) {
        return MongoIndex.parse((ListIndexesIterable<Document>)this.getCollection(tableName).listIndexes());
    }

    private static <K, V, E extends Exception> V getCacheValue(LoadingCache<K, V> cache, K key, Class<E> exceptionClass) throws E {
        try {
            return (V)cache.get(key);
        }
        catch (UncheckedExecutionException | ExecutionException e) {
            Throwable t = e.getCause();
            if (t != null) {
                Throwables.throwIfInstanceOf((Throwable)t, exceptionClass);
            }
            throw Throwables.propagate((Throwable)t);
        }
    }

    public MongoCursor<Document> execute(MongoSplit split, List<MongoColumnHandle> columns) {
        Document output = new Document();
        for (MongoColumnHandle column : columns) {
            output.append(column.getName(), (Object)1);
        }
        MongoCollection<Document> collection = this.getCollection(split.getSchemaTableName());
        FindIterable iterable = collection.find((Bson)MongoSession.buildQuery(split.getTupleDomain())).projection((Bson)output);
        if (this.cursorBatchSize != 0) {
            iterable.batchSize(this.cursorBatchSize);
        }
        return iterable.iterator();
    }

    @VisibleForTesting
    static Document buildQuery(TupleDomain<ColumnHandle> tupleDomain) {
        Document query = new Document();
        if (tupleDomain.getDomains().isPresent()) {
            for (Map.Entry entry : ((Map)tupleDomain.getDomains().get()).entrySet()) {
                MongoColumnHandle column = (MongoColumnHandle)entry.getKey();
                query.putAll((Map)MongoSession.buildPredicate(column, (Domain)entry.getValue()));
            }
        }
        return query;
    }

    private static Document buildPredicate(MongoColumnHandle column, Domain domain) {
        String name = column.getName();
        Type type = column.getType();
        if (domain.getValues().isNone() && domain.isNullAllowed()) {
            return MongoSession.documentOf(name, MongoSession.isNullPredicate());
        }
        if (domain.getValues().isAll() && !domain.isNullAllowed()) {
            return MongoSession.documentOf(name, MongoSession.isNotNullPredicate());
        }
        ArrayList<Object> singleValues = new ArrayList<Object>();
        ArrayList<Document> disjuncts = new ArrayList<Document>();
        for (Range range : domain.getValues().getRanges().getOrderedRanges()) {
            if (range.isSingleValue()) {
                singleValues.add(range.getSingleValue());
                continue;
            }
            Document rangeConjuncts = new Document();
            if (!range.getLow().isLowerUnbounded()) {
                switch (range.getLow().getBound()) {
                    case ABOVE: {
                        rangeConjuncts.put(GT_OP, range.getLow().getValue());
                        break;
                    }
                    case EXACTLY: {
                        rangeConjuncts.put(GTE_OP, range.getLow().getValue());
                        break;
                    }
                    case BELOW: {
                        throw new IllegalArgumentException("Low Marker should never use BELOW bound: " + range);
                    }
                    default: {
                        throw new AssertionError((Object)("Unhandled bound: " + range.getLow().getBound()));
                    }
                }
            }
            if (!range.getHigh().isUpperUnbounded()) {
                switch (range.getHigh().getBound()) {
                    case ABOVE: {
                        throw new IllegalArgumentException("High Marker should never use ABOVE bound: " + range);
                    }
                    case EXACTLY: {
                        rangeConjuncts.put(LTE_OP, range.getHigh().getValue());
                        break;
                    }
                    case BELOW: {
                        rangeConjuncts.put(LT_OP, range.getHigh().getValue());
                        break;
                    }
                    default: {
                        throw new AssertionError((Object)("Unhandled bound: " + range.getHigh().getBound()));
                    }
                }
            }
            Verify.verify((!rangeConjuncts.isEmpty() ? 1 : 0) != 0);
            disjuncts.add(rangeConjuncts);
        }
        if (singleValues.size() == 1) {
            disjuncts.add(MongoSession.documentOf(EQ_OP, MongoSession.translateValue(singleValues.get(0), type)));
        } else if (singleValues.size() > 1) {
            disjuncts.add(MongoSession.documentOf(IN_OP, singleValues.stream().map(value -> MongoSession.translateValue(value, type)).collect(Collectors.toList())));
        }
        if (domain.isNullAllowed()) {
            disjuncts.add(MongoSession.isNullPredicate());
        }
        return MongoSession.orPredicate(disjuncts.stream().map(disjunct -> new Document(name, disjunct)).collect(Collectors.toList()));
    }

    private static Object translateValue(Object source, Type type) {
        if (source instanceof Slice) {
            if (type instanceof ObjectIdType) {
                return new ObjectId(((Slice)source).getBytes());
            }
            return ((Slice)source).toStringUtf8();
        }
        return source;
    }

    private static Document documentOf(String key, Object value) {
        return new Document(key, value);
    }

    private static Document orPredicate(List<Document> values) {
        Preconditions.checkState((!values.isEmpty() ? 1 : 0) != 0);
        if (values.size() == 1) {
            return values.get(0);
        }
        return new Document(OR_OP, values);
    }

    private static Document isNullPredicate() {
        return MongoSession.documentOf(EXISTS_OP, true).append(EQ_OP, null);
    }

    private static Document isNotNullPredicate() {
        return MongoSession.documentOf(NOT_EQ_OP, null);
    }

    private Document getTableMetadata(SchemaTableName schemaTableName) throws TableNotFoundException {
        String schemaName = schemaTableName.getSchemaName();
        String tableName = schemaTableName.getTableName();
        MongoDatabase db = this.client.getDatabase(schemaName);
        MongoCollection schema = db.getCollection(this.schemaCollection);
        Document doc = (Document)schema.find((Bson)new Document(TABLE_NAME_KEY, (Object)tableName)).first();
        if (doc == null) {
            if (!this.collectionExists(db, tableName)) {
                throw new TableNotFoundException(schemaTableName);
            }
            Document metadata = new Document(TABLE_NAME_KEY, (Object)tableName);
            metadata.append(FIELDS_KEY, this.guessTableFields(schemaTableName));
            schema.createIndex((Bson)new Document(TABLE_NAME_KEY, (Object)1), new IndexOptions().unique(true));
            schema.insertOne((Object)metadata);
            return metadata;
        }
        return doc;
    }

    public boolean collectionExists(MongoDatabase db, String collectionName) {
        for (String name : db.listCollectionNames()) {
            if (!name.equalsIgnoreCase(collectionName)) continue;
            return true;
        }
        return false;
    }

    private Set<String> getTableMetadataNames(String schemaName) throws TableNotFoundException {
        MongoDatabase db = this.client.getDatabase(schemaName);
        MongoCursor cursor = db.getCollection(this.schemaCollection).find().projection((Bson)new Document(TABLE_NAME_KEY, (Object)true)).iterator();
        HashSet<String> names = new HashSet<String>();
        while (cursor.hasNext()) {
            names.add(((Document)cursor.next()).getString((Object)TABLE_NAME_KEY));
        }
        return names;
    }

    private void createTableMetadata(SchemaTableName schemaTableName, List<MongoColumnHandle> columns) throws TableNotFoundException {
        String schemaName = schemaTableName.getSchemaName();
        String tableName = schemaTableName.getTableName();
        MongoDatabase db = this.client.getDatabase(schemaName);
        Document metadata = new Document(TABLE_NAME_KEY, (Object)tableName);
        ArrayList<Document> fields = new ArrayList<Document>();
        if (!columns.stream().anyMatch(c -> c.getName().equals("_id"))) {
            fields.add(new MongoColumnHandle("_id", (Type)ObjectIdType.OBJECT_ID, true).getDocument());
        }
        fields.addAll(columns.stream().map(MongoColumnHandle::getDocument).collect(Collectors.toList()));
        metadata.append(FIELDS_KEY, fields);
        MongoCollection schema = db.getCollection(this.schemaCollection);
        schema.createIndex((Bson)new Document(TABLE_NAME_KEY, (Object)1), new IndexOptions().unique(true));
        schema.insertOne((Object)metadata);
    }

    private boolean deleteTableMetadata(SchemaTableName schemaTableName) {
        String schemaName = schemaTableName.getSchemaName();
        String tableName = schemaTableName.getTableName();
        MongoDatabase db = this.client.getDatabase(schemaName);
        if (!this.collectionExists(db, tableName)) {
            return false;
        }
        DeleteResult result = db.getCollection(this.schemaCollection).deleteOne((Bson)new Document(TABLE_NAME_KEY, (Object)tableName));
        return result.getDeletedCount() == 1L;
    }

    private List<Document> guessTableFields(SchemaTableName schemaTableName) {
        String schemaName = schemaTableName.getSchemaName();
        String tableName = schemaTableName.getTableName();
        MongoDatabase db = this.client.getDatabase(schemaName);
        Document doc = (Document)db.getCollection(tableName).find().first();
        if (doc == null) {
            return ImmutableList.of();
        }
        ImmutableList.Builder builder = ImmutableList.builder();
        for (String key : doc.keySet()) {
            Object value = doc.get((Object)key);
            Optional<TypeSignature> fieldType = this.guessFieldType(value);
            if (fieldType.isPresent()) {
                Document metadata = new Document();
                metadata.append(FIELDS_NAME_KEY, (Object)key);
                metadata.append(FIELDS_TYPE_KEY, (Object)fieldType.get().toString());
                metadata.append(FIELDS_HIDDEN_KEY, (Object)(key.equals("_id") && fieldType.get().equals((Object)ObjectIdType.OBJECT_ID.getTypeSignature()) ? 1 : 0));
                builder.add((Object)metadata);
                continue;
            }
            log.debug("Unable to guess field type from %s : %s", new Object[]{value == null ? "null" : value.getClass().getName(), value});
        }
        return builder.build();
    }

    private Optional<TypeSignature> guessFieldType(Object value) {
        if (value == null) {
            return Optional.empty();
        }
        TypeSignature typeSignature = null;
        if (value instanceof String) {
            typeSignature = VarcharType.createUnboundedVarcharType().getTypeSignature();
        } else if (value instanceof Integer || value instanceof Long) {
            typeSignature = BigintType.BIGINT.getTypeSignature();
        } else if (value instanceof Boolean) {
            typeSignature = BooleanType.BOOLEAN.getTypeSignature();
        } else if (value instanceof Float || value instanceof Double) {
            typeSignature = DoubleType.DOUBLE.getTypeSignature();
        } else if (value instanceof Date) {
            typeSignature = TimestampType.TIMESTAMP.getTypeSignature();
        } else if (value instanceof ObjectId) {
            typeSignature = ObjectIdType.OBJECT_ID.getTypeSignature();
        } else if (value instanceof List) {
            List subTypes = ((List)value).stream().map(this::guessFieldType).collect(Collectors.toList());
            if (subTypes.isEmpty() || subTypes.stream().anyMatch(t -> !t.isPresent())) {
                return Optional.empty();
            }
            Set signatures = subTypes.stream().map(t -> (TypeSignature)t.get()).collect(Collectors.toSet());
            typeSignature = signatures.size() == 1 ? new TypeSignature("array", signatures.stream().map(s -> TypeSignatureParameter.of((TypeSignature)s)).collect(Collectors.toList())) : new TypeSignature("row", IntStream.range(0, subTypes.size()).mapToObj(idx -> TypeSignatureParameter.of((NamedTypeSignature)new NamedTypeSignature(String.format("%s%d", this.implicitPrefix, idx + 1), (TypeSignature)((Optional)subTypes.get(idx)).get()))).collect(Collectors.toList()));
        } else if (value instanceof Document) {
            ArrayList<TypeSignatureParameter> parameters = new ArrayList<TypeSignatureParameter>();
            for (String key : ((Document)value).keySet()) {
                Optional<TypeSignature> fieldType = this.guessFieldType(((Document)value).get((Object)key));
                if (!fieldType.isPresent()) {
                    return Optional.empty();
                }
                parameters.add(TypeSignatureParameter.of((NamedTypeSignature)new NamedTypeSignature(key, fieldType.get())));
            }
            typeSignature = new TypeSignature("row", parameters);
        }
        return Optional.ofNullable(typeSignature);
    }
}

