/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.execution;

import com.facebook.presto.OutputBuffers;
import com.facebook.presto.PagePartitionFunction;
import com.facebook.presto.execution.BufferInfo;
import com.facebook.presto.execution.BufferResult;
import com.facebook.presto.execution.PageSplitterUtil;
import com.facebook.presto.execution.SharedBufferInfo;
import com.facebook.presto.execution.StateMachine;
import com.facebook.presto.execution.TaskId;
import com.facebook.presto.spi.Page;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Sets;
import com.google.common.primitives.Ints;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import io.airlift.units.DataSize;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
public class SharedBuffer {
    private final long maxBufferedBytes;
    @GuardedBy(value="this")
    private OutputBuffers outputBuffers = OutputBuffers.INITIAL_EMPTY_OUTPUT_BUFFERS;
    @GuardedBy(value="this")
    private long bufferedBytes;
    @GuardedBy(value="this")
    private final LinkedList<Page> masterBuffer = new LinkedList();
    @GuardedBy(value="this")
    private final BlockingQueue<QueuedPage> queuedPages = new LinkedBlockingQueue<QueuedPage>();
    @GuardedBy(value="this")
    private final AtomicLong masterSequenceId = new AtomicLong();
    @GuardedBy(value="this")
    private final ConcurrentMap<TaskId, NamedBuffer> namedBuffers = new ConcurrentHashMap<TaskId, NamedBuffer>();
    @GuardedBy(value="this")
    private final Set<TaskId> abortedBuffers = new HashSet<TaskId>();
    private final StateMachine<BufferState> state;
    @GuardedBy(value="this")
    private final List<GetBufferResult> stateChangeListeners = new ArrayList<GetBufferResult>();
    private final AtomicLong pagesAdded = new AtomicLong();

    public SharedBuffer(TaskId taskId, Executor executor, DataSize maxBufferSize) {
        Preconditions.checkNotNull((Object)taskId, (Object)"taskId is null");
        Preconditions.checkNotNull((Object)executor, (Object)"executor is null");
        this.state = new StateMachine<BufferState>(taskId + "-buffer", executor, BufferState.OPEN);
        Preconditions.checkNotNull((Object)maxBufferSize, (Object)"maxBufferSize is null");
        Preconditions.checkArgument((maxBufferSize.toBytes() > 0L ? 1 : 0) != 0, (Object)"maxBufferSize must be at least 1");
        this.maxBufferedBytes = maxBufferSize.toBytes();
    }

    public void addStateChangeListener(StateMachine.StateChangeListener<BufferState> stateChangeListener) {
        this.state.addStateChangeListener(stateChangeListener);
    }

    public boolean isFinished() {
        return this.state.get() == BufferState.FINISHED;
    }

    public SharedBufferInfo getInfo() {
        Preconditions.checkState((!Thread.holdsLock(this) ? 1 : 0) != 0, (String)"Thread must NOT hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
        ImmutableList.Builder infos = ImmutableList.builder();
        for (NamedBuffer namedBuffer : this.namedBuffers.values()) {
            infos.add((Object)namedBuffer.getInfo());
        }
        return new SharedBufferInfo(this.state.get(), this.masterSequenceId.get(), this.pagesAdded.get(), (List<BufferInfo>)infos.build());
    }

    public synchronized void setOutputBuffers(OutputBuffers newOutputBuffers) {
        Preconditions.checkNotNull((Object)newOutputBuffers, (Object)"newOutputBuffers is null");
        if (this.state.get() == BufferState.FINISHED || this.outputBuffers.getVersion() >= newOutputBuffers.getVersion()) {
            return;
        }
        Sets.SetView missingBuffers = Sets.difference(this.outputBuffers.getBuffers().keySet(), newOutputBuffers.getBuffers().keySet());
        Preconditions.checkArgument((boolean)missingBuffers.isEmpty(), (String)"newOutputBuffers does not have existing buffers %s", (Object[])new Object[]{missingBuffers});
        Preconditions.checkArgument((!this.outputBuffers.isNoMoreBufferIds() || newOutputBuffers.isNoMoreBufferIds() ? 1 : 0) != 0, (Object)"Expected newOutputBuffers to have noMoreBufferIds set");
        this.outputBuffers = newOutputBuffers;
        for (Map.Entry<TaskId, PagePartitionFunction> entry : this.outputBuffers.getBuffers().entrySet()) {
            TaskId bufferId = entry.getKey();
            if (this.namedBuffers.containsKey(bufferId)) continue;
            Preconditions.checkState((boolean)this.state.get().canAddBuffers(), (String)"Cannot add buffers to %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
            NamedBuffer namedBuffer = new NamedBuffer(bufferId, entry.getValue());
            if (this.abortedBuffers.contains(bufferId)) {
                namedBuffer.abort();
            }
            this.namedBuffers.put(bufferId, namedBuffer);
        }
        if (this.outputBuffers.isNoMoreBufferIds()) {
            this.state.compareAndSet(BufferState.OPEN, BufferState.NO_MORE_BUFFERS);
            this.state.compareAndSet(BufferState.NO_MORE_PAGES, BufferState.FLUSHING);
        }
        this.updateState();
    }

    public synchronized ListenableFuture<?> enqueue(Page page) {
        Preconditions.checkNotNull((Object)page, (Object)"page is null");
        if (!this.state.get().canAddPages()) {
            return Futures.immediateFuture((Object)true);
        }
        if (this.bufferedBytes < this.maxBufferedBytes) {
            this.addInternal(page);
            return Futures.immediateFuture((Object)true);
        }
        QueuedPage queuedPage = new QueuedPage(page);
        this.queuedPages.add(queuedPage);
        this.updateState();
        return queuedPage.getFuture();
    }

    private synchronized void addInternal(Page page) {
        List<Page> pages = PageSplitterUtil.splitPage(page, 0x100000L);
        this.masterBuffer.addAll(pages);
        this.pagesAdded.addAndGet(pages.size());
        for (Page p : pages) {
            this.bufferedBytes += p.getSizeInBytes();
        }
        this.processPendingReads();
    }

    public synchronized ListenableFuture<BufferResult> get(TaskId outputId, long startingSequenceId, DataSize maxSize) {
        Preconditions.checkNotNull((Object)outputId, (Object)"outputId is null");
        Preconditions.checkArgument((maxSize.toBytes() > 0L ? 1 : 0) != 0, (Object)"maxSize must be at least 1 byte");
        if (!this.state.get().canAddBuffers() && this.namedBuffers.get(outputId) == null) {
            return Futures.immediateFuture((Object)BufferResult.emptyResults(0L, true));
        }
        GetBufferResult getBufferResult = new GetBufferResult(outputId, startingSequenceId, maxSize);
        this.stateChangeListeners.add(getBufferResult);
        this.updateState();
        return getBufferResult.getFuture();
    }

    private synchronized List<Page> getPagesInternal(DataSize maxSize, long sequenceId) {
        long maxBytes = maxSize.toBytes();
        ArrayList<Page> pages = new ArrayList<Page>();
        long bytes = 0L;
        int listOffset = Ints.checkedCast((long)(sequenceId - this.masterSequenceId.get()));
        while (listOffset < this.masterBuffer.size()) {
            Page page = this.masterBuffer.get(listOffset++);
            if (!pages.isEmpty() && (bytes += page.getSizeInBytes()) > maxBytes) break;
            pages.add(page);
        }
        return ImmutableList.copyOf(pages);
    }

    public synchronized void abort(TaskId outputId) {
        Preconditions.checkNotNull((Object)outputId, (Object)"outputId is null");
        this.abortedBuffers.add(outputId);
        NamedBuffer namedBuffer = (NamedBuffer)this.namedBuffers.get(outputId);
        if (namedBuffer != null) {
            namedBuffer.abort();
        }
        this.updateState();
    }

    public synchronized void setNoMorePages() {
        if (this.state.compareAndSet(BufferState.OPEN, BufferState.NO_MORE_PAGES) || this.state.compareAndSet(BufferState.NO_MORE_BUFFERS, BufferState.FLUSHING)) {
            this.updateState();
        }
    }

    public synchronized void destroy() {
        this.state.set(BufferState.FINISHED);
        this.masterBuffer.clear();
        this.bufferedBytes = 0L;
        for (QueuedPage queuedPage : this.queuedPages) {
            queuedPage.getFuture().set(null);
        }
        this.queuedPages.clear();
        for (NamedBuffer namedBuffer : this.namedBuffers.values()) {
            namedBuffer.abort();
        }
        this.processPendingReads();
    }

    private void checkFlushComplete() {
        Preconditions.checkState((boolean)Thread.holdsLock(this), (String)"Thread must hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
        if (this.state.get() == BufferState.FLUSHING) {
            for (NamedBuffer namedBuffer : this.namedBuffers.values()) {
                if (namedBuffer.checkCompletion()) continue;
                return;
            }
            this.destroy();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateState() {
        Preconditions.checkState((boolean)Thread.holdsLock(this), (String)"Thread must hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
        try {
            this.processPendingReads();
            BufferState state = this.state.get();
            if (state == BufferState.FINISHED) {
                return;
            }
            if (!state.canAddPages()) {
                for (QueuedPage queuedPage : this.queuedPages) {
                    queuedPage.getFuture().set(null);
                }
                this.queuedPages.clear();
            }
            if (!state.canAddBuffers() && !this.namedBuffers.isEmpty()) {
                long oldMasterSequenceId = this.masterSequenceId.get();
                long newMasterSequenceId = Long.MAX_VALUE;
                for (NamedBuffer namedBuffer : this.namedBuffers.values()) {
                    newMasterSequenceId = Math.min(namedBuffer.getSequenceId(), newMasterSequenceId);
                }
                this.masterSequenceId.set(newMasterSequenceId);
                int pagesToRemove = Ints.checkedCast((long)(newMasterSequenceId - oldMasterSequenceId));
                Preconditions.checkState((pagesToRemove >= 0 ? 1 : 0) != 0, (String)"Master sequence id moved backwards: oldMasterSequenceId=%s, newMasterSequenceId=%s", (Object[])new Object[]{oldMasterSequenceId, newMasterSequenceId});
                for (int i = 0; i < pagesToRemove; ++i) {
                    Page page = this.masterBuffer.removeFirst();
                    this.bufferedBytes -= page.getSizeInBytes();
                }
                while (!this.queuedPages.isEmpty() && this.bufferedBytes < this.maxBufferedBytes) {
                    QueuedPage queuedPage = (QueuedPage)this.queuedPages.remove();
                    this.addInternal(queuedPage.getPage());
                    queuedPage.getFuture().set(null);
                }
            }
            if (!state.canAddPages()) {
                for (NamedBuffer namedBuffer : this.namedBuffers.values()) {
                    namedBuffer.checkCompletion();
                }
            }
        }
        finally {
            this.checkFlushComplete();
        }
    }

    private void processPendingReads() {
        Preconditions.checkState((boolean)Thread.holdsLock(this), (String)"Thread must hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
        for (GetBufferResult getBufferResult : ImmutableList.copyOf(this.stateChangeListeners)) {
            if (!getBufferResult.execute()) continue;
            this.stateChangeListeners.remove(getBufferResult);
        }
    }

    @Immutable
    private class GetBufferResult {
        private final SettableFuture<BufferResult> future = SettableFuture.create();
        private final TaskId outputId;
        private final long startingSequenceId;
        private final DataSize maxSize;

        public GetBufferResult(TaskId outputId, long startingSequenceId, DataSize maxSize) {
            this.outputId = outputId;
            this.startingSequenceId = startingSequenceId;
            this.maxSize = maxSize;
        }

        public SettableFuture<BufferResult> getFuture() {
            return this.future;
        }

        public boolean execute() {
            Preconditions.checkState((boolean)Thread.holdsLock(SharedBuffer.this), (String)"Thread must hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
            if (this.future.isDone()) {
                return true;
            }
            try {
                NamedBuffer namedBuffer = (NamedBuffer)SharedBuffer.this.namedBuffers.get(this.outputId);
                if (SharedBuffer.this.state.get() == BufferState.FINISHED) {
                    this.future.set((Object)BufferResult.emptyResults(namedBuffer == null ? 0L : namedBuffer.getSequenceId(), true));
                    return true;
                }
                if (namedBuffer == null) {
                    return false;
                }
                if (this.startingSequenceId < namedBuffer.getSequenceId()) {
                    this.future.set((Object)BufferResult.emptyResults(this.startingSequenceId, false));
                    return true;
                }
                BufferResult bufferResult = namedBuffer.getPages(this.startingSequenceId, this.maxSize);
                SharedBuffer.this.checkFlushComplete();
                if (bufferResult.isEmpty() && !bufferResult.isBufferClosed()) {
                    return false;
                }
                this.future.set((Object)bufferResult);
            }
            catch (Throwable throwable) {
                this.future.setException(throwable);
            }
            return true;
        }
    }

    @Immutable
    private static final class QueuedPage {
        private final Page page;
        private final SettableFuture<?> future = SettableFuture.create();

        private QueuedPage(Page page) {
            this.page = page;
        }

        private Page getPage() {
            return this.page;
        }

        private SettableFuture<?> getFuture() {
            return this.future;
        }
    }

    @ThreadSafe
    private final class NamedBuffer {
        private final TaskId bufferId;
        private final PagePartitionFunction partitionFunction;
        private final AtomicLong sequenceId = new AtomicLong();
        private final AtomicBoolean finished = new AtomicBoolean();

        private NamedBuffer(TaskId bufferId, PagePartitionFunction partitionFunction) {
            this.bufferId = bufferId;
            this.partitionFunction = partitionFunction;
        }

        public BufferInfo getInfo() {
            Preconditions.checkState((!Thread.holdsLock(this) ? 1 : 0) != 0, (String)"Thread must NOT hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
            long sequenceId = this.sequenceId.get();
            if (this.finished.get()) {
                return new BufferInfo(this.bufferId, true, 0, sequenceId);
            }
            int size = Math.max(Ints.checkedCast((long)(SharedBuffer.this.pagesAdded.get() + (long)SharedBuffer.this.queuedPages.size() - sequenceId)), 0);
            return new BufferInfo(this.bufferId, this.finished.get(), size, sequenceId);
        }

        public long getSequenceId() {
            Preconditions.checkState((boolean)Thread.holdsLock(SharedBuffer.this), (String)"Thread must hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
            return this.sequenceId.get();
        }

        public BufferResult getPages(long startingSequenceId, DataSize maxSize) {
            Preconditions.checkState((boolean)Thread.holdsLock(SharedBuffer.this), (String)"Thread must hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
            Preconditions.checkArgument((maxSize.toBytes() > 0L ? 1 : 0) != 0, (Object)"maxSize must be at least 1 byte");
            long sequenceId = this.sequenceId.get();
            Preconditions.checkArgument((startingSequenceId >= sequenceId ? 1 : 0) != 0, (Object)"startingSequenceId is before the beginning of the buffer");
            if (startingSequenceId > sequenceId) {
                this.sequenceId.set(startingSequenceId);
                sequenceId = startingSequenceId;
            }
            if (this.checkCompletion()) {
                return BufferResult.emptyResults(startingSequenceId, true);
            }
            List pages = SharedBuffer.this.getPagesInternal(maxSize, sequenceId);
            return new BufferResult(startingSequenceId, startingSequenceId + (long)pages.size(), false, pages, this.partitionFunction);
        }

        public void abort() {
            Preconditions.checkState((boolean)Thread.holdsLock(SharedBuffer.this), (String)"Thread must hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
            this.finished.set(true);
        }

        public boolean checkCompletion() {
            Preconditions.checkState((boolean)Thread.holdsLock(SharedBuffer.this), (String)"Thread must hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
            if (this.finished.get()) {
                return true;
            }
            if (!((BufferState)((Object)SharedBuffer.this.state.get())).canAddPages() && this.sequenceId.get() >= SharedBuffer.this.pagesAdded.get()) {
                this.finished.set(true);
                SharedBuffer.this.checkFlushComplete();
            }
            return this.finished.get();
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("bufferId", (Object)this.bufferId).add("sequenceId", this.sequenceId.get()).add("finished", this.finished.get()).toString();
        }
    }

    public static enum BufferState {
        OPEN(true, true),
        NO_MORE_BUFFERS(true, false),
        NO_MORE_PAGES(false, true),
        FLUSHING(false, false),
        FINISHED(false, false);

        private final boolean newPagesAllowed;
        private final boolean newBuffersAllowed;

        private BufferState(boolean newPagesAllowed, boolean newBuffersAllowed) {
            this.newPagesAllowed = newPagesAllowed;
            this.newBuffersAllowed = newBuffersAllowed;
        }

        public boolean canAddPages() {
            return this.newPagesAllowed;
        }

        public boolean canAddBuffers() {
            return this.newBuffersAllowed;
        }
    }
}

