/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.operator.scalar;

import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.util.Failures;
import com.facebook.presto.util.ThreadLocalCache;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.core.SerializableString;
import com.fasterxml.jackson.core.io.SerializedString;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;

public final class JsonExtract {
    private static final Pattern EXPECTED_PATH = Pattern.compile("\\$(\\[\\d+\\])*(\\.[^@\\.\\[\\]\\$\\*]+(\\[\\d+\\])*)*");
    private static final int ESTIMATED_JSON_OUTPUT_SIZE = 512;
    private static final List<StringReplacer> PATH_STRING_REPLACERS = ImmutableList.of((Object)new StringReplacer("[", ".["), (Object)new StringReplacer("]", ""));
    private static final Splitter DOT_SPLITTER = Splitter.on((String)".").trimResults();
    private static final JsonFactory JSON_FACTORY = new JsonFactory().disable(JsonFactory.Feature.CANONICALIZE_FIELD_NAMES);
    private static final JsonExtractCache SCALAR_CACHE = new JsonExtractCache(20, true);
    private static final JsonExtractCache JSON_CACHE = new JsonExtractCache(20, false);

    private JsonExtract() {
    }

    public static Slice extractScalar(@Nullable Slice jsonInput, Slice jsonPath) throws IOException {
        return JsonExtract.extract(SCALAR_CACHE, jsonInput, jsonPath);
    }

    public static Slice extractJson(@Nullable Slice jsonInput, Slice jsonPath) throws IOException {
        return JsonExtract.extract(JSON_CACHE, jsonInput, jsonPath);
    }

    public static Slice extract(ThreadLocalCache<Slice, JsonExtractor> cache, @Nullable Slice jsonInput, Slice jsonPath) throws IOException {
        Preconditions.checkNotNull((Object)jsonPath, (Object)"jsonPath is null");
        if (jsonInput == null) {
            return null;
        }
        try {
            return JsonExtract.extractInternal(jsonInput, cache.get(jsonPath));
        }
        catch (JsonParseException e) {
            return null;
        }
    }

    public static Slice extract(Slice jsonInput, JsonExtractor jsonExtractor) throws IOException {
        try {
            return JsonExtract.extractInternal(jsonInput, jsonExtractor);
        }
        catch (JsonParseException e) {
            return null;
        }
    }

    @VisibleForTesting
    static Slice extractInternal(Slice jsonInput, JsonExtractor jsonExtractor) throws IOException {
        Preconditions.checkNotNull((Object)jsonInput, (Object)"jsonInput is null");
        try (JsonParser jsonParser = JSON_FACTORY.createJsonParser((InputStream)jsonInput.getInput());){
            if (jsonParser.nextToken() == null) {
                throw new JsonParseException("Missing starting token", jsonParser.getCurrentLocation());
            }
            Slice slice = jsonExtractor.extract(jsonParser);
            return slice;
        }
    }

    private static Iterable<String> tokenizePath(String path) {
        Failures.checkCondition(EXPECTED_PATH.matcher(path).matches(), StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid/unsupported JSON path: '%s'", path);
        for (StringReplacer replacer : PATH_STRING_REPLACERS) {
            path = replacer.replace(path);
        }
        return DOT_SPLITTER.split((CharSequence)path);
    }

    public static JsonExtractor generateExtractor(String path, boolean scalarValue) {
        Iterator<String> iterator = JsonExtract.tokenizePath(path).iterator();
        Failures.checkCondition(iterator.hasNext() && iterator.next().equals("$"), StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "JSON path must begin with root: '$'", new Object[0]);
        return JsonExtract.generateExtractor(iterator, scalarValue);
    }

    private static JsonExtractor generateExtractor(Iterator<String> filters, boolean scalarValue) {
        if (!filters.hasNext()) {
            return scalarValue ? new ScalarValueJsonExtractor() : new JsonValueJsonExtractor();
        }
        String filter = filters.next();
        if (filter.startsWith("[")) {
            int index = Integer.parseInt(filter.substring(1).trim());
            return new ArrayElementJsonExtractor(index, JsonExtract.generateExtractor(filters, scalarValue));
        }
        return new ObjectFieldJsonExtractor(filter, JsonExtract.generateExtractor(filters, scalarValue));
    }

    public static class JsonExtractCache
    extends ThreadLocalCache<Slice, JsonExtractor> {
        private final boolean isScalarValue;

        public JsonExtractCache(int sizePerThread, boolean isScalarValue) {
            super(sizePerThread);
            this.isScalarValue = isScalarValue;
        }

        @Override
        protected JsonExtractor load(Slice jsonPath) {
            return JsonExtract.generateExtractor(jsonPath.toString(Charsets.UTF_8), this.isScalarValue);
        }
    }

    private static class StringReplacer {
        private final Pattern pattern;
        private final String replacement;

        private StringReplacer(String original, String replacement) {
            this.pattern = Pattern.compile(original, 16);
            this.replacement = Matcher.quoteReplacement(replacement);
        }

        public String replace(String target) {
            return this.pattern.matcher(target).replaceAll(this.replacement);
        }
    }

    public static class JsonValueJsonExtractor
    implements JsonExtractor {
        @Override
        public Slice extract(JsonParser jsonParser) throws IOException {
            if (!jsonParser.hasCurrentToken()) {
                throw new JsonParseException("Unexpected end of value", jsonParser.getCurrentLocation());
            }
            DynamicSliceOutput dynamicSliceOutput = new DynamicSliceOutput(512);
            try (JsonGenerator jsonGenerator = JSON_FACTORY.createJsonGenerator((OutputStream)dynamicSliceOutput);){
                jsonGenerator.copyCurrentStructure(jsonParser);
            }
            return dynamicSliceOutput.slice();
        }
    }

    public static class ScalarValueJsonExtractor
    implements JsonExtractor {
        @Override
        public Slice extract(JsonParser jsonParser) throws IOException {
            JsonToken token = jsonParser.getCurrentToken();
            if (token == null) {
                throw new JsonParseException("Unexpected end of value", jsonParser.getCurrentLocation());
            }
            if (!token.isScalarValue() || token == JsonToken.VALUE_NULL) {
                return null;
            }
            return Slices.wrappedBuffer((byte[])jsonParser.getText().getBytes(Charsets.UTF_8));
        }
    }

    public static class ArrayElementJsonExtractor
    implements JsonExtractor {
        private final int index;
        private final JsonExtractor delegate;

        public ArrayElementJsonExtractor(int index, JsonExtractor delegate) {
            Preconditions.checkArgument((index >= 0 ? 1 : 0) != 0, (String)"index must be greater than or equal to zero: %s", (Object[])new Object[]{index});
            Preconditions.checkNotNull((Object)delegate, (Object)"delegate is null");
            this.index = index;
            this.delegate = delegate;
        }

        @Override
        public Slice extract(JsonParser jsonParser) throws IOException {
            if (jsonParser.getCurrentToken() != JsonToken.START_ARRAY) {
                throw new JsonParseException("Expected a Json array", jsonParser.getCurrentLocation());
            }
            int currentIndex = 0;
            while (true) {
                JsonToken token;
                if ((token = jsonParser.nextToken()) == null) {
                    throw new JsonParseException("Unexpected end of array", jsonParser.getCurrentLocation());
                }
                if (token == JsonToken.END_ARRAY) {
                    return null;
                }
                if (currentIndex == this.index) break;
                ++currentIndex;
                jsonParser.skipChildren();
            }
            return this.delegate.extract(jsonParser);
        }
    }

    public static class ObjectFieldJsonExtractor
    implements JsonExtractor {
        private final SerializedString fieldName;
        private final JsonExtractor delegate;

        public ObjectFieldJsonExtractor(String fieldName, JsonExtractor delegate) {
            this.fieldName = new SerializedString((String)Preconditions.checkNotNull((Object)fieldName, (Object)"fieldName is null"));
            this.delegate = (JsonExtractor)Preconditions.checkNotNull((Object)delegate, (Object)"delegate is null");
        }

        @Override
        public Slice extract(JsonParser jsonParser) throws IOException {
            if (jsonParser.getCurrentToken() != JsonToken.START_OBJECT) {
                throw new JsonParseException("Expected a Json object", jsonParser.getCurrentLocation());
            }
            while (!jsonParser.nextFieldName((SerializableString)this.fieldName)) {
                if (!jsonParser.hasCurrentToken()) {
                    throw new JsonParseException("Unexpected end of object", jsonParser.getCurrentLocation());
                }
                if (jsonParser.getCurrentToken() == JsonToken.END_OBJECT) {
                    return null;
                }
                jsonParser.skipChildren();
            }
            jsonParser.nextToken();
            return this.delegate.extract(jsonParser);
        }
    }

    public static interface JsonExtractor {
        public Slice extract(JsonParser var1) throws IOException;
    }
}

