/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.failureDetector;

import com.facebook.presto.failureDetector.FailureDetector;
import com.facebook.presto.failureDetector.FailureDetectorConfig;
import com.facebook.presto.failureDetector.ForFailureDetector;
import com.facebook.presto.util.IterableTransformer;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableMap;
import io.airlift.concurrent.Threads;
import io.airlift.discovery.client.ServiceDescriptor;
import io.airlift.discovery.client.ServiceSelector;
import io.airlift.discovery.client.ServiceType;
import io.airlift.http.client.AsyncHttpClient;
import io.airlift.http.client.Request;
import io.airlift.http.client.Response;
import io.airlift.http.client.ResponseHandler;
import io.airlift.log.Logger;
import io.airlift.node.NodeInfo;
import io.airlift.stats.DecayCounter;
import io.airlift.stats.ExponentialDecay;
import io.airlift.units.Duration;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import javax.inject.Inject;
import org.joda.time.DateTime;
import org.weakref.jmx.Managed;

public class HeartbeatFailureDetector
implements FailureDetector {
    private static final Logger log = Logger.get(HeartbeatFailureDetector.class);
    private final ServiceSelector selector;
    private final AsyncHttpClient httpClient;
    private final NodeInfo nodeInfo;
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(Threads.daemonThreadsNamed((String)"failure-detector"));
    private final ConcurrentMap<UUID, MonitoringTask> tasks = new ConcurrentHashMap<UUID, MonitoringTask>();
    private final double failureRatioThreshold;
    private final Duration heartbeat;
    private final boolean isEnabled;
    private final Duration warmupInterval;
    private final Duration gcGraceInterval;
    private final AtomicBoolean started = new AtomicBoolean();

    @Inject
    public HeartbeatFailureDetector(@ServiceType(value="presto") ServiceSelector selector, @ForFailureDetector AsyncHttpClient httpClient, FailureDetectorConfig config, NodeInfo nodeInfo) {
        Preconditions.checkNotNull((Object)selector, (Object)"selector is null");
        Preconditions.checkNotNull((Object)httpClient, (Object)"httpClient is null");
        Preconditions.checkNotNull((Object)nodeInfo, (Object)"nodeInfo is null");
        Preconditions.checkNotNull((Object)config, (Object)"config is null");
        Preconditions.checkArgument((config.getHeartbeatInterval().toMillis() >= 1L ? 1 : 0) != 0, (Object)"heartbeat interval must be >= 1ms");
        this.selector = selector;
        this.httpClient = httpClient;
        this.nodeInfo = nodeInfo;
        this.failureRatioThreshold = config.getFailureRatioThreshold();
        this.heartbeat = config.getHeartbeatInterval();
        this.warmupInterval = config.getWarmupInterval();
        this.gcGraceInterval = config.getExpirationGraceInterval();
        this.isEnabled = config.isEnabled();
    }

    @PostConstruct
    public void start() {
        if (this.isEnabled && this.started.compareAndSet(false, true)) {
            this.executor.scheduleWithFixedDelay(new Runnable(){

                @Override
                public void run() {
                    try {
                        HeartbeatFailureDetector.this.updateMonitoredServices();
                    }
                    catch (Throwable e) {
                        log.warn(e, "Error updating services");
                    }
                }
            }, 0L, 5L, TimeUnit.SECONDS);
        }
    }

    @PreDestroy
    public void shutdown() {
        this.executor.shutdownNow();
    }

    @Override
    public Set<ServiceDescriptor> getFailed() {
        return IterableTransformer.on(this.tasks.values()).select(HeartbeatFailureDetector.isFailedPredicate()).transform(HeartbeatFailureDetector.serviceGetter()).set();
    }

    @Managed(description="Number of failed services")
    public int getFailedCount() {
        return this.getFailed().size();
    }

    @Managed(description="Total number of known services")
    public int getTotalCount() {
        return this.tasks.size();
    }

    @Managed
    public int getActiveCount() {
        return this.tasks.size() - this.getFailed().size();
    }

    public Map<ServiceDescriptor, Stats> getStats() {
        ImmutableMap.Builder builder = ImmutableMap.builder();
        for (MonitoringTask task : this.tasks.values()) {
            builder.put((Object)task.getService(), (Object)task.getStats());
        }
        return builder.build();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    void updateMonitoredServices() {
        Set online = IterableTransformer.on(this.selector.selectAllServices()).select(Predicates.not(HeartbeatFailureDetector.serviceDescriptorHasNodeId(this.nodeInfo.getNodeId()))).set();
        Set<UUID> onlineIds = IterableTransformer.on(online).transform(HeartbeatFailureDetector.idGetter()).set();
        ConcurrentMap<UUID, MonitoringTask> concurrentMap = this.tasks;
        synchronized (concurrentMap) {
            List<UUID> expiredIds = IterableTransformer.on(this.tasks.values()).select(HeartbeatFailureDetector.isExpiredPredicate()).transform(HeartbeatFailureDetector.serviceIdGetter()).list();
            this.tasks.keySet().removeAll(expiredIds);
            Iterable toDisable = IterableTransformer.on(this.tasks.values()).select(Predicates.compose((Predicate)Predicates.not((Predicate)Predicates.in(onlineIds)), HeartbeatFailureDetector.serviceIdGetter())).all();
            for (MonitoringTask task : toDisable) {
                task.disable();
            }
            Set newServices = IterableTransformer.on(online).select(Predicates.compose((Predicate)Predicates.not((Predicate)Predicates.in(this.tasks.keySet())), HeartbeatFailureDetector.idGetter())).set();
            for (ServiceDescriptor service : newServices) {
                URI uri = HeartbeatFailureDetector.getHttpUri(service);
                if (uri == null) continue;
                this.tasks.put(service.getId(), new MonitoringTask(service, uri));
            }
            Iterable toEnable = IterableTransformer.on(this.tasks.values()).select(Predicates.compose((Predicate)Predicates.in(onlineIds), HeartbeatFailureDetector.serviceIdGetter())).all();
            for (MonitoringTask task : toEnable) {
                task.enable();
            }
        }
    }

    private static URI getHttpUri(ServiceDescriptor service) {
        try {
            String uri = (String)service.getProperties().get("http");
            if (uri != null) {
                return new URI(uri);
            }
        }
        catch (URISyntaxException uRISyntaxException) {
            // empty catch block
        }
        return null;
    }

    private static Predicate<ServiceDescriptor> serviceDescriptorHasNodeId(final String nodeId) {
        Preconditions.checkNotNull((Object)nodeId, (Object)"nodeId is null");
        return new Predicate<ServiceDescriptor>(){

            public boolean apply(ServiceDescriptor descriptor) {
                return nodeId.equals(descriptor.getNodeId());
            }
        };
    }

    private static Function<ServiceDescriptor, UUID> idGetter() {
        return new Function<ServiceDescriptor, UUID>(){

            public UUID apply(ServiceDescriptor descriptor) {
                return descriptor.getId();
            }
        };
    }

    private static Function<MonitoringTask, ServiceDescriptor> serviceGetter() {
        return new Function<MonitoringTask, ServiceDescriptor>(){

            public ServiceDescriptor apply(MonitoringTask task) {
                return task.getService();
            }
        };
    }

    private static Function<MonitoringTask, UUID> serviceIdGetter() {
        return new Function<MonitoringTask, UUID>(){

            public UUID apply(MonitoringTask task) {
                return task.getService().getId();
            }
        };
    }

    private static Predicate<MonitoringTask> isExpiredPredicate() {
        return new Predicate<MonitoringTask>(){

            public boolean apply(MonitoringTask task) {
                return task.isExpired();
            }
        };
    }

    private static Predicate<MonitoringTask> isFailedPredicate() {
        return new Predicate<MonitoringTask>(){

            public boolean apply(MonitoringTask task) {
                return task.isFailed();
            }
        };
    }

    public static class Stats {
        private final long start = System.nanoTime();
        private final URI uri;
        private final DecayCounter recentRequests = new DecayCounter(ExponentialDecay.oneMinute());
        private final DecayCounter recentFailures = new DecayCounter(ExponentialDecay.oneMinute());
        private final DecayCounter recentSuccesses = new DecayCounter(ExponentialDecay.oneMinute());
        private final AtomicReference<DateTime> lastRequestTime = new AtomicReference();
        private final AtomicReference<DateTime> lastResponseTime = new AtomicReference();
        @GuardedBy(value="this")
        private final Map<Class<? extends Throwable>, DecayCounter> failureCountByType = new HashMap<Class<? extends Throwable>, DecayCounter>();

        public Stats(URI uri) {
            this.uri = uri;
        }

        public void recordStart() {
            this.recentRequests.add(1L);
            this.lastRequestTime.set(new DateTime());
        }

        public void recordSuccess() {
            this.recentSuccesses.add(1L);
            this.lastResponseTime.set(new DateTime());
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void recordFailure(Exception exception) {
            this.recentFailures.add(1L);
            this.lastResponseTime.set(new DateTime());
            Throwable cause = exception;
            while (cause.getClass() == RuntimeException.class && cause.getCause() != null) {
                cause = cause.getCause();
            }
            Stats stats = this;
            synchronized (stats) {
                DecayCounter counter = this.failureCountByType.get(cause.getClass());
                if (counter == null) {
                    counter = new DecayCounter(ExponentialDecay.oneMinute());
                    this.failureCountByType.put(cause.getClass(), counter);
                }
                counter.add(1L);
            }
        }

        @JsonProperty
        public Duration getAge() {
            return Duration.nanosSince((long)this.start);
        }

        @JsonProperty
        public URI getUri() {
            return this.uri;
        }

        @JsonProperty
        public double getRecentFailures() {
            return this.recentFailures.getCount();
        }

        @JsonProperty
        public double getRecentSuccesses() {
            return this.recentSuccesses.getCount();
        }

        @JsonProperty
        public double getRecentRequests() {
            return this.recentRequests.getCount();
        }

        @JsonProperty
        public double getRecentFailureRatio() {
            return this.recentFailures.getCount() / this.recentRequests.getCount();
        }

        @JsonProperty
        public DateTime getLastRequestTime() {
            return this.lastRequestTime.get();
        }

        @JsonProperty
        public DateTime getLastResponseTime() {
            return this.lastResponseTime.get();
        }

        @JsonProperty
        public synchronized Map<String, Double> getRecentFailuresByType() {
            ImmutableMap.Builder builder = ImmutableMap.builder();
            for (Map.Entry<Class<? extends Throwable>, DecayCounter> entry : this.failureCountByType.entrySet()) {
                builder.put((Object)entry.getKey().getName(), (Object)entry.getValue().getCount());
            }
            return builder.build();
        }
    }

    @ThreadSafe
    private class MonitoringTask {
        private final ServiceDescriptor service;
        private final URI uri;
        private final Stats stats;
        @GuardedBy(value="this")
        private ScheduledFuture<?> future;
        @GuardedBy(value="this")
        private Long disabledTimestamp;
        @GuardedBy(value="this")
        private Long successTransitionTimestamp;

        private MonitoringTask(ServiceDescriptor service, URI uri) {
            this.uri = uri;
            this.service = service;
            this.stats = new Stats(uri);
        }

        public Stats getStats() {
            return this.stats;
        }

        public ServiceDescriptor getService() {
            return this.service;
        }

        public synchronized void enable() {
            if (this.future == null) {
                this.future = HeartbeatFailureDetector.this.executor.scheduleAtFixedRate(new Runnable(){

                    @Override
                    public void run() {
                        try {
                            MonitoringTask.this.ping();
                            MonitoringTask.this.updateState();
                        }
                        catch (Throwable e) {
                            log.warn(e, "Error pinging service %s (%s)", new Object[]{MonitoringTask.this.service.getId(), MonitoringTask.this.uri});
                        }
                    }
                }, HeartbeatFailureDetector.this.heartbeat.toMillis(), HeartbeatFailureDetector.this.heartbeat.toMillis(), TimeUnit.MILLISECONDS);
                this.disabledTimestamp = null;
            }
        }

        public synchronized void disable() {
            if (this.future != null) {
                this.future.cancel(true);
                this.future = null;
                this.disabledTimestamp = System.nanoTime();
            }
        }

        public synchronized boolean isExpired() {
            return this.future == null && this.disabledTimestamp != null && Duration.nanosSince((long)this.disabledTimestamp).compareTo(HeartbeatFailureDetector.this.gcGraceInterval) > 0;
        }

        public synchronized boolean isFailed() {
            return this.future == null || this.successTransitionTimestamp == null || Duration.nanosSince((long)this.successTransitionTimestamp).compareTo(HeartbeatFailureDetector.this.warmupInterval) < 0;
        }

        private void ping() {
            try {
                this.stats.recordStart();
                HeartbeatFailureDetector.this.httpClient.executeAsync(Request.Builder.prepareHead().setUri(this.uri).build(), (ResponseHandler)new ResponseHandler<Object, Exception>(){

                    public Exception handleException(Request request, Exception exception) {
                        MonitoringTask.this.stats.recordFailure(exception);
                        return null;
                    }

                    public Object handle(Request request, Response response) throws Exception {
                        MonitoringTask.this.stats.recordSuccess();
                        return null;
                    }
                });
            }
            catch (RuntimeException e) {
                log.warn((Throwable)e, "Error scheduling request for %s", new Object[]{this.uri});
            }
        }

        private synchronized void updateState() {
            if (this.stats.getRecentFailureRatio() > HeartbeatFailureDetector.this.failureRatioThreshold) {
                this.successTransitionTimestamp = null;
            } else if (this.successTransitionTimestamp == null) {
                this.successTransitionTimestamp = System.nanoTime();
            }
        }
    }
}

