/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.execution;

import com.facebook.presto.execution.BufferInfo;
import com.facebook.presto.execution.BufferResult;
import com.facebook.presto.execution.NoSuchBufferException;
import com.facebook.presto.execution.SharedBufferInfo;
import com.facebook.presto.operator.Page;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ComparisonChain;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.primitives.Ints;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import io.airlift.units.DataSize;
import io.airlift.units.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.NotThreadSafe;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
public class SharedBuffer {
    private final long maxBufferedBytes;
    @GuardedBy(value="this")
    private long bufferedBytes;
    @GuardedBy(value="this")
    private final LinkedList<Page> masterQueue = new LinkedList();
    @GuardedBy(value="this")
    private final LinkedList<QueuedPage> queuedPages = new LinkedList();
    @GuardedBy(value="this")
    private long masterSequenceId;
    @GuardedBy(value="this")
    private Map<String, NamedQueue> namedQueues = new HashMap<String, NamedQueue>();
    @GuardedBy(value="this")
    private final SortedSet<NamedQueue> openQueuesBySequenceId = new TreeSet<NamedQueue>();
    @GuardedBy(value="this")
    private QueueState state = QueueState.OPEN;
    private final AtomicLong pagesAdded = new AtomicLong();
    private final AtomicBoolean closed = new AtomicBoolean();

    public SharedBuffer(DataSize maxBufferSize) {
        Preconditions.checkArgument((maxBufferSize.toBytes() > 0L ? 1 : 0) != 0, (Object)"maxBufferSize must be at least 1");
        this.maxBufferedBytes = maxBufferSize.toBytes();
    }

    public synchronized boolean isFinished() {
        return this.state == QueueState.FINISHED;
    }

    public synchronized SharedBufferInfo getInfo() {
        ImmutableList.Builder infos = ImmutableList.builder();
        for (NamedQueue namedQueue : this.namedQueues.values()) {
            infos.add((Object)new BufferInfo(namedQueue.getQueueId(), namedQueue.isFinished(), namedQueue.size(), namedQueue.pagesRemoved()));
        }
        return new SharedBufferInfo(this.state, this.masterSequenceId, this.pagesAdded.get(), (List<BufferInfo>)infos.build());
    }

    public synchronized void addQueue(String queueId) {
        Preconditions.checkNotNull((Object)queueId, (Object)"queueId is null");
        if (this.state == QueueState.FINISHED || this.namedQueues.containsKey(queueId)) {
            return;
        }
        Preconditions.checkState((this.state == QueueState.OPEN ? 1 : 0) != 0, (String)"%s is not OPEN", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
        NamedQueue namedQueue = new NamedQueue(queueId);
        this.namedQueues.put(queueId, namedQueue);
        this.openQueuesBySequenceId.add(namedQueue);
    }

    public synchronized void noMoreQueues() {
        this.namedQueues = ImmutableMap.copyOf(this.namedQueues);
        if (this.state != QueueState.OPEN) {
            return;
        }
        this.state = QueueState.NO_MORE_QUEUES;
        this.updateState();
    }

    public synchronized ListenableFuture<?> enqueue(Page page) {
        Preconditions.checkNotNull((Object)page, (Object)"page is null");
        if (this.closed.get()) {
            return Futures.immediateFuture((Object)true);
        }
        if (this.bufferedBytes < this.maxBufferedBytes) {
            this.addInternal(page);
            return Futures.immediateFuture((Object)true);
        }
        QueuedPage queuedPage = new QueuedPage(page);
        this.queuedPages.addLast(queuedPage);
        return queuedPage.getFuture();
    }

    private synchronized void addInternal(Page page) {
        this.masterQueue.add(page);
        this.pagesAdded.incrementAndGet();
        this.bufferedBytes += page.getDataSize().toBytes();
        this.notifyAll();
    }

    @VisibleForTesting
    public synchronized void acknowledge(String outputId, long sequenceId) {
        Preconditions.checkNotNull((Object)outputId, (Object)"outputId is null");
        NamedQueue namedQueue = this.namedQueues.get(outputId);
        if (namedQueue == null) {
            throw new NoSuchBufferException(outputId, this.namedQueues.keySet());
        }
        if (this.state == QueueState.FINISHED) {
            return;
        }
        this.openQueuesBySequenceId.remove(namedQueue);
        namedQueue.acknowledge(sequenceId);
        if (!this.closed.get()) {
            this.openQueuesBySequenceId.add(namedQueue);
        } else {
            namedQueue.setFinished();
        }
        this.updateState();
    }

    public synchronized BufferResult get(String outputId, long startingSequenceId, DataSize maxSize, Duration maxWait) throws InterruptedException {
        Preconditions.checkNotNull((Object)outputId, (Object)"outputId is null");
        Preconditions.checkArgument((maxSize.toBytes() > 0L ? 1 : 0) != 0, (Object)"maxSize must be at least 1 byte");
        Preconditions.checkNotNull((Object)maxWait, (Object)"maxWait is null");
        NamedQueue namedQueue = this.namedQueues.get(outputId);
        if (namedQueue == null) {
            throw new NoSuchBufferException(outputId, this.namedQueues.keySet());
        }
        if (this.state == QueueState.FINISHED) {
            return BufferResult.emptyResults(namedQueue.getSequenceId(), true);
        }
        if (namedQueue.isEmpty()) {
            long remainingNanos = maxWait.roundTo(TimeUnit.NANOSECONDS);
            long end = System.nanoTime() + remainingNanos;
            while (remainingNanos > 0L && namedQueue.isEmpty() && !namedQueue.isFinished()) {
                TimeUnit.NANOSECONDS.timedWait(this, remainingNanos);
                remainingNanos = end - System.nanoTime();
            }
        }
        this.openQueuesBySequenceId.remove(namedQueue);
        BufferResult results = namedQueue.getPages(startingSequenceId, maxSize);
        if (!this.closed.get() || !results.isBufferClosed()) {
            this.openQueuesBySequenceId.add(namedQueue);
        } else {
            namedQueue.setFinished();
        }
        this.updateState();
        return results;
    }

    public synchronized void abort(String outputId) {
        Preconditions.checkNotNull((Object)outputId, (Object)"outputId is null");
        NamedQueue namedQueue = this.namedQueues.get(outputId);
        if (namedQueue == null || namedQueue.isFinished()) {
            return;
        }
        namedQueue.setFinished();
        this.openQueuesBySequenceId.remove(namedQueue);
        this.updateState();
    }

    private synchronized void updateState() {
        if (this.closed.get()) {
            Iterator iterator = this.openQueuesBySequenceId.iterator();
            while (iterator.hasNext()) {
                NamedQueue namedQueue = (NamedQueue)iterator.next();
                if (!namedQueue.isEmpty()) continue;
                namedQueue.setFinished();
                iterator.remove();
            }
            for (QueuedPage queuedPage : this.queuedPages) {
                queuedPage.getFuture().set(null);
            }
            this.queuedPages.clear();
        }
        if (this.state == QueueState.NO_MORE_QUEUES && !this.openQueuesBySequenceId.isEmpty()) {
            long oldMasterSequenceId = this.masterSequenceId;
            this.masterSequenceId = ((NamedQueue)this.openQueuesBySequenceId.iterator().next()).getSequenceId();
            int pagesToRemove = Ints.checkedCast((long)(this.masterSequenceId - oldMasterSequenceId));
            Preconditions.checkState((pagesToRemove >= 0 ? 1 : 0) != 0, (String)"Master sequence id moved backwards: oldMasterSequenceId=%s, newMasterSequenceId=%s", (Object[])new Object[]{oldMasterSequenceId, this.masterSequenceId});
            for (int i = 0; i < pagesToRemove; ++i) {
                Page page = this.masterQueue.removeFirst();
                this.bufferedBytes -= page.getDataSize().toBytes();
            }
            while (!this.queuedPages.isEmpty() && this.bufferedBytes < this.maxBufferedBytes) {
                QueuedPage queuedPage = this.queuedPages.removeFirst();
                this.addInternal(queuedPage.getPage());
                queuedPage.getFuture().set(null);
            }
        }
        if (this.state == QueueState.NO_MORE_QUEUES && this.closed.get() && this.openQueuesBySequenceId.isEmpty()) {
            this.destroy();
        }
        this.notifyAll();
    }

    public synchronized void finish() {
        this.closed.set(true);
        this.updateState();
    }

    public synchronized void destroy() {
        this.closed.set(true);
        this.state = QueueState.FINISHED;
        for (NamedQueue namedQueue : this.openQueuesBySequenceId) {
            namedQueue.setFinished();
        }
        this.openQueuesBySequenceId.clear();
        this.masterQueue.clear();
        this.bufferedBytes = 0L;
        for (QueuedPage queuedPage : this.queuedPages) {
            queuedPage.getFuture().set(null);
        }
        this.queuedPages.clear();
        this.notifyAll();
    }

    private static class QueuedPage {
        private final Page page;
        private final SettableFuture<?> future = SettableFuture.create();

        private QueuedPage(Page page) {
            this.page = page;
        }

        private Page getPage() {
            return this.page;
        }

        private SettableFuture<?> getFuture() {
            return this.future;
        }
    }

    @NotThreadSafe
    private class NamedQueue
    implements Comparable<NamedQueue> {
        private final String queueId;
        private long sequenceId;
        private boolean finished;

        private NamedQueue(String queueId) {
            this.queueId = queueId;
        }

        public String getQueueId() {
            return this.queueId;
        }

        public boolean isFinished() {
            Preconditions.checkState((boolean)Thread.holdsLock(SharedBuffer.this), (String)"Thread must hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
            return this.finished;
        }

        public void setFinished() {
            Preconditions.checkState((boolean)Thread.holdsLock(SharedBuffer.this), (String)"Thread must hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
            this.finished = true;
        }

        public boolean isEmpty() {
            return this.size() == 0;
        }

        public long getSequenceId() {
            Preconditions.checkState((boolean)Thread.holdsLock(SharedBuffer.this), (String)"Thread must hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
            return this.sequenceId;
        }

        public long pagesRemoved() {
            return this.getSequenceId();
        }

        public int size() {
            Preconditions.checkState((boolean)Thread.holdsLock(SharedBuffer.this), (String)"Thread must hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
            if (this.finished) {
                return 0;
            }
            int listOffset = Ints.checkedCast((long)(this.sequenceId - SharedBuffer.this.masterSequenceId));
            if (listOffset >= SharedBuffer.this.masterQueue.size()) {
                return 0;
            }
            return SharedBuffer.this.masterQueue.size() - listOffset;
        }

        public void acknowledge(long sequenceId) {
            if (this.sequenceId < sequenceId) {
                this.sequenceId = sequenceId;
            }
        }

        public BufferResult getPages(long startingSequenceId, DataSize maxSize) {
            Preconditions.checkState((boolean)Thread.holdsLock(SharedBuffer.this), (String)"Thread must hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
            Preconditions.checkArgument((maxSize.toBytes() > 0L ? 1 : 0) != 0, (Object)"maxSize must be at least 1 byte");
            this.acknowledge(startingSequenceId);
            if (this.finished) {
                return BufferResult.emptyResults(this.sequenceId, true);
            }
            int listOffset = Ints.checkedCast((long)(this.sequenceId - SharedBuffer.this.masterSequenceId));
            if (listOffset >= SharedBuffer.this.masterQueue.size()) {
                return BufferResult.emptyResults(this.sequenceId, false);
            }
            long maxBytes = maxSize.toBytes();
            ArrayList<Page> pages = new ArrayList<Page>();
            long bytes = 0L;
            while (listOffset < SharedBuffer.this.masterQueue.size()) {
                Page page = (Page)SharedBuffer.this.masterQueue.get(listOffset++);
                if (!pages.isEmpty() && (bytes += page.getDataSize().toBytes()) > maxBytes) break;
                pages.add(page);
            }
            return new BufferResult(startingSequenceId, false, (List<Page>)ImmutableList.copyOf(pages));
        }

        @Override
        public int compareTo(NamedQueue other) {
            Preconditions.checkState((boolean)Thread.holdsLock(SharedBuffer.this), (String)"Thread must hold a lock on the %s", (Object[])new Object[]{SharedBuffer.class.getSimpleName()});
            return ComparisonChain.start().compare(this.sequenceId, other.sequenceId).compare((Comparable)((Object)this.queueId), (Comparable)((Object)other.queueId)).result();
        }

        public String toString() {
            return Objects.toStringHelper((Object)this).add("queueId", (Object)this.queueId).add("sequenceId", this.sequenceId).add("finished", this.finished).toString();
        }
    }

    public static enum QueueState {
        OPEN,
        NO_MORE_QUEUES,
        FINISHED;

    }
}

