/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.server.security.oauth2;

import com.facebook.airlift.log.Logger;
import com.facebook.presto.server.security.oauth2.ChallengeFailedException;
import com.facebook.presto.server.security.oauth2.ForRefreshTokens;
import com.facebook.presto.server.security.oauth2.JwtUtil;
import com.facebook.presto.server.security.oauth2.NonceCookie;
import com.facebook.presto.server.security.oauth2.OAuth2Client;
import com.facebook.presto.server.security.oauth2.OAuth2Config;
import com.facebook.presto.server.security.oauth2.OAuth2ErrorCode;
import com.facebook.presto.server.security.oauth2.OAuth2TokenHandler;
import com.facebook.presto.server.security.oauth2.OAuth2Utils;
import com.facebook.presto.server.security.oauth2.OAuthWebUiCookie;
import com.facebook.presto.server.security.oauth2.TokenPairSerializer;
import com.google.common.base.Strings;
import com.google.common.base.Verify;
import com.google.common.hash.Hashing;
import com.google.common.io.Resources;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.JwtParser;
import io.jsonwebtoken.security.Keys;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.ws.rs.core.NewCookie;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.UriBuilder;
import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.Key;
import java.security.SecureRandom;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.TemporalAmount;
import java.util.Date;
import java.util.Objects;
import java.util.Optional;
import java.util.Random;
import javax.inject.Inject;

public class OAuth2Service {
    private static final Logger logger = Logger.get(OAuth2Service.class);
    public static final String OPENID_SCOPE = "openid";
    private static final String STATE_AUDIENCE_UI = "presto_oauth_ui";
    private static final String FAILURE_REPLACEMENT_TEXT = "<!-- ERROR_MESSAGE -->";
    private static final Random SECURE_RANDOM = new SecureRandom();
    public static final String HANDLER_STATE_CLAIM = "handler_state";
    private final OAuth2Client client;
    private final Optional<Duration> tokenExpiration;
    private final TokenPairSerializer tokenPairSerializer;
    private final String successHtml;
    private final String failureHtml;
    private final TemporalAmount challengeTimeout;
    private final Key stateHmac;
    private final JwtParser jwtParser;
    private final OAuth2TokenHandler tokenHandler;

    @Inject
    public OAuth2Service(OAuth2Client client, OAuth2Config oauth2Config, OAuth2TokenHandler tokenHandler, TokenPairSerializer tokenPairSerializer, @ForRefreshTokens Optional<Duration> tokenExpiration) throws IOException {
        this.client = Objects.requireNonNull(client, "client is null");
        Objects.requireNonNull(oauth2Config, "oauth2Config is null");
        this.successHtml = Resources.toString((URL)Resources.getResource(this.getClass(), (String)"/webapp/oauth2/success.html"), (Charset)StandardCharsets.UTF_8);
        this.failureHtml = Resources.toString((URL)Resources.getResource(this.getClass(), (String)"/webapp/oauth2/failure.html"), (Charset)StandardCharsets.UTF_8);
        Verify.verify((boolean)this.failureHtml.contains(FAILURE_REPLACEMENT_TEXT), (String)"failure.html does not contain the replacement text", (Object[])new Object[0]);
        this.challengeTimeout = Duration.ofMillis(oauth2Config.getChallengeTimeout().toMillis());
        this.stateHmac = Keys.hmacShaKeyFor((byte[])oauth2Config.getStateKey().map(key -> Hashing.sha256().hashString((CharSequence)key, StandardCharsets.UTF_8).asBytes()).orElseGet(() -> OAuth2Service.secureRandomBytes(32)));
        this.jwtParser = JwtUtil.newJwtParserBuilder().setSigningKey(this.stateHmac).requireAudience(STATE_AUDIENCE_UI).build();
        this.tokenHandler = Objects.requireNonNull(tokenHandler, "tokenHandler is null");
        this.tokenPairSerializer = Objects.requireNonNull(tokenPairSerializer, "tokenPairSerializer is null");
        this.tokenExpiration = Objects.requireNonNull(tokenExpiration, "tokenExpiration is null");
    }

    public Response startOAuth2Challenge(URI callbackUri, Optional<String> handlerState) {
        Instant challengeExpiration = Instant.now().plus(this.challengeTimeout);
        String state = JwtUtil.newJwtBuilder().signWith(this.stateHmac).setAudience(STATE_AUDIENCE_UI).claim(HANDLER_STATE_CLAIM, handlerState.orElse(null)).setExpiration(Date.from(challengeExpiration)).compact();
        OAuth2Client.Request request = this.client.createAuthorizationRequest(state, callbackUri);
        Response.ResponseBuilder response = Response.seeOther((URI)request.getAuthorizationUri());
        request.getNonce().ifPresent(nce -> response.cookie(new NewCookie[]{NonceCookie.create(nce, challengeExpiration)}));
        return response.build();
    }

    public void startOAuth2Challenge(URI callbackUri, Optional<String> handlerState, HttpServletResponse servletResponse) throws IOException {
        Instant challengeExpiration = Instant.now().plus(this.challengeTimeout);
        OAuth2Client.Request challengeRequest = this.startChallenge(callbackUri, handlerState);
        challengeRequest.getNonce().ifPresent(nce -> servletResponse.addCookie(NonceCookie.createServletCookie(nce, challengeExpiration)));
        this.servletResponseSeeOther(challengeRequest.getAuthorizationUri().toString(), servletResponse);
    }

    public void servletResponseSeeOther(String location, HttpServletResponse servletResponse) throws IOException {
        servletResponse.addHeader("Location", location);
        servletResponse.sendError(303);
    }

    private OAuth2Client.Request startChallenge(URI callbackUri, Optional<String> handlerState) {
        Instant challengeExpiration = Instant.now().plus(this.challengeTimeout);
        String state = JwtUtil.newJwtBuilder().signWith(this.stateHmac).setAudience(STATE_AUDIENCE_UI).claim(HANDLER_STATE_CLAIM, handlerState.orElse(null)).setExpiration(Date.from(challengeExpiration)).compact();
        return this.client.createAuthorizationRequest(state, callbackUri);
    }

    public Response handleOAuth2Error(String state, String error, String errorDescription, String errorUri) {
        try {
            Claims stateClaims = this.parseState(state);
            Optional.ofNullable((String)stateClaims.get(HANDLER_STATE_CLAIM, String.class)).ifPresent(value -> this.tokenHandler.setTokenExchangeError((String)value, String.format("Authentication response could not be verified: error=%s, errorDescription=%s, errorUri=%s", error, errorDescription, errorDescription)));
        }
        catch (ChallengeFailedException | RuntimeException e) {
            logger.error((Throwable)e, "Authentication response could not be verified invalid state: state=%s", new Object[]{state});
            return Response.status((Response.Status)Response.Status.FORBIDDEN).entity((Object)this.getInternalFailureHtml("Authentication response could not be verified")).cookie(new NewCookie[]{NonceCookie.delete()}).build();
        }
        logger.error("OAuth server returned an error: error=%s, error_description=%s, error_uri=%s, state=%s", new Object[]{error, errorDescription, errorUri, state});
        return Response.ok().entity((Object)this.getCallbackErrorHtml(error)).cookie(new NewCookie[]{NonceCookie.delete()}).build();
    }

    public Response finishOAuth2Challenge(String state, String code, URI callbackUri, Optional<String> nonce, HttpServletRequest request) {
        Optional<String> handlerState;
        try {
            Claims stateClaims = this.parseState(state);
            handlerState = Optional.ofNullable((String)stateClaims.get(HANDLER_STATE_CLAIM, String.class));
        }
        catch (ChallengeFailedException | RuntimeException e) {
            logger.error((Throwable)e, "Authentication response could not be verified invalid state: state=%s", new Object[]{state});
            return Response.status((Response.Status)Response.Status.BAD_REQUEST).entity((Object)this.getInternalFailureHtml("Authentication response could not be verified")).cookie(new NewCookie[]{NonceCookie.delete()}).build();
        }
        try {
            OAuth2Client.Response oauth2Response = this.client.getOAuth2Response(code, callbackUri, nonce);
            if (!handlerState.isPresent()) {
                UriBuilder uriBuilder = OAuth2Utils.getSchemeUriBuilder(request);
                return Response.seeOther((URI)uriBuilder.build(new Object[0]).resolve("/ui/")).cookie(new NewCookie[]{OAuthWebUiCookie.create(this.tokenPairSerializer.serialize(TokenPairSerializer.TokenPair.fromOAuth2Response(oauth2Response)), this.tokenExpiration.map(expiration -> Instant.now().plus((TemporalAmount)expiration)).orElse(oauth2Response.getExpiration())), NonceCookie.delete()}).build();
            }
            this.tokenHandler.setAccessToken(handlerState.get(), this.tokenPairSerializer.serialize(TokenPairSerializer.TokenPair.fromOAuth2Response(oauth2Response)));
            Response.ResponseBuilder builder = Response.ok((Object)this.getSuccessHtml());
            builder.cookie(new NewCookie[]{OAuthWebUiCookie.create(this.tokenPairSerializer.serialize(TokenPairSerializer.TokenPair.fromOAuth2Response(oauth2Response)), this.tokenExpiration.map(expiration -> Instant.now().plus((TemporalAmount)expiration)).orElse(oauth2Response.getExpiration()))});
            return builder.cookie(new NewCookie[]{NonceCookie.delete()}).build();
        }
        catch (ChallengeFailedException | RuntimeException e) {
            logger.error((Throwable)e, "Authentication response could not be verified: state=%s", new Object[]{state});
            handlerState.ifPresent(value -> this.tokenHandler.setTokenExchangeError((String)value, String.format("Authentication response could not be verified: state=%s", value)));
            return Response.status((Response.Status)Response.Status.BAD_REQUEST).cookie(new NewCookie[]{NonceCookie.delete()}).entity((Object)this.getInternalFailureHtml("Authentication response could not be verified")).build();
        }
    }

    private Claims parseState(String state) throws ChallengeFailedException {
        try {
            return (Claims)this.jwtParser.parseClaimsJws(state).getBody();
        }
        catch (RuntimeException e) {
            throw new ChallengeFailedException("State validation failed", e);
        }
    }

    public String getSuccessHtml() {
        return this.successHtml;
    }

    public String getCallbackErrorHtml(String errorCode) {
        return this.failureHtml.replace(FAILURE_REPLACEMENT_TEXT, OAuth2Service.getOAuth2ErrorMessage(errorCode));
    }

    public String getInternalFailureHtml(String errorMessage) {
        return this.failureHtml.replace(FAILURE_REPLACEMENT_TEXT, Strings.nullToEmpty((String)errorMessage));
    }

    private static byte[] secureRandomBytes(int count) {
        byte[] bytes = new byte[count];
        SECURE_RANDOM.nextBytes(bytes);
        return bytes;
    }

    private static String getOAuth2ErrorMessage(String errorCode) {
        try {
            OAuth2ErrorCode code = OAuth2ErrorCode.fromString(errorCode);
            return code.getMessage();
        }
        catch (IllegalArgumentException e) {
            logger.error((Throwable)e, "Unknown error code received code=%s", new Object[]{errorCode});
            return "OAuth2 unknown error code: " + errorCode;
        }
    }
}

