/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.server.security.oauth2;

import com.facebook.airlift.http.server.AuthenticationException;
import com.facebook.airlift.http.server.Authenticator;
import com.facebook.airlift.http.server.BasicPrincipal;
import com.facebook.airlift.log.Logger;
import com.facebook.presto.server.security.oauth2.OAuth2Client;
import com.facebook.presto.server.security.oauth2.OAuth2Config;
import com.facebook.presto.server.security.oauth2.OAuth2TokenExchangeResource;
import com.facebook.presto.server.security.oauth2.OAuth2Utils;
import com.facebook.presto.server.security.oauth2.TokenPairSerializer;
import com.facebook.presto.server.security.oauth2.TokenRefresher;
import com.google.common.base.Strings;
import jakarta.servlet.http.Cookie;
import jakarta.servlet.http.HttpServletRequest;
import java.net.URI;
import java.security.Principal;
import java.sql.Date;
import java.time.Instant;
import java.util.Arrays;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import javax.inject.Inject;
import org.apache.commons.lang3.StringUtils;

public class OAuth2Authenticator
implements Authenticator {
    private static final Logger Log = Logger.get(OAuth2Authenticator.class);
    private final String principalField;
    private final OAuth2Client client;
    private final TokenPairSerializer tokenPairSerializer;
    private final TokenRefresher tokenRefresher;

    @Inject
    public OAuth2Authenticator(OAuth2Client client, OAuth2Config config, TokenRefresher tokenRefresher, TokenPairSerializer tokenPairSerializer) {
        this.client = Objects.requireNonNull(client, "service is null");
        this.principalField = config.getPrincipalField();
        Objects.requireNonNull(config, "oauth2Config is null");
        this.tokenRefresher = Objects.requireNonNull(tokenRefresher, "tokenRefresher is null");
        this.tokenPairSerializer = Objects.requireNonNull(tokenPairSerializer, "tokenPairSerializer is null");
    }

    public Principal authenticate(HttpServletRequest request) throws AuthenticationException {
        TokenPairSerializer.TokenPair tokenPair;
        String token = this.extractToken(request);
        try {
            tokenPair = this.tokenPairSerializer.deserialize(token);
        }
        catch (IllegalArgumentException e) {
            Log.error((Throwable)e, "Failed to deserialize the OAuth token");
            throw this.needAuthentication(request, Optional.empty(), "Invalid Credentials");
        }
        if (tokenPair.getExpiration().before(Date.from(Instant.now()))) {
            throw this.needAuthentication(request, Optional.of(token), "Invalid Credentials");
        }
        Optional<Map<String, Object>> claims = this.client.getClaims(tokenPair.getAccessToken());
        if (!claims.isPresent()) {
            throw this.needAuthentication(request, Optional.ofNullable(token), "Invalid Credentials");
        }
        String principal = (String)claims.get().get(this.principalField);
        if (StringUtils.isEmpty((CharSequence)principal)) {
            Log.warn("The subject is not present we need to authenticate");
            this.needAuthentication(request, Optional.empty(), "Invalid Credentials");
        }
        return new BasicPrincipal(principal);
    }

    public String extractToken(HttpServletRequest request) throws AuthenticationException {
        Optional<String> cookieToken = OAuth2Authenticator.extractTokenFromCookie(request);
        Optional<String> headerToken = this.extractTokenFromHeader(request);
        if (!cookieToken.isPresent() && !headerToken.isPresent()) {
            throw this.needAuthentication(request, Optional.empty(), "Invalid Credentials");
        }
        return cookieToken.orElseGet(() -> (String)headerToken.get());
    }

    public Optional<String> extractTokenFromHeader(HttpServletRequest request) {
        String authHeader = Strings.nullToEmpty((String)request.getHeader("Authorization"));
        int space = authHeader.indexOf(32);
        if (space < 0 || !authHeader.substring(0, space).equalsIgnoreCase("bearer")) {
            return Optional.empty();
        }
        return Optional.ofNullable(authHeader.substring(space + 1).trim()).filter(t -> !t.isEmpty());
    }

    public static Optional<String> extractTokenFromCookie(HttpServletRequest request) {
        Cookie[] cookies = Optional.ofNullable(request.getCookies()).orElse(new Cookie[0]);
        return Optional.ofNullable(Arrays.stream(cookies).filter(cookie -> cookie.getName().equals("__Secure-Presto-OAuth2-Token")).findFirst().map(c -> c.getValue()).orElse(null));
    }

    private AuthenticationException needAuthentication(HttpServletRequest request, Optional<String> currentToken, String message) {
        URI baseUri = OAuth2Utils.getSchemeUriBuilder(request).build(new Object[0]);
        return currentToken.map(this.tokenPairSerializer::deserialize).flatMap(this.tokenRefresher::refreshToken).map(refreshId -> baseUri.resolve(OAuth2TokenExchangeResource.getTokenUri(refreshId))).map(tokenUri -> new AuthenticationException(message, String.format("Bearer x_token_server=\"%s\"", tokenUri))).orElseGet(() -> this.buildNeedAuthentication(request, message));
    }

    private AuthenticationException buildNeedAuthentication(HttpServletRequest request, String message) {
        UUID authId = UUID.randomUUID();
        URI baseUri = OAuth2Utils.getSchemeUriBuilder(request).build(new Object[0]);
        URI initiateUri = baseUri.resolve(OAuth2TokenExchangeResource.getInitiateUri(authId));
        URI tokenUri = baseUri.resolve(OAuth2TokenExchangeResource.getTokenUri(authId));
        return new AuthenticationException(message, String.format("Bearer x_redirect_server=\"%s\", x_token_server=\"%s\"", initiateUri, tokenUri));
    }
}

