/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.operator.scalar;

import com.facebook.presto.common.block.Block;
import com.facebook.presto.common.block.BlockBuilder;
import com.facebook.presto.common.type.Type;
import com.facebook.presto.operator.scalar.ArraySortFunction;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.function.Description;
import com.facebook.presto.spi.function.LiteralParameters;
import com.facebook.presto.spi.function.ScalarFunction;
import com.facebook.presto.spi.function.SqlType;
import com.facebook.presto.type.IpAddressOperators;
import com.facebook.presto.type.IpAddressType;
import com.facebook.presto.type.IpPrefixOperators;
import com.facebook.presto.type.IpPrefixType;
import com.facebook.presto.util.Failures;
import com.google.common.net.InetAddresses;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import java.math.BigInteger;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public final class IpPrefixFunctions {
    private static final BigInteger TWO = BigInteger.valueOf(2L);

    private IpPrefixFunctions() {
    }

    @Description(value="IP prefix for a given IP address and subnet size")
    @ScalarFunction(value="ip_prefix")
    @SqlType(value="ipprefix")
    public static Slice ipPrefix(@SqlType(value="ipaddress") Slice value, @SqlType(value="bigint") long subnetSize) {
        InetAddress address = IpPrefixFunctions.toInetAddress(value);
        int addressLength = address.getAddress().length;
        if (addressLength == 4) {
            Failures.checkCondition(0L <= subnetSize && subnetSize <= 32L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "IPv4 subnet size must be in range [0, 32]", new Object[0]);
        } else if (addressLength == 16) {
            Failures.checkCondition(0L <= subnetSize && subnetSize <= 128L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "IPv6 subnet size must be in range [0, 128]", new Object[0]);
        } else {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Invalid InetAddress length: " + addressLength);
        }
        return IpPrefixOperators.castFromVarcharToIpPrefix(Slices.utf8Slice((String)(InetAddresses.toAddrString((InetAddress)address) + "/" + subnetSize)));
    }

    @Description(value="IP prefix for a given IP address and subnet size")
    @ScalarFunction(value="ip_prefix")
    @LiteralParameters(value={"x"})
    @SqlType(value="ipprefix")
    public static Slice stringIpPrefix(@SqlType(value="varchar(x)") Slice slice, @SqlType(value="bigint") long subnetSize) {
        return IpPrefixFunctions.ipPrefix(IpAddressOperators.castFromVarcharToIpAddress(slice), subnetSize);
    }

    @Description(value="Smallest IP address for a given IP prefix")
    @ScalarFunction(value="ip_subnet_min")
    @SqlType(value="ipaddress")
    public static Slice ipSubnetMin(@SqlType(value="ipprefix") Slice value) {
        return IpPrefixOperators.castFromIpPrefixToIpAddress(value);
    }

    @Description(value="Largest IP address for a given IP prefix")
    @ScalarFunction(value="ip_subnet_max")
    @SqlType(value="ipaddress")
    public static Slice ipSubnetMax(@SqlType(value="ipprefix") Slice value) {
        byte[] address = IpPrefixFunctions.toInetAddress(value.slice(0, IpAddressType.IPADDRESS.getFixedSize())).getAddress();
        int subnetSize = value.getByte(IpPrefixType.IPPREFIX.getFixedSize() - 1) & 0xFF;
        if (address.length == 4) {
            for (int i = 0; i < 4; ++i) {
                int n = 3 - i;
                address[n] = (byte)(address[n] | (byte)(~(-1 << Math.min(Math.max(32 - subnetSize - 8 * i, 0), 8))));
            }
            byte[] bytes = new byte[16];
            bytes[10] = -1;
            bytes[11] = -1;
            System.arraycopy(address, 0, bytes, 12, 4);
            address = bytes;
        } else if (address.length == 16) {
            for (int i = 0; i < 16; ++i) {
                int n = 15 - i;
                address[n] = (byte)(address[n] | (byte)(~(-1 << Math.min(Math.max(128 - subnetSize - 8 * i, 0), 8))));
            }
        }
        return Slices.wrappedBuffer((byte[])address);
    }

    @Description(value="Array of smallest and largest IP address in the subnet of the given IP prefix")
    @ScalarFunction(value="ip_subnet_range")
    @SqlType(value="array(IPADDRESS)")
    public static Block ipSubnetRange(@SqlType(value="ipprefix") Slice value) {
        BlockBuilder blockBuilder = IpAddressType.IPADDRESS.createBlockBuilder(null, 2);
        IpAddressType.IPADDRESS.writeSlice(blockBuilder, IpPrefixFunctions.ipSubnetMin(value));
        IpAddressType.IPADDRESS.writeSlice(blockBuilder, IpPrefixFunctions.ipSubnetMax(value));
        return blockBuilder.build();
    }

    @Description(value="Is the IP address in the subnet of IP prefix")
    @ScalarFunction(value="is_subnet_of")
    @SqlType(value="boolean")
    public static boolean isSubnetOf(@SqlType(value="ipprefix") Slice ipPrefix, @SqlType(value="ipaddress") Slice ipAddress) {
        IpPrefixFunctions.toInetAddress(ipAddress);
        return IpAddressOperators.between(ipAddress, IpPrefixFunctions.ipSubnetMin(ipPrefix), IpPrefixFunctions.ipSubnetMax(ipPrefix));
    }

    @Description(value="Is the second IP prefix in the subnet of the first IP prefix")
    @ScalarFunction(value="is_subnet_of")
    @SqlType(value="boolean")
    public static boolean isPrefixSubnetOf(@SqlType(value="ipprefix") Slice first, @SqlType(value="ipprefix") Slice second) {
        return IpAddressOperators.between(IpPrefixFunctions.ipSubnetMin(second), IpPrefixFunctions.ipSubnetMin(first), IpPrefixFunctions.ipSubnetMax(first)) && IpAddressOperators.between(IpPrefixFunctions.ipSubnetMax(second), IpPrefixFunctions.ipSubnetMin(first), IpPrefixFunctions.ipSubnetMax(first));
    }

    @Description(value="Combines the input set of IP prefixes into the fewest contiguous CIDR ranges possible.")
    @ScalarFunction(value="ip_prefix_collapse")
    @SqlType(value="array(IPPREFIX)")
    public static Block collapseIpPrefixes(@SqlType(value="array(IPPREFIX)") Block unsortedIpPrefixArray) {
        int inputPrefixCount = unsortedIpPrefixArray.getPositionCount();
        if (inputPrefixCount == 0 || inputPrefixCount == 1 && !unsortedIpPrefixArray.isNull(0)) {
            return unsortedIpPrefixArray;
        }
        Block ipPrefixArray = ArraySortFunction.sort(null, (Type)IpPrefixType.IPPREFIX, unsortedIpPrefixArray);
        if (ipPrefixArray.isNull(0) || ipPrefixArray.isNull(inputPrefixCount - 1)) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "ip_prefix_collapse does not support null elements");
        }
        Slice firstIpPrefix = IpPrefixType.IPPREFIX.getSlice(ipPrefixArray, 0);
        boolean v4 = IpPrefixFunctions.isIpv4(firstIpPrefix);
        Slice lastIpPrefix = IpPrefixType.IPPREFIX.getSlice(ipPrefixArray, inputPrefixCount - 1);
        if (IpPrefixFunctions.isIpv4(lastIpPrefix) != v4) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "All IPPREFIX elements must be the same IP version.");
        }
        ArrayList<List<Slice>> outputIpPrefixes = new ArrayList<List<Slice>>();
        int outputPrefixCount = 0;
        int ipMaxBitLength = v4 ? 32 : 128;
        List<BigInteger[]> mergedIpRanges = IpPrefixFunctions.mergeIpRanges(ipPrefixArray);
        for (BigInteger[] ipRange : mergedIpRanges) {
            List<Slice> list = IpPrefixFunctions.generateMinIpPrefixes(ipRange[0], ipRange[1], ipMaxBitLength);
            outputIpPrefixes.add(list);
            outputPrefixCount += list.size();
        }
        BlockBuilder blockBuilder = IpPrefixType.IPPREFIX.createBlockBuilder(null, outputPrefixCount);
        for (List list : outputIpPrefixes) {
            for (Slice ipPrefix : list) {
                IpPrefixType.IPPREFIX.writeSlice(blockBuilder, ipPrefix);
            }
        }
        return blockBuilder.build();
    }

    private static List<Slice> generateMinIpPrefixes(BigInteger firstIpAddress, BigInteger lastIpAddress, int ipVersionMaxBits) {
        ArrayList<Slice> ipPrefixSlices = new ArrayList<Slice>();
        while (firstIpAddress.compareTo(lastIpAddress) <= 0) {
            long rangeBits = IpPrefixFunctions.findRangeBits(firstIpAddress, lastIpAddress);
            int prefixLength = (int)((long)ipVersionMaxBits - rangeBits);
            try {
                InetAddress asInetAddress = IpPrefixFunctions.bigIntegerToIpAddress(firstIpAddress);
                Slice ipPrefixAsSlice = IpPrefixOperators.castFromVarcharToIpPrefix(Slices.utf8Slice((String)(InetAddresses.toAddrString((InetAddress)asInetAddress) + "/" + prefixLength)));
                ipPrefixSlices.add(ipPrefixAsSlice);
            }
            catch (UnknownHostException ex) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Unable to convert " + firstIpAddress + " to IP prefix", (Throwable)ex);
            }
            BigInteger ipCount = TWO.pow(ipVersionMaxBits - prefixLength);
            firstIpAddress = firstIpAddress.add(ipCount);
        }
        return ipPrefixSlices;
    }

    private static long findRangeBits(BigInteger firstIpAddress, BigInteger lastIpAddress) {
        BigInteger ipCount = lastIpAddress.subtract(firstIpAddress).add(BigInteger.ONE);
        int firstAddressMaxBits = firstIpAddress.getLowestSetBit();
        int ipRangeMaxBits = TWO.pow(ipCount.bitLength()).compareTo(ipCount) > 0 ? ipCount.bitLength() - 1 : ipCount.bitLength();
        return Math.min(firstAddressMaxBits, ipRangeMaxBits);
    }

    private static List<BigInteger[]> mergeIpRanges(Block ipPrefixArray) {
        ArrayList<BigInteger[]> mergedRanges = new ArrayList<BigInteger[]>();
        Slice startingIpPrefix = IpPrefixType.IPPREFIX.getSlice(ipPrefixArray, 0);
        BigInteger firstIpAddress = IpPrefixFunctions.toBigInteger(IpPrefixFunctions.ipSubnetMin(startingIpPrefix));
        BigInteger lastIpAddress = IpPrefixFunctions.toBigInteger(IpPrefixFunctions.ipSubnetMax(startingIpPrefix));
        for (int i = 1; i < ipPrefixArray.getPositionCount(); ++i) {
            Slice ipPrefix = IpPrefixType.IPPREFIX.getSlice(ipPrefixArray, i);
            BigInteger nextFirstIpAddress = IpPrefixFunctions.toBigInteger(IpPrefixFunctions.ipSubnetMin(ipPrefix));
            BigInteger nextLastIpAddress = IpPrefixFunctions.toBigInteger(IpPrefixFunctions.ipSubnetMax(ipPrefix));
            if (lastIpAddress.compareTo(nextFirstIpAddress) >= 0) continue;
            if (lastIpAddress.add(BigInteger.ONE).compareTo(nextFirstIpAddress) != 0) {
                BigInteger[] finalizedRange = new BigInteger[]{firstIpAddress, lastIpAddress};
                mergedRanges.add(finalizedRange);
                firstIpAddress = nextFirstIpAddress;
            }
            lastIpAddress = nextLastIpAddress;
        }
        BigInteger[] finalizedRange = new BigInteger[]{firstIpAddress, lastIpAddress};
        mergedRanges.add(finalizedRange);
        return mergedRanges;
    }

    private static byte[] bigIntegerToIPAddressBytes(BigInteger ipAddress) {
        byte[] ipAddressBytes = ipAddress.toByteArray();
        if ((ipAddressBytes.length == 5 || ipAddressBytes.length == 17) && ipAddressBytes[0] == 0) {
            ipAddressBytes = Arrays.copyOfRange(ipAddressBytes, 1, ipAddressBytes.length);
        } else if (ipAddressBytes.length <= 3 || ipAddressBytes.length != 4 && ipAddressBytes.length <= 15) {
            byte[] emptyRange = new byte[ipAddressBytes.length <= 3 ? 4 : 16];
            int j = emptyRange.length - ipAddressBytes.length;
            int i = 0;
            while (i < ipAddressBytes.length) {
                emptyRange[j] = ipAddressBytes[i];
                ++i;
                ++j;
            }
            ipAddressBytes = emptyRange;
        }
        return ipAddressBytes;
    }

    private static InetAddress bigIntegerToIpAddress(BigInteger ipAddress) throws UnknownHostException {
        byte[] ipAddressBytes = IpPrefixFunctions.bigIntegerToIPAddressBytes(ipAddress);
        return InetAddress.getByAddress(ipAddressBytes);
    }

    private static BigInteger toBigInteger(Slice ipAddress) {
        return new BigInteger(1, ipAddress.getBytes());
    }

    private static boolean isIpv4(Slice ipPrefix) {
        byte[] ipPartBytes = ipPrefix.getBytes(0, 16);
        for (int i = 0; i <= 9; ++i) {
            if (ipPartBytes[i] == 0) continue;
            return false;
        }
        return ipPartBytes[10] == -1 && ipPartBytes[11] == -1;
    }

    private static InetAddress toInetAddress(Slice ipAddress) {
        try {
            return InetAddress.getByAddress(ipAddress.getBytes());
        }
        catch (UnknownHostException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid IP address binary: " + ipAddress.toStringUtf8(), (Throwable)e);
        }
    }
}

