import { Column, PrestoClientConfig, PrestoQuery, QueryInfo, Table } from './types';
export declare class PrestoClient {
    private baseUrl;
    private catalog?;
    private headers;
    private interval?;
    private retryInterval;
    private schema?;
    private source?;
    private timezone?;
    private user;
    /**
     * Creates an instance of PrestoClient.
     * @param {PrestoClientConfig} config - Configuration object for the PrestoClient.
     * @param {Object} config.basicAuthorization - Optional object for basic authorization.
     * @param {Object} config.basicAuthorization.user - The basic auth user name.
     * @param {Object} config.basicAuthorization.password - The basic auth password.
     * @param {string} config.authorizationToken - An optional token to be sent in the authorization header. Takes precedence over the basic auth.
     * @param {string} config.catalog - The default catalog to be used.
     * @param {Record<string, string>} config.extraHeaders - Any extra headers to include in the API requests. Optional.
     * @param {string} config.host - The host address of the Presto server.
     * @param {number} config.interval - The polling interval in milliseconds for query status checks.
     * @param {number} config.port - The port number on which the Presto server is listening.
     * @param {string} [config.schema] - The default schema to be used. Optional.
     * @param {string} [config.source] - The name of the source making the query. Optional.
     * @param {string} [config.timezone] - The timezone to be used for the session. Optional.
     * @param {string} config.user - The username to be used for the Presto session.
     */
    constructor({ basicAuthentication, authorizationToken, catalog, extraHeaders, host, interval, port, schema, source, timezone, user, }: PrestoClientConfig);
    /**
     * Retrieves all catalogs.
     * @returns {Promise<string[] | undefined>} An array of all the catalog names.
     */
    getCatalogs(): Promise<string[] | undefined>;
    /**
     * Retrieves a list of columns filtered for the given catalog and optional filters.
     * @param {Object} options - The options for retrieving columns.
     * @param {string} options.catalog - The catalog name.
     * @param {string} [options.schema] - The schema name. Optional.
     * @param {string} [options.table] - The table name. Optional.
     * @returns {Promise<Column[] | undefined>} An array of all the columns that match the given filters.
     */
    getColumns({ catalog, schema, table, }: {
        catalog: string;
        schema?: string;
        table?: string;
    }): Promise<Column[] | undefined>;
    /**
     * Retrieves all the information for a given query
     * @param {string} queryId The query identifier string
     * @returns {Promise<QueryInfo | undefined>} All the query information
     */
    getQueryInfo(queryId: string): Promise<QueryInfo | undefined>;
    /**
     * Retrieves all schemas within a given catalog.
     * @param {string} catalog - The name of the catalog for which to retrieve schemas.
     * @returns {Promise<string[] | undefined>} An array of schema names within the specified catalog.
     */
    getSchemas(catalog: string): Promise<string[] | undefined>;
    /**
     * Retrieves a list of tables filtered by the given catalog and optional schema.
     * @param {Object} options - The options for retrieving tables.
     * @param {string} options.catalog - The catalog name.
     * @param {string} [options.schema] - The schema name. Optional.
     * @returns {Promise<Table[] | undefined>} An array of tables that match the given filters.
     */
    getTables({ catalog, schema }: {
        catalog: string;
        schema?: string;
    }): Promise<Table[] | undefined>;
    /**
     * Executes a given query with optional catalog and schema settings.
     * @param {string} query - The SQL query string to be executed.
     * @param {Object} [options] - Optional parameters for the query.
     * @param {string} [options.catalog] - The catalog to be used for the query. Optional.
     * @param {string} [options.schema] - The schema to be used for the query. Optional.
     * @returns {Promise<PrestoQuery>} A promise that resolves to the result of the query execution.
     * @throws {PrestoError} If the underlying Presto engine returns an error
     */
    query(query: string, options?: {
        catalog?: string;
        schema?: string;
    }): Promise<PrestoQuery>;
    private delay;
    private getWhereCondition;
    private request;
}
export default PrestoClient;
