/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.operator.scalar;

import com.facebook.presto.common.block.Block;
import com.facebook.presto.common.type.Decimals;
import com.facebook.presto.common.type.DoubleType;
import com.facebook.presto.common.type.Type;
import com.facebook.presto.common.type.UnscaledDecimal128Arithmetic;
import com.facebook.presto.common.type.VarcharType;
import com.facebook.presto.metadata.SqlScalarFunction;
import com.facebook.presto.operator.aggregation.TypedSet;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.function.Description;
import com.facebook.presto.spi.function.FunctionKind;
import com.facebook.presto.spi.function.LiteralParameters;
import com.facebook.presto.spi.function.ScalarFunction;
import com.facebook.presto.spi.function.Signature;
import com.facebook.presto.spi.function.SqlNullable;
import com.facebook.presto.spi.function.SqlType;
import com.facebook.presto.type.Constraint;
import com.facebook.presto.type.Constraints;
import com.facebook.presto.type.DecimalOperators;
import com.facebook.presto.type.LiteralParameter;
import com.facebook.presto.util.Failures;
import com.google.common.primitives.Doubles;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.concurrent.ThreadLocalRandom;
import org.apache.commons.math3.distribution.BetaDistribution;
import org.apache.commons.math3.distribution.BinomialDistribution;
import org.apache.commons.math3.distribution.ChiSquaredDistribution;
import org.apache.commons.math3.distribution.PoissonDistribution;
import org.apache.commons.math3.special.Erf;

public final class MathFunctions {
    public static final SqlScalarFunction DECIMAL_MOD_FUNCTION = MathFunctions.decimalModFunction();
    private static final Slice[] DECIMAL_HALF_UNSCALED_FOR_SCALE = new Slice[38];
    private static final Slice[] DECIMAL_ALMOST_HALF_UNSCALED_FOR_SCALE = new Slice[38];

    private MathFunctions() {
    }

    @Description(value="absolute value")
    @ScalarFunction(value="abs")
    @SqlType(value="tinyint")
    public static long absTinyint(@SqlType(value="tinyint") long num) {
        Failures.checkCondition(num != -128L, (ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, "Value -128 is out of range for abs(tinyint)", new Object[0]);
        return Math.abs(num);
    }

    @Description(value="absolute value")
    @ScalarFunction(value="abs")
    @SqlType(value="smallint")
    public static long absSmallint(@SqlType(value="smallint") long num) {
        Failures.checkCondition(num != -32768L, (ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, "Value -32768 is out of range for abs(smallint)", new Object[0]);
        return Math.abs(num);
    }

    @Description(value="absolute value")
    @ScalarFunction(value="abs")
    @SqlType(value="integer")
    public static long absInteger(@SqlType(value="integer") long num) {
        Failures.checkCondition(num != Integer.MIN_VALUE, (ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, "Value -2147483648 is out of range for abs(integer)", new Object[0]);
        return Math.abs(num);
    }

    @Description(value="absolute value")
    @ScalarFunction
    @SqlType(value="bigint")
    public static long abs(@SqlType(value="bigint") long num) {
        Failures.checkCondition(num != Long.MIN_VALUE, (ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, "Value -9223372036854775808 is out of range for abs(bigint)", new Object[0]);
        return Math.abs(num);
    }

    @Description(value="absolute value")
    @ScalarFunction
    @SqlType(value="double")
    public static double abs(@SqlType(value="double") double num) {
        return Math.abs(num);
    }

    @Description(value="absolute value")
    @ScalarFunction(value="abs")
    @SqlType(value="real")
    public static long absFloat(@SqlType(value="real") long num) {
        return Float.floatToRawIntBits(Math.abs(Float.intBitsToFloat((int)num)));
    }

    @Description(value="arc cosine")
    @ScalarFunction
    @SqlType(value="double")
    public static double acos(@SqlType(value="double") double num) {
        return Math.acos(num);
    }

    @Description(value="arc sine")
    @ScalarFunction
    @SqlType(value="double")
    public static double asin(@SqlType(value="double") double num) {
        return Math.asin(num);
    }

    @Description(value="arc tangent")
    @ScalarFunction
    @SqlType(value="double")
    public static double atan(@SqlType(value="double") double num) {
        return Math.atan(num);
    }

    @Description(value="arc tangent of given fraction")
    @ScalarFunction
    @SqlType(value="double")
    public static double atan2(@SqlType(value="double") double num1, @SqlType(value="double") double num2) {
        return Math.atan2(num1, num2);
    }

    @Description(value="Binomial cdf given numberOfTrials, successProbability, and a value")
    @ScalarFunction
    @SqlType(value="double")
    public static double binomialCdf(@SqlType(value="integer") long numberOfTrials, @SqlType(value="double") double successProbability, @SqlType(value="integer") long value) {
        Failures.checkCondition(successProbability >= 0.0 && successProbability <= 1.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "successProbability must be in the interval [0, 1]", new Object[0]);
        Failures.checkCondition(numberOfTrials > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "numberOfTrials must be greater than 0", new Object[0]);
        BinomialDistribution distribution = new BinomialDistribution(null, (int)numberOfTrials, successProbability);
        return distribution.cumulativeProbability((int)value);
    }

    @Description(value="cube root")
    @ScalarFunction
    @SqlType(value="double")
    public static double cbrt(@SqlType(value="double") double num) {
        return Math.cbrt(num);
    }

    @Description(value="round up to nearest integer")
    @ScalarFunction(value="ceiling", alias={"ceil"})
    @SqlType(value="tinyint")
    public static long ceilingTinyint(@SqlType(value="tinyint") long num) {
        return num;
    }

    @Description(value="round up to nearest integer")
    @ScalarFunction(value="ceiling", alias={"ceil"})
    @SqlType(value="smallint")
    public static long ceilingSmallint(@SqlType(value="smallint") long num) {
        return num;
    }

    @Description(value="round up to nearest integer")
    @ScalarFunction(value="ceiling", alias={"ceil"})
    @SqlType(value="integer")
    public static long ceilingInteger(@SqlType(value="integer") long num) {
        return num;
    }

    @Description(value="round up to nearest integer")
    @ScalarFunction(alias={"ceil"})
    @SqlType(value="bigint")
    public static long ceiling(@SqlType(value="bigint") long num) {
        return num;
    }

    @Description(value="round up to nearest integer")
    @ScalarFunction(alias={"ceil"})
    @SqlType(value="double")
    public static double ceiling(@SqlType(value="double") double num) {
        return Math.ceil(num);
    }

    @Description(value="round up to nearest integer")
    @ScalarFunction(value="ceiling", alias={"ceil"})
    @SqlType(value="real")
    public static long ceilingFloat(@SqlType(value="real") long num) {
        return Float.floatToRawIntBits((float)MathFunctions.ceiling(Float.intBitsToFloat((int)num)));
    }

    @Description(value="round to integer by dropping digits after decimal point")
    @ScalarFunction
    @SqlType(value="double")
    public static double truncate(@SqlType(value="double") double num) {
        return Math.signum(num) * Math.floor(Math.abs(num));
    }

    @Description(value="round to integer by dropping digits after decimal point")
    @ScalarFunction
    @SqlType(value="real")
    public static long truncate(@SqlType(value="real") long num) {
        float numInFloat = Float.intBitsToFloat((int)num);
        return Float.floatToRawIntBits((float)((double)Math.signum(numInFloat) * Math.floor(Math.abs(numInFloat))));
    }

    @Description(value="truncate to double by dropping digits after decimal point")
    @ScalarFunction
    @SqlType(value="double")
    public static double truncate(@SqlType(value="double") double num, @SqlType(value="integer") long decimals) {
        if (Double.isNaN(num) || Double.isInfinite(num)) {
            return num;
        }
        if (decimals == 0L) {
            if (num >= 0.0) {
                return Math.floor(num);
            }
            return Math.ceil(num);
        }
        return BigDecimal.valueOf(num).setScale((int)decimals, 1).doubleValue();
    }

    @Description(value="truncate to float by dropping digits after decimal point")
    @ScalarFunction
    @SqlType(value="real")
    public static long truncate(@SqlType(value="real") long num, @SqlType(value="integer") long decimals) {
        float numBitsToFloats = Float.intBitsToFloat((int)num);
        if (Float.isNaN(numBitsToFloats) || Float.isInfinite(numBitsToFloats)) {
            return num;
        }
        if (decimals == 0L) {
            if (numBitsToFloats >= 0.0f) {
                return Float.floatToRawIntBits((float)Math.floor(numBitsToFloats));
            }
            return Float.floatToRawIntBits((float)Math.ceil(numBitsToFloats));
        }
        return Float.floatToRawIntBits(new BigDecimal(Float.toString(numBitsToFloats)).setScale((int)decimals, 1).floatValue());
    }

    @Description(value="cosine")
    @ScalarFunction
    @SqlType(value="double")
    public static double cos(@SqlType(value="double") double num) {
        return Math.cos(num);
    }

    @Description(value="hyperbolic cosine")
    @ScalarFunction
    @SqlType(value="double")
    public static double cosh(@SqlType(value="double") double num) {
        return Math.cosh(num);
    }

    @Description(value="converts an angle in radians to degrees")
    @ScalarFunction
    @SqlType(value="double")
    public static double degrees(@SqlType(value="double") double radians) {
        return Math.toDegrees(radians);
    }

    @Description(value="Euler's number")
    @ScalarFunction
    @SqlType(value="double")
    public static double e() {
        return Math.E;
    }

    @Description(value="Euler's number raised to the given power")
    @ScalarFunction
    @SqlType(value="double")
    public static double exp(@SqlType(value="double") double num) {
        return Math.exp(num);
    }

    @Description(value="round down to nearest integer")
    @ScalarFunction(value="floor")
    @SqlType(value="tinyint")
    public static long floorTinyint(@SqlType(value="tinyint") long num) {
        return num;
    }

    @Description(value="round down to nearest integer")
    @ScalarFunction(value="floor")
    @SqlType(value="smallint")
    public static long floorSmallint(@SqlType(value="smallint") long num) {
        return num;
    }

    @Description(value="round down to nearest integer")
    @ScalarFunction(value="floor")
    @SqlType(value="integer")
    public static long floorInteger(@SqlType(value="integer") long num) {
        return num;
    }

    @Description(value="round down to nearest integer")
    @ScalarFunction
    @SqlType(value="bigint")
    public static long floor(@SqlType(value="bigint") long num) {
        return num;
    }

    @Description(value="round down to nearest integer")
    @ScalarFunction
    @SqlType(value="double")
    public static double floor(@SqlType(value="double") double num) {
        return Math.floor(num);
    }

    @Description(value="round down to nearest integer")
    @ScalarFunction(value="floor")
    @SqlType(value="real")
    public static long floorFloat(@SqlType(value="real") long num) {
        return Float.floatToRawIntBits((float)MathFunctions.floor(Float.intBitsToFloat((int)num)));
    }

    @Description(value="inverse of Binomial cdf given numberOfTrials, successProbability parameters and p")
    @ScalarFunction
    @SqlType(value="integer")
    public static long inverseBinomialCdf(@SqlType(value="integer") long numberOfTrials, @SqlType(value="double") double successProbability, @SqlType(value="double") double p) {
        Failures.checkCondition(p >= 0.0 && p <= 1.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "p must be in the interval [0, 1]", new Object[0]);
        Failures.checkCondition(successProbability >= 0.0 && successProbability <= 1.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "successProbability must be in the interval [0, 1]", new Object[0]);
        Failures.checkCondition(numberOfTrials > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "numberOfTrials must be greater than 0", new Object[0]);
        BinomialDistribution distribution = new BinomialDistribution(null, (int)numberOfTrials, successProbability);
        return distribution.inverseCumulativeProbability(p);
    }

    @Description(value="natural logarithm")
    @ScalarFunction
    @SqlType(value="double")
    public static double ln(@SqlType(value="double") double num) {
        return Math.log(num);
    }

    @Description(value="logarithm to base 2")
    @ScalarFunction
    @SqlType(value="double")
    public static double log2(@SqlType(value="double") double num) {
        return Math.log(num) / Math.log(2.0);
    }

    @Description(value="logarithm to base 10")
    @ScalarFunction
    @SqlType(value="double")
    public static double log10(@SqlType(value="double") double num) {
        return Math.log10(num);
    }

    @Description(value="remainder of given quotient")
    @ScalarFunction(value="mod")
    @SqlType(value="tinyint")
    public static long modTinyint(@SqlType(value="tinyint") long num1, @SqlType(value="tinyint") long num2) {
        return num1 % num2;
    }

    @Description(value="remainder of given quotient")
    @ScalarFunction(value="mod")
    @SqlType(value="smallint")
    public static long modSmallint(@SqlType(value="smallint") long num1, @SqlType(value="smallint") long num2) {
        return num1 % num2;
    }

    @Description(value="remainder of given quotient")
    @ScalarFunction(value="mod")
    @SqlType(value="integer")
    public static long modInteger(@SqlType(value="integer") long num1, @SqlType(value="integer") long num2) {
        return num1 % num2;
    }

    @Description(value="remainder of given quotient")
    @ScalarFunction
    @SqlType(value="bigint")
    public static long mod(@SqlType(value="bigint") long num1, @SqlType(value="bigint") long num2) {
        return num1 % num2;
    }

    @Description(value="remainder of given quotient")
    @ScalarFunction
    @SqlType(value="double")
    public static double mod(@SqlType(value="double") double num1, @SqlType(value="double") double num2) {
        return num1 % num2;
    }

    private static SqlScalarFunction decimalModFunction() {
        Signature signature = DecimalOperators.modulusSignatureBuilder().kind(FunctionKind.SCALAR).name("mod").build();
        return DecimalOperators.modulusScalarFunction(signature);
    }

    @Description(value="remainder of given quotient")
    @ScalarFunction(value="mod")
    @SqlType(value="real")
    public static long modFloat(@SqlType(value="real") long num1, @SqlType(value="real") long num2) {
        return Float.floatToRawIntBits(Float.intBitsToFloat((int)num1) % Float.intBitsToFloat((int)num2));
    }

    @Description(value="the constant Pi")
    @ScalarFunction
    @SqlType(value="double")
    public static double pi() {
        return Math.PI;
    }

    @Description(value="value raised to the power of exponent")
    @ScalarFunction(alias={"pow"})
    @SqlType(value="double")
    public static double power(@SqlType(value="double") double num, @SqlType(value="double") double exponent) {
        return Math.pow(num, exponent);
    }

    @Description(value="converts an angle in degrees to radians")
    @ScalarFunction
    @SqlType(value="double")
    public static double radians(@SqlType(value="double") double degrees) {
        return Math.toRadians(degrees);
    }

    @Description(value="a pseudo-random value")
    @ScalarFunction(alias={"rand"}, deterministic=false)
    @SqlType(value="double")
    public static double random() {
        return ThreadLocalRandom.current().nextDouble();
    }

    @Description(value="a pseudo-random number between 0 and value (exclusive)")
    @ScalarFunction(value="random", alias={"rand"}, deterministic=false)
    @SqlType(value="tinyint")
    public static long randomTinyint(@SqlType(value="tinyint") long value) {
        Failures.checkCondition(value > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "bound must be positive", new Object[0]);
        return ThreadLocalRandom.current().nextInt((int)value);
    }

    @Description(value="a pseudo-random number between 0 and value (exclusive)")
    @ScalarFunction(value="random", alias={"rand"}, deterministic=false)
    @SqlType(value="smallint")
    public static long randomSmallint(@SqlType(value="smallint") long value) {
        Failures.checkCondition(value > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "bound must be positive", new Object[0]);
        return ThreadLocalRandom.current().nextInt((int)value);
    }

    @Description(value="a pseudo-random number between 0 and value (exclusive)")
    @ScalarFunction(value="random", alias={"rand"}, deterministic=false)
    @SqlType(value="integer")
    public static long randomInteger(@SqlType(value="integer") long value) {
        Failures.checkCondition(value > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "bound must be positive", new Object[0]);
        return ThreadLocalRandom.current().nextInt((int)value);
    }

    @Description(value="a pseudo-random number between 0 and value (exclusive)")
    @ScalarFunction(alias={"rand"}, deterministic=false)
    @SqlType(value="bigint")
    public static long random(@SqlType(value="bigint") long value) {
        Failures.checkCondition(value > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "bound must be positive", new Object[0]);
        return ThreadLocalRandom.current().nextLong(value);
    }

    @Description(value="inverse of normal cdf given a mean, std, and probability")
    @ScalarFunction
    @SqlType(value="double")
    public static double inverseNormalCdf(@SqlType(value="double") double mean, @SqlType(value="double") double sd, @SqlType(value="double") double p) {
        Failures.checkCondition(p > 0.0 && p < 1.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "p must be 0 > p > 1", new Object[0]);
        Failures.checkCondition(sd > 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "sd must be > 0", new Object[0]);
        return mean + sd * 1.4142135623730951 * Erf.erfInv((double)(2.0 * p - 1.0));
    }

    @Description(value="normal cdf given a mean, standard deviation, and value")
    @ScalarFunction
    @SqlType(value="double")
    public static double normalCdf(@SqlType(value="double") double mean, @SqlType(value="double") double standardDeviation, @SqlType(value="double") double value) {
        Failures.checkCondition(standardDeviation > 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "standardDeviation must be > 0", new Object[0]);
        return 0.5 * (1.0 + Erf.erf((double)((value - mean) / (standardDeviation * Math.sqrt(2.0)))));
    }

    @Description(value="inverse of Beta cdf given a, b parameters and probability")
    @ScalarFunction
    @SqlType(value="double")
    public static double inverseBetaCdf(@SqlType(value="double") double a, @SqlType(value="double") double b, @SqlType(value="double") double p) {
        Failures.checkCondition(p >= 0.0 && p <= 1.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "p must be in the interval [0, 1]", new Object[0]);
        Failures.checkCondition(a > 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "a must be > 0", new Object[0]);
        Failures.checkCondition(b > 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "b must be > 0", new Object[0]);
        BetaDistribution distribution = new BetaDistribution(null, a, b, 1.0E-9);
        return distribution.inverseCumulativeProbability(p);
    }

    @Description(value="Beta cdf given the a, b parameters and value")
    @ScalarFunction
    @SqlType(value="double")
    public static double betaCdf(@SqlType(value="double") double a, @SqlType(value="double") double b, @SqlType(value="double") double value) {
        Failures.checkCondition(value >= 0.0 && value <= 1.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "value must be in the interval [0, 1]", new Object[0]);
        Failures.checkCondition(a > 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "a must be > 0", new Object[0]);
        Failures.checkCondition(b > 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "b must be > 0", new Object[0]);
        BetaDistribution distribution = new BetaDistribution(null, a, b, 1.0E-9);
        return distribution.cumulativeProbability(value);
    }

    @Description(value="inverse of ChiSquared cdf given df parameter and probability")
    @ScalarFunction
    @SqlType(value="double")
    public static double inverseChiSquaredCdf(@SqlType(value="double") double df, @SqlType(value="double") double p) {
        Failures.checkCondition(p >= 0.0 && p <= 1.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "p must be in the interval [0, 1]", new Object[0]);
        Failures.checkCondition(df > 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "df must be greater than 0", new Object[0]);
        ChiSquaredDistribution distribution = new ChiSquaredDistribution(null, df, 1.0E-9);
        return distribution.inverseCumulativeProbability(p);
    }

    @Description(value="ChiSquared cdf given the df parameter and value")
    @ScalarFunction
    @SqlType(value="double")
    public static double chiSquaredCdf(@SqlType(value="double") double df, @SqlType(value="double") double value) {
        Failures.checkCondition(value >= 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "value must non-negative", new Object[0]);
        Failures.checkCondition(df > 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "df must be greater than 0", new Object[0]);
        ChiSquaredDistribution distribution = new ChiSquaredDistribution(null, df, 1.0E-9);
        return distribution.cumulativeProbability(value);
    }

    @Description(value="Inverse of Poisson cdf given lambda (mean) parameter and probability")
    @ScalarFunction
    @SqlType(value="integer")
    public static long inversePoissonCdf(@SqlType(value="double") double lambda, @SqlType(value="double") double p) {
        Failures.checkCondition(p >= 0.0 && p < 1.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "p must be in the interval [0, 1)", new Object[0]);
        Failures.checkCondition(lambda > 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "lambda must be greater than 0", new Object[0]);
        PoissonDistribution distribution = new PoissonDistribution(lambda);
        return distribution.inverseCumulativeProbability(p);
    }

    @Description(value="Poisson cdf given the lambda (mean) parameter and value")
    @ScalarFunction
    @SqlType(value="double")
    public static double poissonCdf(@SqlType(value="double") double lambda, @SqlType(value="integer") long value) {
        Failures.checkCondition(value >= 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "value must be a non-negative integer", new Object[0]);
        Failures.checkCondition(lambda > 0.0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "lambda must be greater than 0", new Object[0]);
        PoissonDistribution distribution = new PoissonDistribution(lambda);
        return distribution.cumulativeProbability((int)value);
    }

    @Description(value="round to nearest integer")
    @ScalarFunction(value="round")
    @SqlType(value="tinyint")
    public static long roundTinyint(@SqlType(value="tinyint") long num) {
        return num;
    }

    @Description(value="round to nearest integer")
    @ScalarFunction(value="round")
    @SqlType(value="smallint")
    public static long roundSmallint(@SqlType(value="smallint") long num) {
        return num;
    }

    @Description(value="round to nearest integer")
    @ScalarFunction(value="round")
    @SqlType(value="integer")
    public static long roundInteger(@SqlType(value="integer") long num) {
        return num;
    }

    @Description(value="round to nearest integer")
    @ScalarFunction
    @SqlType(value="bigint")
    public static long round(@SqlType(value="bigint") long num) {
        return num;
    }

    @Description(value="round to nearest integer")
    @ScalarFunction(value="round")
    @SqlType(value="tinyint")
    public static long roundTinyint(@SqlType(value="tinyint") long num, @SqlType(value="integer") long decimals) {
        return num;
    }

    @Description(value="round to nearest integer")
    @ScalarFunction(value="round")
    @SqlType(value="smallint")
    public static long roundSmallint(@SqlType(value="smallint") long num, @SqlType(value="integer") long decimals) {
        return num;
    }

    @Description(value="round to nearest integer")
    @ScalarFunction(value="round")
    @SqlType(value="integer")
    public static long roundInteger(@SqlType(value="integer") long num, @SqlType(value="integer") long decimals) {
        return num;
    }

    @Description(value="round to nearest integer")
    @ScalarFunction
    @SqlType(value="bigint")
    public static long round(@SqlType(value="bigint") long num, @SqlType(value="integer") long decimals) {
        return num;
    }

    @Description(value="round to nearest integer")
    @ScalarFunction
    @SqlType(value="double")
    public static double round(@SqlType(value="double") double num) {
        return MathFunctions.round(num, 0L);
    }

    @Description(value="round to given number of decimal places")
    @ScalarFunction(value="round")
    @SqlType(value="real")
    public static long roundFloat(@SqlType(value="real") long num) {
        return MathFunctions.roundFloat(num, 0L);
    }

    @Description(value="round to given number of decimal places")
    @ScalarFunction
    @SqlType(value="double")
    public static double round(@SqlType(value="double") double num, @SqlType(value="integer") long decimals) {
        if (Double.isNaN(num) || Double.isInfinite(num)) {
            return num;
        }
        double factor = Math.pow(10.0, decimals);
        if (num < 0.0) {
            return -((double)Math.round(-num * factor) / factor);
        }
        return (double)Math.round(num * factor) / factor;
    }

    @Description(value="round to given number of decimal places")
    @ScalarFunction(value="round")
    @SqlType(value="real")
    public static long roundFloat(@SqlType(value="real") long num, @SqlType(value="integer") long decimals) {
        float numInFloat = Float.intBitsToFloat((int)num);
        if (Float.isNaN(numInFloat) || Float.isInfinite(numInFloat)) {
            return num;
        }
        double factor = Math.pow(10.0, decimals);
        if (numInFloat < 0.0f) {
            return Float.floatToRawIntBits((float)(-((double)Math.round((double)(-numInFloat) * factor) / factor)));
        }
        return Float.floatToRawIntBits((float)((double)Math.round((double)numInFloat * factor) / factor));
    }

    @ScalarFunction
    @SqlType(value="bigint")
    public static long sign(@SqlType(value="bigint") long num) {
        return (long)Math.signum(num);
    }

    @Description(value="signum")
    @ScalarFunction(value="sign")
    @SqlType(value="integer")
    public static long signInteger(@SqlType(value="integer") long num) {
        return (long)Math.signum(num);
    }

    @Description(value="signum")
    @ScalarFunction(value="sign")
    @SqlType(value="smallint")
    public static long signSmallint(@SqlType(value="smallint") long num) {
        return (long)Math.signum(num);
    }

    @Description(value="signum")
    @ScalarFunction(value="sign")
    @SqlType(value="tinyint")
    public static long signTinyint(@SqlType(value="tinyint") long num) {
        return (long)Math.signum(num);
    }

    @Description(value="signum")
    @ScalarFunction
    @SqlType(value="double")
    public static double sign(@SqlType(value="double") double num) {
        return Math.signum(num);
    }

    @Description(value="signum")
    @ScalarFunction(value="sign")
    @SqlType(value="real")
    public static long signFloat(@SqlType(value="real") long num) {
        return Float.floatToRawIntBits(Math.signum(Float.intBitsToFloat((int)num)));
    }

    @Description(value="sine")
    @ScalarFunction
    @SqlType(value="double")
    public static double sin(@SqlType(value="double") double num) {
        return Math.sin(num);
    }

    @Description(value="square root")
    @ScalarFunction
    @SqlType(value="double")
    public static double sqrt(@SqlType(value="double") double num) {
        return Math.sqrt(num);
    }

    @Description(value="tangent")
    @ScalarFunction
    @SqlType(value="double")
    public static double tan(@SqlType(value="double") double num) {
        return Math.tan(num);
    }

    @Description(value="hyperbolic tangent")
    @ScalarFunction
    @SqlType(value="double")
    public static double tanh(@SqlType(value="double") double num) {
        return Math.tanh(num);
    }

    @Description(value="test if value is not-a-number")
    @ScalarFunction(value="is_nan")
    @SqlType(value="boolean")
    public static boolean isNaN(@SqlType(value="double") double num) {
        return Double.isNaN(num);
    }

    @Description(value="test if value is finite")
    @ScalarFunction
    @SqlType(value="boolean")
    public static boolean isFinite(@SqlType(value="double") double num) {
        return Doubles.isFinite((double)num);
    }

    @Description(value="test if value is infinite")
    @ScalarFunction
    @SqlType(value="boolean")
    public static boolean isInfinite(@SqlType(value="double") double num) {
        return Double.isInfinite(num);
    }

    @Description(value="constant representing not-a-number")
    @ScalarFunction(value="nan")
    @SqlType(value="double")
    public static double NaN() {
        return Double.NaN;
    }

    @Description(value="Infinity")
    @ScalarFunction
    @SqlType(value="double")
    public static double infinity() {
        return Double.POSITIVE_INFINITY;
    }

    @Description(value="convert a number to a string in the given base")
    @ScalarFunction
    @SqlType(value="varchar(64)")
    public static Slice toBase(@SqlType(value="bigint") long value, @SqlType(value="bigint") long radix) {
        MathFunctions.checkRadix(radix);
        return Slices.utf8Slice((String)Long.toString(value, (int)radix));
    }

    @Description(value="convert a string in the given base to a number")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="bigint")
    public static long fromBase(@SqlType(value="varchar(x)") Slice value, @SqlType(value="bigint") long radix) {
        MathFunctions.checkRadix(radix);
        try {
            return Long.parseLong(value.toStringUtf8(), (int)radix);
        }
        catch (NumberFormatException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, String.format("Not a valid base-%d number: %s", radix, value.toStringUtf8()), (Throwable)e);
        }
    }

    private static void checkRadix(long radix) {
        Failures.checkCondition(radix >= 2L && radix <= 36L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Radix must be between %d and %d", 2, 36);
    }

    @Description(value="The bucket number of a value given a lower and upper bound and the number of buckets")
    @ScalarFunction(value="width_bucket")
    @SqlType(value="bigint")
    public static long widthBucket(@SqlType(value="double") double operand, @SqlType(value="double") double bound1, @SqlType(value="double") double bound2, @SqlType(value="bigint") long bucketCount) {
        Failures.checkCondition(bucketCount > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "bucketCount must be greater than 0", new Object[0]);
        Failures.checkCondition(!MathFunctions.isNaN(operand), (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "operand must not be NaN", new Object[0]);
        Failures.checkCondition(MathFunctions.isFinite(bound1), (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "first bound must be finite", new Object[0]);
        Failures.checkCondition(MathFunctions.isFinite(bound2), (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "second bound must be finite", new Object[0]);
        Failures.checkCondition(bound1 != bound2, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "bounds cannot equal each other", new Object[0]);
        long result = 0L;
        double lower = Math.min(bound1, bound2);
        double upper = Math.max(bound1, bound2);
        if (operand < lower) {
            result = 0L;
        } else if (operand >= upper) {
            try {
                result = Math.addExact(bucketCount, 1L);
            }
            catch (ArithmeticException e) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, String.format("Bucket for value %s is out of range", operand));
            }
        } else {
            result = (long)((double)bucketCount * (operand - lower) / (upper - lower) + 1.0);
        }
        if (bound1 > bound2) {
            result = bucketCount - result + 1L;
        }
        return result;
    }

    @Description(value="The bucket number of a value given an array of bins")
    @ScalarFunction(value="width_bucket")
    @SqlType(value="bigint")
    public static long widthBucket(@SqlType(value="double") double operand, @SqlType(value="array(double)") Block bins) {
        int numberOfBins = bins.getPositionCount();
        Failures.checkCondition(numberOfBins > 0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Bins cannot be an empty array", new Object[0]);
        Failures.checkCondition(!MathFunctions.isNaN(operand), (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Operand cannot be NaN", new Object[0]);
        int lower = 0;
        int upper = numberOfBins;
        while (lower < upper) {
            if (DoubleType.DOUBLE.getDouble(bins, lower) > DoubleType.DOUBLE.getDouble(bins, upper - 1)) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Bin values are not sorted in ascending order");
            }
            int index = (lower + upper) / 2;
            double bin = DoubleType.DOUBLE.getDouble(bins, index);
            if (!MathFunctions.isFinite(bin)) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Bin value must be finite, got " + bin);
            }
            if (operand < bin) {
                upper = index;
                continue;
            }
            lower = index + 1;
        }
        return lower;
    }

    @Description(value="cosine similarity between the given sparse vectors")
    @ScalarFunction
    @SqlNullable
    @SqlType(value="double")
    public static Double cosineSimilarity(@SqlType(value="map(varchar,double)") Block leftMap, @SqlType(value="map(varchar,double)") Block rightMap) {
        Double normLeftMap = MathFunctions.mapL2Norm(leftMap);
        Double normRightMap = MathFunctions.mapL2Norm(rightMap);
        if (normLeftMap == null || normRightMap == null) {
            return null;
        }
        double dotProduct = MathFunctions.mapDotProduct(leftMap, rightMap);
        return dotProduct / (normLeftMap * normRightMap);
    }

    private static double mapDotProduct(Block leftMap, Block rightMap) {
        TypedSet rightMapKeys = new TypedSet((Type)VarcharType.VARCHAR, rightMap.getPositionCount(), "cosine_similarity");
        for (int i = 0; i < rightMap.getPositionCount(); i += 2) {
            rightMapKeys.add(rightMap, i);
        }
        double result = 0.0;
        for (int i = 0; i < leftMap.getPositionCount(); i += 2) {
            int position = rightMapKeys.positionOf(leftMap, i);
            if (position == -1) continue;
            result += DoubleType.DOUBLE.getDouble(leftMap, i + 1) * DoubleType.DOUBLE.getDouble(rightMap, 2 * position + 1);
        }
        return result;
    }

    private static Double mapL2Norm(Block map) {
        double norm = 0.0;
        for (int i = 1; i < map.getPositionCount(); i += 2) {
            if (map.isNull(i)) {
                return null;
            }
            norm += DoubleType.DOUBLE.getDouble(map, i) * DoubleType.DOUBLE.getDouble(map, i);
        }
        return Math.sqrt(norm);
    }

    static {
        MathFunctions.DECIMAL_HALF_UNSCALED_FOR_SCALE[0] = UnscaledDecimal128Arithmetic.unscaledDecimal((long)0L);
        MathFunctions.DECIMAL_ALMOST_HALF_UNSCALED_FOR_SCALE[0] = UnscaledDecimal128Arithmetic.unscaledDecimal((long)0L);
        for (int scale = 1; scale < 38; ++scale) {
            MathFunctions.DECIMAL_HALF_UNSCALED_FOR_SCALE[scale] = UnscaledDecimal128Arithmetic.unscaledDecimal((BigInteger)BigInteger.TEN.pow(scale).divide(BigInteger.valueOf(2L)));
            MathFunctions.DECIMAL_ALMOST_HALF_UNSCALED_FOR_SCALE[scale] = UnscaledDecimal128Arithmetic.unscaledDecimal((BigInteger)BigInteger.TEN.pow(scale).divide(BigInteger.valueOf(2L)).subtract(BigInteger.ONE));
        }
    }

    @Description(value="signum")
    @ScalarFunction(value="sign")
    public static final class Sign {
        private Sign() {
        }

        @LiteralParameters(value={"p", "s"})
        @SqlType(value="decimal(1,0)")
        public static long signDecimalShort(@SqlType(value="decimal(p, s)") long num) {
            return (long)Math.signum(num);
        }

        @LiteralParameters(value={"p", "s"})
        @SqlType(value="decimal(1,0)")
        public static long signDecimalLong(@SqlType(value="decimal(p, s)") Slice num) {
            if (UnscaledDecimal128Arithmetic.isZero((Slice)num)) {
                return 0L;
            }
            if (UnscaledDecimal128Arithmetic.isNegative((Slice)num)) {
                return -1L;
            }
            return 1L;
        }
    }

    @ScalarFunction(value="truncate")
    @Description(value="round to integer by dropping given number of digits after decimal point")
    public static final class TruncateN {
        private TruncateN() {
        }

        @LiteralParameters(value={"p", "s"})
        @SqlType(value="decimal(p, s)")
        public static long truncateShort(@LiteralParameter(value="p") long numPrecision, @LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") long num, @SqlType(value="integer") long roundScale) {
            if (num == 0L || numPrecision - numScale + roundScale <= 0L) {
                return 0L;
            }
            if (roundScale >= numScale) {
                return num;
            }
            long rescaleFactor = Decimals.longTenToNth((int)((int)(numScale - roundScale)));
            long remainder = num % rescaleFactor;
            return num - remainder;
        }

        @LiteralParameters(value={"p", "s"})
        @SqlType(value="decimal(p, s)")
        public static Slice truncateLong(@LiteralParameter(value="p") long numPrecision, @LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num, @SqlType(value="integer") long roundScale) {
            if (numPrecision - numScale + roundScale <= 0L) {
                return UnscaledDecimal128Arithmetic.unscaledDecimal((long)0L);
            }
            if (roundScale >= numScale) {
                return num;
            }
            int rescaleFactor = (int)(numScale - roundScale);
            return UnscaledDecimal128Arithmetic.rescaleTruncate((Slice)UnscaledDecimal128Arithmetic.rescaleTruncate((Slice)num, (int)(-rescaleFactor)), (int)rescaleFactor);
        }
    }

    @ScalarFunction(value="truncate")
    @Description(value="round to integer by dropping digits after decimal point")
    public static final class Truncate {
        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="max(1, p - s)")
        public static long truncateShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") long num) {
            if (num == 0L) {
                return 0L;
            }
            if (numScale == 0L) {
                return num;
            }
            long rescaleFactor = Decimals.longTenToNth((int)((int)numScale));
            return num / rescaleFactor;
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="max(1, p - s)")
        public static Slice truncateLong(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            if (numScale == 0L) {
                return num;
            }
            return UnscaledDecimal128Arithmetic.rescaleTruncate((Slice)num, (int)(-((int)numScale)));
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="max(1, p - s)")
        public static long truncateLongShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            return UnscaledDecimal128Arithmetic.unscaledDecimalToUnscaledLong((Slice)UnscaledDecimal128Arithmetic.rescaleTruncate((Slice)num, (int)(-((int)numScale))));
        }
    }

    @ScalarFunction(value="round")
    @Description(value="round to given number of decimal places")
    public static final class RoundN {
        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp, s)")
        @Constraint(variable="rp", expression="min(38, p + 1)")
        public static long roundNShort(@LiteralParameter(value="p") long numPrecision, @LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") long num, @SqlType(value="integer") long decimals) {
            if (num == 0L || numPrecision - numScale + decimals <= 0L) {
                return 0L;
            }
            if (decimals >= numScale) {
                return num;
            }
            if (num < 0L) {
                return -RoundN.roundNShort(numPrecision, numScale, -num, decimals);
            }
            long rescaleFactor = Decimals.longTenToNth((int)((int)(numScale - decimals)));
            long remainder = num % rescaleFactor;
            int roundUp = remainder >= rescaleFactor / 2L ? 1 : 0;
            return (num / rescaleFactor + (long)roundUp) * rescaleFactor;
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp, s)")
        @Constraint(variable="rp", expression="min(38, p + 1)")
        public static Slice roundNLong(@LiteralParameter(value="s") long numScale, @LiteralParameter(value="rp") long resultPrecision, @SqlType(value="decimal(p, s)") Slice num, @SqlType(value="integer") long decimals) {
            if (decimals >= numScale) {
                return num;
            }
            int rescaleFactor = (int)numScale - (int)decimals;
            try {
                Slice result = UnscaledDecimal128Arithmetic.rescale((Slice)UnscaledDecimal128Arithmetic.rescale((Slice)num, (int)(-rescaleFactor)), (int)rescaleFactor);
                UnscaledDecimal128Arithmetic.throwIfOverflows((Slice)result, (int)((int)resultPrecision));
                return result;
            }
            catch (ArithmeticException e) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.NUMERIC_VALUE_OUT_OF_RANGE, "decimal overflow: " + num, (Throwable)e);
            }
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp, s)")
        @Constraint(variable="rp", expression="min(38, p + 1)")
        public static Slice roundNShortLong(@LiteralParameter(value="s") long numScale, @LiteralParameter(value="rp") long resultPrecision, @SqlType(value="decimal(p, s)") long num, @SqlType(value="integer") long decimals) {
            return RoundN.roundNLong(numScale, resultPrecision, UnscaledDecimal128Arithmetic.unscaledDecimal((long)num), decimals);
        }
    }

    @ScalarFunction(value="round")
    @Description(value="round to nearest integer")
    public static final class Round {
        private Round() {
        }

        @LiteralParameters(value={"p", "s", "rp", "rs"})
        @SqlType(value="decimal(rp, rs)")
        @Constraints(value={@Constraint(variable="rp", expression="min(38, p - s + min(1, s))"), @Constraint(variable="rs", expression="0")})
        public static long roundShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") long num) {
            long remainderBoundary;
            if (num == 0L) {
                return 0L;
            }
            if (numScale == 0L) {
                return num;
            }
            if (num < 0L) {
                return -Round.roundShort(numScale, -num);
            }
            long rescaleFactor = Decimals.longTenToNth((int)((int)numScale));
            long remainder = num % rescaleFactor;
            int roundUp = remainder >= (remainderBoundary = rescaleFactor / 2L) ? 1 : 0;
            return num / rescaleFactor + (long)roundUp;
        }

        @LiteralParameters(value={"p", "s", "rp", "rs"})
        @SqlType(value="decimal(rp, rs)")
        @Constraints(value={@Constraint(variable="rp", expression="min(38, p - s + min(1, s))"), @Constraint(variable="rs", expression="0")})
        public static Slice roundLongLong(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            if (numScale == 0L) {
                return num;
            }
            return UnscaledDecimal128Arithmetic.rescale((Slice)num, (int)(-((int)numScale)));
        }

        @LiteralParameters(value={"p", "s", "rp", "rs"})
        @SqlType(value="decimal(rp, rs)")
        @Constraints(value={@Constraint(variable="rp", expression="min(38, p - s + min(1, s))"), @Constraint(variable="rs", expression="0")})
        public static long roundLongShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            return UnscaledDecimal128Arithmetic.unscaledDecimalToUnscaledLong((Slice)UnscaledDecimal128Arithmetic.rescale((Slice)num, (int)(-((int)numScale))));
        }
    }

    @ScalarFunction(value="floor")
    @Description(value="round down to nearest integer")
    public static final class Floor {
        private Floor() {
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="p - s + min(s, 1)")
        public static long floorShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") long num) {
            long rescaleFactor = Decimals.longTenToNth((int)((int)numScale));
            long increment = num % rescaleFactor < 0L ? -1L : 0L;
            return num / rescaleFactor + increment;
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="p - s + min(s, 1)")
        public static Slice floorLong(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            if (UnscaledDecimal128Arithmetic.isZero((Slice)num)) {
                return num;
            }
            Slice tmp = UnscaledDecimal128Arithmetic.isNegative((Slice)num) ? UnscaledDecimal128Arithmetic.subtract((Slice)num, (Slice)DECIMAL_ALMOST_HALF_UNSCALED_FOR_SCALE[(int)numScale]) : UnscaledDecimal128Arithmetic.subtract((Slice)num, (Slice)DECIMAL_HALF_UNSCALED_FOR_SCALE[(int)numScale]);
            return UnscaledDecimal128Arithmetic.rescale((Slice)tmp, (int)(-((int)numScale)));
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="p - s + min(s, 1)")
        public static long floorLongShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            return UnscaledDecimal128Arithmetic.unscaledDecimalToUnscaledLong((Slice)Floor.floorLong(numScale, num));
        }
    }

    @ScalarFunction(value="ceiling", alias={"ceil"})
    @Description(value="round up to nearest integer")
    public static final class Ceiling {
        private Ceiling() {
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="p - s + min(s, 1)")
        public static long ceilingShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") long num) {
            long rescaleFactor = Decimals.longTenToNth((int)((int)numScale));
            long increment = num % rescaleFactor > 0L ? 1L : 0L;
            return num / rescaleFactor + increment;
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="p - s + min(s, 1)")
        public static Slice ceilingLong(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            Slice tmp = UnscaledDecimal128Arithmetic.isNegative((Slice)num) ? UnscaledDecimal128Arithmetic.add((Slice)num, (Slice)DECIMAL_HALF_UNSCALED_FOR_SCALE[(int)numScale]) : UnscaledDecimal128Arithmetic.add((Slice)num, (Slice)DECIMAL_ALMOST_HALF_UNSCALED_FOR_SCALE[(int)numScale]);
            return UnscaledDecimal128Arithmetic.rescale((Slice)tmp, (int)(-((int)numScale)));
        }

        @LiteralParameters(value={"p", "s", "rp"})
        @SqlType(value="decimal(rp,0)")
        @Constraint(variable="rp", expression="p - s + min(s, 1)")
        public static long ceilingLongShort(@LiteralParameter(value="s") long numScale, @SqlType(value="decimal(p, s)") Slice num) {
            return UnscaledDecimal128Arithmetic.unscaledDecimalToUnscaledLong((Slice)Ceiling.ceilingLong(numScale, num));
        }
    }

    @ScalarFunction(value="log")
    @Description(value="logarithm to given base")
    public static final class LegacyLogFunction {
        private LegacyLogFunction() {
        }

        @SqlType(value="double")
        public static double log(@SqlType(value="double") double number, @SqlType(value="double") double base) {
            return Math.log(number) / Math.log(base);
        }
    }

    @ScalarFunction(value="abs")
    @Description(value="absolute value")
    public static final class Abs {
        private Abs() {
        }

        @LiteralParameters(value={"p", "s"})
        @SqlType(value="decimal(p, s)")
        public static long absShort(@SqlType(value="decimal(p, s)") long arg) {
            return arg > 0L ? arg : -arg;
        }

        @LiteralParameters(value={"p", "s"})
        @SqlType(value="decimal(p, s)")
        public static Slice absLong(@SqlType(value="decimal(p, s)") Slice arg) {
            if (UnscaledDecimal128Arithmetic.isNegative((Slice)arg)) {
                Slice result = UnscaledDecimal128Arithmetic.unscaledDecimal((Slice)arg);
                UnscaledDecimal128Arithmetic.negate((Slice)result);
                return result;
            }
            return arg;
        }
    }
}

