/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.execution;

import com.facebook.airlift.log.Logger;
import com.facebook.airlift.stats.Distribution;
import com.facebook.presto.execution.BasicStageExecutionStats;
import com.facebook.presto.execution.ExecutionFailureInfo;
import com.facebook.presto.execution.StageExecutionId;
import com.facebook.presto.execution.StageExecutionInfo;
import com.facebook.presto.execution.StageExecutionState;
import com.facebook.presto.execution.StateMachine;
import com.facebook.presto.execution.TaskInfo;
import com.facebook.presto.execution.TaskState;
import com.facebook.presto.execution.scheduler.SplitSchedulerStats;
import com.facebook.presto.operator.BlockedReason;
import com.facebook.presto.operator.TaskStats;
import com.facebook.presto.util.Failures;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import io.airlift.units.DataSize;
import io.airlift.units.Duration;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalDouble;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import javax.annotation.concurrent.ThreadSafe;
import org.joda.time.DateTime;

@ThreadSafe
public class StageExecutionStateMachine {
    private static final Logger log = Logger.get(StageExecutionStateMachine.class);
    private final StageExecutionId stageExecutionId;
    private final SplitSchedulerStats scheduledStats;
    private final boolean containsTableScans;
    private final StateMachine<StageExecutionState> state;
    private final StateMachine<Optional<StageExecutionInfo>> finalInfo;
    private final AtomicReference<ExecutionFailureInfo> failureCause = new AtomicReference();
    private final AtomicReference<DateTime> schedulingComplete = new AtomicReference();
    private final Distribution getSplitDistribution = new Distribution();
    private final AtomicLong peakUserMemory = new AtomicLong();
    private final AtomicLong peakNodeTotalMemory = new AtomicLong();
    private final AtomicLong currentUserMemory = new AtomicLong();
    private final AtomicLong currentTotalMemory = new AtomicLong();

    public StageExecutionStateMachine(StageExecutionId stageExecutionId, ExecutorService executor, SplitSchedulerStats schedulerStats, boolean containsTableScans) {
        this.stageExecutionId = Objects.requireNonNull(stageExecutionId, "stageId is null");
        this.scheduledStats = Objects.requireNonNull(schedulerStats, "schedulerStats is null");
        this.containsTableScans = containsTableScans;
        this.state = new StateMachine<StageExecutionState>("stage execution " + stageExecutionId, executor, StageExecutionState.PLANNED, StageExecutionState.TERMINAL_STAGE_STATES);
        this.state.addStateChangeListener((T state) -> log.debug("Stage Execution %s is %s", new Object[]{stageExecutionId, state}));
        this.finalInfo = new StateMachine("final stage execution " + stageExecutionId, executor, Optional.empty());
    }

    public StageExecutionId getStageExecutionId() {
        return this.stageExecutionId;
    }

    public StageExecutionState getState() {
        return this.state.get();
    }

    public void addStateChangeListener(StateMachine.StateChangeListener<StageExecutionState> stateChangeListener) {
        this.state.addStateChangeListener(stateChangeListener);
    }

    public synchronized boolean transitionToScheduling() {
        return this.state.compareAndSet(StageExecutionState.PLANNED, StageExecutionState.SCHEDULING);
    }

    public synchronized boolean transitionToFinishedTaskScheduling() {
        return this.state.compareAndSet(StageExecutionState.SCHEDULING, StageExecutionState.FINISHED_TASK_SCHEDULING);
    }

    public synchronized boolean transitionToSchedulingSplits() {
        return this.state.setIf(StageExecutionState.SCHEDULING_SPLITS, currentState -> currentState == StageExecutionState.PLANNED || currentState == StageExecutionState.SCHEDULING || currentState == StageExecutionState.FINISHED_TASK_SCHEDULING);
    }

    public synchronized boolean transitionToScheduled() {
        this.schedulingComplete.compareAndSet(null, DateTime.now());
        return this.state.setIf(StageExecutionState.SCHEDULED, currentState -> currentState == StageExecutionState.PLANNED || currentState == StageExecutionState.SCHEDULING || currentState == StageExecutionState.FINISHED_TASK_SCHEDULING || currentState == StageExecutionState.SCHEDULING_SPLITS);
    }

    public boolean transitionToRunning() {
        return this.state.setIf(StageExecutionState.RUNNING, currentState -> currentState != StageExecutionState.RUNNING && !currentState.isDone());
    }

    public boolean transitionToFinished() {
        return this.state.setIf(StageExecutionState.FINISHED, currentState -> !currentState.isDone());
    }

    public boolean transitionToCanceled() {
        return this.state.setIf(StageExecutionState.CANCELED, currentState -> !currentState.isDone());
    }

    public boolean transitionToAborted() {
        return this.state.setIf(StageExecutionState.ABORTED, currentState -> !currentState.isDone());
    }

    public boolean transitionToFailed(Throwable throwable) {
        Objects.requireNonNull(throwable, "throwable is null");
        this.failureCause.compareAndSet(null, Failures.toFailure(throwable));
        boolean failed = this.state.setIf(StageExecutionState.FAILED, currentState -> !currentState.isDone());
        if (failed) {
            log.error(throwable, "Stage execution %s failed", new Object[]{this.stageExecutionId});
        } else {
            log.debug(throwable, "Failure after stage execution %s finished", new Object[]{this.stageExecutionId});
        }
        return failed;
    }

    public void addFinalStageInfoListener(StateMachine.StateChangeListener<StageExecutionInfo> finalStatusListener) {
        AtomicBoolean done = new AtomicBoolean();
        StateMachine.StateChangeListener<Optional> fireOnceStateChangeListener = finalStageInfo -> {
            if (finalStageInfo.isPresent() && done.compareAndSet(false, true)) {
                finalStatusListener.stateChanged((StageExecutionInfo)finalStageInfo.get());
            }
        };
        this.finalInfo.addStateChangeListener(fireOnceStateChangeListener);
    }

    public void setAllTasksFinal(Iterable<TaskInfo> finalTaskInfos, int totalLifespans) {
        Objects.requireNonNull(finalTaskInfos, "finalTaskInfos is null");
        Preconditions.checkState((boolean)this.state.get().isDone());
        StageExecutionInfo stageInfo = this.getStageExecutionInfo(() -> finalTaskInfos, totalLifespans, totalLifespans);
        Preconditions.checkArgument((boolean)stageInfo.isFinal(), (Object)"finalTaskInfos are not all done");
        this.finalInfo.compareAndSet(Optional.empty(), Optional.of(stageInfo));
    }

    public long getUserMemoryReservation() {
        return this.currentUserMemory.get();
    }

    public long getTotalMemoryReservation() {
        return this.currentTotalMemory.get();
    }

    public void updateMemoryUsage(long deltaUserMemoryInBytes, long deltaTotalMemoryInBytes, long peakNodeTotalMemoryReservationInBytes) {
        this.currentTotalMemory.addAndGet(deltaTotalMemoryInBytes);
        this.currentUserMemory.addAndGet(deltaUserMemoryInBytes);
        this.peakUserMemory.updateAndGet(currentPeakValue -> Math.max(this.currentUserMemory.get(), currentPeakValue));
        this.peakNodeTotalMemory.accumulateAndGet(peakNodeTotalMemoryReservationInBytes, Math::max);
    }

    public BasicStageExecutionStats getBasicStageStats(Supplier<Iterable<TaskInfo>> taskInfosSupplier) {
        Optional<StageExecutionInfo> finalStageInfo = this.finalInfo.get();
        if (finalStageInfo.isPresent()) {
            return finalStageInfo.get().getStats().toBasicStageStats(finalStageInfo.get().getState());
        }
        StageExecutionState state = this.state.get();
        boolean isScheduled = state == StageExecutionState.RUNNING || state.isDone();
        ImmutableList taskInfos = ImmutableList.copyOf(taskInfosSupplier.get());
        int totalDrivers = 0;
        int queuedDrivers = 0;
        int runningDrivers = 0;
        int completedDrivers = 0;
        long cumulativeUserMemory = 0L;
        long userMemoryReservation = 0L;
        long totalMemoryReservation = 0L;
        long totalScheduledTime = 0L;
        long totalCpuTime = 0L;
        long rawInputDataSize = 0L;
        long rawInputPositions = 0L;
        boolean fullyBlocked = true;
        HashSet<BlockedReason> blockedReasons = new HashSet<BlockedReason>();
        long totalAllocation = 0L;
        for (TaskInfo taskInfo : taskInfos) {
            TaskState taskState = taskInfo.getTaskStatus().getState();
            TaskStats taskStats = taskInfo.getStats();
            totalDrivers += taskStats.getTotalDrivers();
            queuedDrivers += taskStats.getQueuedDrivers();
            runningDrivers += taskStats.getRunningDrivers();
            completedDrivers += taskStats.getCompletedDrivers();
            cumulativeUserMemory = (long)((double)cumulativeUserMemory + taskStats.getCumulativeUserMemory());
            long taskUserMemory = taskStats.getUserMemoryReservationInBytes();
            long taskSystemMemory = taskStats.getSystemMemoryReservationInBytes();
            userMemoryReservation += taskUserMemory;
            totalMemoryReservation += taskUserMemory + taskSystemMemory;
            totalScheduledTime += taskStats.getTotalScheduledTimeInNanos();
            totalCpuTime += taskStats.getTotalCpuTimeInNanos();
            if (!taskState.isDone()) {
                fullyBlocked &= taskStats.isFullyBlocked();
                blockedReasons.addAll(taskStats.getBlockedReasons());
            }
            totalAllocation += taskStats.getTotalAllocationInBytes();
            if (!this.containsTableScans) continue;
            rawInputDataSize += taskStats.getRawInputDataSizeInBytes();
            rawInputPositions += taskStats.getRawInputPositions();
        }
        OptionalDouble progressPercentage = OptionalDouble.empty();
        if (isScheduled && totalDrivers != 0) {
            progressPercentage = OptionalDouble.of(Math.min(100.0, (double)completedDrivers * 100.0 / (double)totalDrivers));
        }
        return new BasicStageExecutionStats(isScheduled, totalDrivers, queuedDrivers, runningDrivers, completedDrivers, DataSize.succinctBytes((long)rawInputDataSize), rawInputPositions, cumulativeUserMemory, DataSize.succinctBytes((long)userMemoryReservation), DataSize.succinctBytes((long)totalMemoryReservation), Duration.succinctNanos((long)totalCpuTime), Duration.succinctNanos((long)totalScheduledTime), fullyBlocked, blockedReasons, DataSize.succinctBytes((long)totalAllocation), progressPercentage);
    }

    public StageExecutionInfo getStageExecutionInfo(Supplier<Iterable<TaskInfo>> taskInfosSupplier, int finishedLifespans, int totalLifespans) {
        Optional<StageExecutionInfo> finalStageInfo = this.finalInfo.get();
        if (finalStageInfo.isPresent()) {
            return finalStageInfo.get();
        }
        StageExecutionState state = this.state.get();
        ImmutableList taskInfos = ImmutableList.copyOf(taskInfosSupplier.get());
        Optional<ExecutionFailureInfo> failureInfo = Optional.empty();
        if (state == StageExecutionState.FAILED) {
            failureInfo = Optional.of(this.failureCause.get());
        }
        return StageExecutionInfo.create(this.stageExecutionId, state, failureInfo, (List<TaskInfo>)taskInfos, this.schedulingComplete.get(), this.getSplitDistribution.snapshot(), DataSize.succinctBytes((long)this.peakUserMemory.get()), DataSize.succinctBytes((long)this.peakNodeTotalMemory.get()), finishedLifespans, totalLifespans);
    }

    public void recordGetSplitTime(long startNanos) {
        long elapsedNanos = System.nanoTime() - startNanos;
        this.getSplitDistribution.add(elapsedNanos);
        this.scheduledStats.getGetSplitTime().add((double)elapsedNanos, TimeUnit.NANOSECONDS);
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("stageExecutionId", (Object)this.stageExecutionId).add("state", this.state).toString();
    }
}

