/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.operator.scalar;

import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.block.Block;
import com.facebook.presto.spi.block.BlockBuilder;
import com.facebook.presto.spi.function.Description;
import com.facebook.presto.spi.function.LiteralParameters;
import com.facebook.presto.spi.function.OperatorType;
import com.facebook.presto.spi.function.ScalarFunction;
import com.facebook.presto.spi.function.ScalarOperator;
import com.facebook.presto.spi.function.SqlNullable;
import com.facebook.presto.spi.function.SqlType;
import com.facebook.presto.spi.type.Chars;
import com.facebook.presto.spi.type.VarcharType;
import com.facebook.presto.type.Constraint;
import com.facebook.presto.type.LiteralParameter;
import com.facebook.presto.util.Failures;
import com.google.common.primitives.Ints;
import io.airlift.slice.InvalidCodePointException;
import io.airlift.slice.InvalidUtf8Exception;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceUtf8;
import io.airlift.slice.Slices;
import java.text.Normalizer;
import java.util.OptionalInt;

public final class StringFunctions {
    private StringFunctions() {
    }

    @Description(value="convert Unicode code point to a string")
    @ScalarFunction
    @SqlType(value="varchar(1)")
    public static Slice chr(@SqlType(value="bigint") long codepoint) {
        try {
            return SliceUtf8.codePointToUtf8((int)Ints.saturatedCast((long)codepoint));
        }
        catch (InvalidCodePointException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Not a valid Unicode code point: " + codepoint, (Throwable)e);
        }
    }

    @Description(value="returns Unicode code point of a single character string")
    @ScalarFunction(value="codepoint")
    @SqlType(value="integer")
    public static long codepoint(@SqlType(value="varchar(1)") Slice slice) {
        Failures.checkCondition(SliceUtf8.countCodePoints((Slice)slice) == 1, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Input string must be a single character string", new Object[0]);
        return SliceUtf8.getCodePointAt((Slice)slice, (int)0);
    }

    @Description(value="count of code points of the given string")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="bigint")
    public static long length(@SqlType(value="varchar(x)") Slice slice) {
        return SliceUtf8.countCodePoints((Slice)slice);
    }

    @Description(value="count of code points of the given string")
    @ScalarFunction(value="length")
    @LiteralParameters(value={"x"})
    @SqlType(value="bigint")
    public static long charLength(@LiteralParameter(value="x") long x, @SqlType(value="char(x)") Slice slice) {
        return x;
    }

    @Description(value="greedily removes occurrences of a pattern in a string")
    @ScalarFunction
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="varchar(x)")
    public static Slice replace(@SqlType(value="varchar(x)") Slice str, @SqlType(value="varchar(y)") Slice search) {
        return StringFunctions.replace(str, search, Slices.EMPTY_SLICE);
    }

    @Description(value="greedily replaces occurrences of a pattern with a string")
    @ScalarFunction
    @LiteralParameters(value={"x", "y", "z", "u"})
    @Constraint(variable="u", expression="min(2147483647, x + z * (x + 1))")
    @SqlType(value="varchar(u)")
    public static Slice replace(@SqlType(value="varchar(x)") Slice str, @SqlType(value="varchar(y)") Slice search, @SqlType(value="varchar(z)") Slice replace) {
        if (search.length() == 0) {
            int codePointLength;
            Slice buffer = Slices.allocate((int)((SliceUtf8.countCodePoints((Slice)str) + 1) * replace.length() + str.length()));
            buffer.setBytes(0, replace);
            int indexBuffer = replace.length();
            for (int index = 0; index < str.length(); index += codePointLength) {
                codePointLength = SliceUtf8.lengthOfCodePointSafe((Slice)str, (int)index);
                buffer.setBytes(indexBuffer, str, index, codePointLength);
                buffer.setBytes(indexBuffer += codePointLength, replace);
                indexBuffer += replace.length();
            }
            return buffer;
        }
        Slice buffer = Slices.allocate((int)str.length());
        int index = 0;
        int indexBuffer = 0;
        while (index < str.length()) {
            int bytesToCopy;
            int matchIndex = str.indexOf(search, index);
            if (matchIndex < 0) {
                bytesToCopy = str.length() - index;
                buffer = Slices.ensureSize((Slice)buffer, (int)(indexBuffer + bytesToCopy));
                buffer.setBytes(indexBuffer, str, index, bytesToCopy);
                indexBuffer += bytesToCopy;
                break;
            }
            bytesToCopy = matchIndex - index;
            buffer = Slices.ensureSize((Slice)buffer, (int)(indexBuffer + bytesToCopy + replace.length()));
            if (bytesToCopy > 0) {
                buffer.setBytes(indexBuffer, str, index, bytesToCopy);
                indexBuffer += bytesToCopy;
            }
            if (replace.length() > 0) {
                buffer.setBytes(indexBuffer, replace);
                indexBuffer += replace.length();
            }
            index = matchIndex + search.length();
        }
        return buffer.slice(0, indexBuffer);
    }

    @Description(value="reverse all code points in a given string")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice reverse(@SqlType(value="varchar(x)") Slice slice) {
        return SliceUtf8.reverse((Slice)slice);
    }

    @Description(value="returns index of first occurrence of a substring (or 0 if not found)")
    @ScalarFunction(value="strpos")
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="bigint")
    public static long stringPosition(@SqlType(value="varchar(x)") Slice string, @SqlType(value="varchar(y)") Slice substring) {
        if (substring.length() == 0) {
            return 1L;
        }
        int index = string.indexOf(substring);
        if (index < 0) {
            return 0L;
        }
        return SliceUtf8.countCodePoints((Slice)string, (int)0, (int)index) + 1;
    }

    @Description(value="suffix starting at given index")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice substr(@SqlType(value="varchar(x)") Slice utf8, @SqlType(value="bigint") long start) {
        if (start == 0L || utf8.length() == 0) {
            return Slices.EMPTY_SLICE;
        }
        int startCodePoint = Ints.saturatedCast((long)start);
        if (startCodePoint > 0) {
            int indexStart = SliceUtf8.offsetOfCodePoint((Slice)utf8, (int)(startCodePoint - 1));
            if (indexStart < 0) {
                return Slices.EMPTY_SLICE;
            }
            int indexEnd = utf8.length();
            return utf8.slice(indexStart, indexEnd - indexStart);
        }
        int codePoints = SliceUtf8.countCodePoints((Slice)utf8);
        if ((startCodePoint += codePoints) < 0) {
            return Slices.EMPTY_SLICE;
        }
        int indexStart = SliceUtf8.offsetOfCodePoint((Slice)utf8, (int)startCodePoint);
        int indexEnd = utf8.length();
        return utf8.slice(indexStart, indexEnd - indexStart);
    }

    @Description(value="suffix starting at given index")
    @ScalarFunction(value="substr")
    @LiteralParameters(value={"x"})
    @SqlType(value="char(x)")
    public static Slice charSubstr(@SqlType(value="char(x)") Slice utf8, @SqlType(value="bigint") long start) {
        return StringFunctions.substr(utf8, start);
    }

    @Description(value="substring of given length starting at an index")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice substr(@SqlType(value="varchar(x)") Slice utf8, @SqlType(value="bigint") long start, @SqlType(value="bigint") long length) {
        if (start == 0L || length <= 0L || utf8.length() == 0) {
            return Slices.EMPTY_SLICE;
        }
        int startCodePoint = Ints.saturatedCast((long)start);
        int lengthCodePoints = Ints.saturatedCast((long)length);
        if (startCodePoint > 0) {
            int indexStart = SliceUtf8.offsetOfCodePoint((Slice)utf8, (int)(startCodePoint - 1));
            if (indexStart < 0) {
                return Slices.EMPTY_SLICE;
            }
            int indexEnd = SliceUtf8.offsetOfCodePoint((Slice)utf8, (int)indexStart, (int)lengthCodePoints);
            if (indexEnd < 0) {
                indexEnd = utf8.length();
            }
            return utf8.slice(indexStart, indexEnd - indexStart);
        }
        int codePoints = SliceUtf8.countCodePoints((Slice)utf8);
        if ((startCodePoint += codePoints) < 0) {
            return Slices.EMPTY_SLICE;
        }
        int indexStart = SliceUtf8.offsetOfCodePoint((Slice)utf8, (int)startCodePoint);
        int indexEnd = startCodePoint + lengthCodePoints < codePoints ? SliceUtf8.offsetOfCodePoint((Slice)utf8, (int)indexStart, (int)lengthCodePoints) : utf8.length();
        return utf8.slice(indexStart, indexEnd - indexStart);
    }

    @Description(value="substring of given length starting at an index")
    @ScalarFunction(value="substr")
    @LiteralParameters(value={"x"})
    @SqlType(value="char(x)")
    public static Slice charSubstr(@SqlType(value="char(x)") Slice utf8, @SqlType(value="bigint") long start, @SqlType(value="bigint") long length) {
        return Chars.trimTrailingSpaces((Slice)StringFunctions.substr(utf8, start, length));
    }

    @ScalarFunction
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="array(varchar(x))")
    public static Block split(@SqlType(value="varchar(x)") Slice string, @SqlType(value="varchar(y)") Slice delimiter) {
        return StringFunctions.split(string, delimiter, string.length() + 1);
    }

    @ScalarFunction
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="array(varchar(x))")
    public static Block split(@SqlType(value="varchar(x)") Slice string, @SqlType(value="varchar(y)") Slice delimiter, @SqlType(value="bigint") long limit) {
        int splitIndex;
        Failures.checkCondition(limit > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Limit must be positive", new Object[0]);
        Failures.checkCondition(limit <= Integer.MAX_VALUE, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Limit is too large", new Object[0]);
        BlockBuilder parts = VarcharType.VARCHAR.createBlockBuilder(null, 1, string.length());
        if (limit == 1L) {
            VarcharType.VARCHAR.writeSlice(parts, string);
            return parts.build();
        }
        int index = 0;
        while (index < string.length() && (splitIndex = string.indexOf(delimiter, index)) >= 0) {
            if (delimiter.length() == 0) {
                ++splitIndex;
            }
            VarcharType.VARCHAR.writeSlice(parts, string, index, splitIndex - index);
            index = splitIndex + delimiter.length();
            if ((long)parts.getPositionCount() != limit - 1L) continue;
            break;
        }
        VarcharType.VARCHAR.writeSlice(parts, string, index, string.length() - index);
        return parts.build();
    }

    @SqlNullable
    @Description(value="splits a string by a delimiter and returns the specified field (counting from one)")
    @ScalarFunction
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="varchar(x)")
    public static Slice splitPart(@SqlType(value="varchar(x)") Slice string, @SqlType(value="varchar(y)") Slice delimiter, @SqlType(value="bigint") long index) {
        int matchIndex;
        Failures.checkCondition(index > 0L, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Index must be greater than zero", new Object[0]);
        if (delimiter.length() == 0) {
            int startCodePoint = Math.toIntExact(index);
            int indexStart = SliceUtf8.offsetOfCodePoint((Slice)string, (int)(startCodePoint - 1));
            if (indexStart < 0) {
                return null;
            }
            int length = SliceUtf8.lengthOfCodePoint((Slice)string, (int)indexStart);
            if (indexStart + length > string.length()) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid UTF-8 encoding");
            }
            return string.slice(indexStart, length);
        }
        int matchCount = 0;
        int previousIndex = 0;
        while (previousIndex < string.length() && (matchIndex = string.indexOf(delimiter, previousIndex)) >= 0) {
            if ((long)(++matchCount) == index) {
                return string.slice(previousIndex, matchIndex - previousIndex);
            }
            previousIndex = matchIndex + delimiter.length();
        }
        if ((long)matchCount == index - 1L) {
            return string.slice(previousIndex, string.length() - previousIndex);
        }
        return null;
    }

    @Description(value="removes whitespace from the beginning of a string")
    @ScalarFunction(value="ltrim")
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice leftTrim(@SqlType(value="varchar(x)") Slice slice) {
        return SliceUtf8.leftTrim((Slice)slice);
    }

    @Description(value="removes whitespace from the beginning of a string")
    @ScalarFunction(value="ltrim")
    @LiteralParameters(value={"x"})
    @SqlType(value="char(x)")
    public static Slice charLeftTrim(@SqlType(value="char(x)") Slice slice) {
        return SliceUtf8.leftTrim((Slice)slice);
    }

    @Description(value="removes whitespace from the end of a string")
    @ScalarFunction(value="rtrim")
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice rightTrim(@SqlType(value="varchar(x)") Slice slice) {
        return SliceUtf8.rightTrim((Slice)slice);
    }

    @Description(value="removes whitespace from the end of a string")
    @ScalarFunction(value="rtrim")
    @LiteralParameters(value={"x"})
    @SqlType(value="char(x)")
    public static Slice charRightTrim(@SqlType(value="char(x)") Slice slice) {
        return StringFunctions.rightTrim(slice);
    }

    @Description(value="removes whitespace from the beginning and end of a string")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice trim(@SqlType(value="varchar(x)") Slice slice) {
        return SliceUtf8.trim((Slice)slice);
    }

    @Description(value="removes whitespace from the beginning and end of a string")
    @ScalarFunction(value="trim")
    @LiteralParameters(value={"x"})
    @SqlType(value="char(x)")
    public static Slice charTrim(@SqlType(value="char(x)") Slice slice) {
        return StringFunctions.trim(slice);
    }

    @Description(value="remove the longest string containing only given characters from the beginning of a string")
    @ScalarFunction(value="ltrim")
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice leftTrim(@SqlType(value="varchar(x)") Slice slice, @SqlType(value="CodePoints") int[] codePointsToTrim) {
        return SliceUtf8.leftTrim((Slice)slice, (int[])codePointsToTrim);
    }

    @Description(value="remove the longest string containing only given characters from the beginning of a string")
    @ScalarFunction(value="ltrim")
    @LiteralParameters(value={"x"})
    @SqlType(value="char(x)")
    public static Slice charLeftTrim(@SqlType(value="char(x)") Slice slice, @SqlType(value="CodePoints") int[] codePointsToTrim) {
        return StringFunctions.leftTrim(slice, codePointsToTrim);
    }

    @Description(value="remove the longest string containing only given characters from the end of a string")
    @ScalarFunction(value="rtrim")
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice rightTrim(@SqlType(value="varchar(x)") Slice slice, @SqlType(value="CodePoints") int[] codePointsToTrim) {
        return SliceUtf8.rightTrim((Slice)slice, (int[])codePointsToTrim);
    }

    @Description(value="remove the longest string containing only given characters from the end of a string")
    @ScalarFunction(value="rtrim")
    @LiteralParameters(value={"x"})
    @SqlType(value="char(x)")
    public static Slice charRightTrim(@SqlType(value="char(x)") Slice slice, @SqlType(value="CodePoints") int[] codePointsToTrim) {
        return Chars.trimTrailingSpaces((Slice)StringFunctions.rightTrim(slice, codePointsToTrim));
    }

    @Description(value="remove the longest string containing only given characters from the beginning and end of a string")
    @ScalarFunction(value="trim")
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice trim(@SqlType(value="varchar(x)") Slice slice, @SqlType(value="CodePoints") int[] codePointsToTrim) {
        return SliceUtf8.trim((Slice)slice, (int[])codePointsToTrim);
    }

    @Description(value="remove the longest string containing only given characters from the beginning and end of a string")
    @ScalarFunction(value="trim")
    @LiteralParameters(value={"x"})
    @SqlType(value="char(x)")
    public static Slice charTrim(@SqlType(value="char(x)") Slice slice, @SqlType(value="CodePoints") int[] codePointsToTrim) {
        return Chars.trimTrailingSpaces((Slice)StringFunctions.trim(slice, codePointsToTrim));
    }

    @ScalarOperator(value=OperatorType.CAST)
    @LiteralParameters(value={"x"})
    @SqlType(value="CodePoints")
    public static int[] castVarcharToCodePoints(@SqlType(value="varchar(x)") Slice slice) {
        return StringFunctions.castToCodePoints(slice);
    }

    @ScalarOperator(value=OperatorType.CAST)
    @SqlType(value="CodePoints")
    @LiteralParameters(value={"x"})
    public static int[] castCharToCodePoints(@LiteralParameter(value="x") Long charLength, @SqlType(value="char(x)") Slice slice) {
        return StringFunctions.castToCodePoints(Chars.padSpaces((Slice)slice, (int)charLength.intValue()));
    }

    private static int[] castToCodePoints(Slice slice) {
        int[] codePoints = new int[StringFunctions.safeCountCodePoints(slice)];
        int position = 0;
        for (int index = 0; index < codePoints.length; ++index) {
            codePoints[index] = SliceUtf8.getCodePointAt((Slice)slice, (int)position);
            position += SliceUtf8.lengthOfCodePoint((Slice)slice, (int)position);
        }
        return codePoints;
    }

    private static int safeCountCodePoints(Slice slice) {
        int codePoints = 0;
        int position = 0;
        while (position < slice.length()) {
            int codePoint = SliceUtf8.tryGetCodePointAt((Slice)slice, (int)position);
            if (codePoint < 0) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid UTF-8 encoding in characters: " + slice.toStringUtf8());
            }
            position += SliceUtf8.lengthOfCodePoint((int)codePoint);
            ++codePoints;
        }
        return codePoints;
    }

    @Description(value="converts the string to lower case")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice lower(@SqlType(value="varchar(x)") Slice slice) {
        return SliceUtf8.toLowerCase((Slice)slice);
    }

    @Description(value="converts the string to lower case")
    @ScalarFunction(value="lower")
    @LiteralParameters(value={"x"})
    @SqlType(value="char(x)")
    public static Slice charLower(@SqlType(value="char(x)") Slice slice) {
        return StringFunctions.lower(slice);
    }

    @Description(value="converts the string to upper case")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar(x)")
    public static Slice upper(@SqlType(value="varchar(x)") Slice slice) {
        return SliceUtf8.toUpperCase((Slice)slice);
    }

    @Description(value="converts the string to upper case")
    @ScalarFunction(value="upper")
    @LiteralParameters(value={"x"})
    @SqlType(value="char(x)")
    public static Slice charUpper(@SqlType(value="char(x)") Slice slice) {
        return StringFunctions.upper(slice);
    }

    private static Slice pad(Slice text, long targetLength, Slice padString, int paddingOffset) {
        Failures.checkCondition(0L <= targetLength && targetLength <= Integer.MAX_VALUE, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Target length must be in the range [0..2147483647]", new Object[0]);
        Failures.checkCondition(padString.length() > 0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Padding string must not be empty", new Object[0]);
        int textLength = SliceUtf8.countCodePoints((Slice)text);
        int resultLength = (int)targetLength;
        if (textLength == resultLength) {
            return text;
        }
        if (textLength > resultLength) {
            return SliceUtf8.substring((Slice)text, (int)0, (int)resultLength);
        }
        int padStringLength = SliceUtf8.countCodePoints((Slice)padString);
        int[] padStringCounts = new int[padStringLength];
        for (int i = 0; i < padStringLength; ++i) {
            padStringCounts[i] = SliceUtf8.lengthOfCodePointSafe((Slice)padString, (int)SliceUtf8.offsetOfCodePoint((Slice)padString, (int)i));
        }
        int bufferSize = text.length();
        for (int i = 0; i < resultLength - textLength; ++i) {
            bufferSize += padStringCounts[i % padStringLength];
        }
        Slice buffer = Slices.allocate((int)bufferSize);
        int countBytes = bufferSize - text.length();
        int startPointOfExistingText = (paddingOffset + countBytes) % bufferSize;
        buffer.setBytes(startPointOfExistingText, text);
        int byteIndex = paddingOffset;
        for (int i = 0; i < countBytes / padString.length(); ++i) {
            buffer.setBytes(byteIndex, padString);
            byteIndex += padString.length();
        }
        buffer.setBytes(byteIndex, padString.getBytes(0, paddingOffset + countBytes - byteIndex));
        return buffer;
    }

    @Description(value="pads a string on the left")
    @ScalarFunction(value="lpad")
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="varchar")
    public static Slice leftPad(@SqlType(value="varchar(x)") Slice text, @SqlType(value="bigint") long targetLength, @SqlType(value="varchar(y)") Slice padString) {
        return StringFunctions.pad(text, targetLength, padString, 0);
    }

    @Description(value="pads a string on the right")
    @ScalarFunction(value="rpad")
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="varchar")
    public static Slice rightPad(@SqlType(value="varchar(x)") Slice text, @SqlType(value="bigint") long targetLength, @SqlType(value="varchar(y)") Slice padString) {
        return StringFunctions.pad(text, targetLength, padString, text.length());
    }

    @Description(value="computes Levenshtein distance between two strings")
    @ScalarFunction
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="bigint")
    public static long levenshteinDistance(@SqlType(value="varchar(x)") Slice left, @SqlType(value="varchar(y)") Slice right) {
        int i;
        int[] rightCodePoints;
        int[] leftCodePoints = StringFunctions.castToCodePoints(left);
        if (leftCodePoints.length < (rightCodePoints = StringFunctions.castToCodePoints(right)).length) {
            int[] tempCodePoints = leftCodePoints;
            leftCodePoints = rightCodePoints;
            rightCodePoints = tempCodePoints;
        }
        if (rightCodePoints.length == 0) {
            return leftCodePoints.length;
        }
        Failures.checkCondition(leftCodePoints.length * (rightCodePoints.length - 1) <= 1000000, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "The combined inputs for Levenshtein distance are too large", new Object[0]);
        int[] distances = new int[rightCodePoints.length];
        for (i = 0; i < rightCodePoints.length; ++i) {
            distances[i] = i + 1;
        }
        for (i = 0; i < leftCodePoints.length; ++i) {
            int leftUpDistance = distances[0];
            distances[0] = leftCodePoints[i] == rightCodePoints[0] ? i : Math.min(i, distances[0]) + 1;
            for (int j = 1; j < rightCodePoints.length; ++j) {
                int leftUpDistanceNext = distances[j];
                distances[j] = leftCodePoints[i] == rightCodePoints[j] ? leftUpDistance : Math.min(distances[j - 1], Math.min(leftUpDistance, distances[j])) + 1;
                leftUpDistance = leftUpDistanceNext;
            }
        }
        return distances[rightCodePoints.length - 1];
    }

    @Description(value="computes Hamming distance between two strings")
    @ScalarFunction
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="bigint")
    public static long hammingDistance(@SqlType(value="varchar(x)") Slice left, @SqlType(value="varchar(y)") Slice right) {
        int rightPosition;
        int codePointRight;
        int codePointLeft;
        int distance = 0;
        int leftPosition = 0;
        for (rightPosition = 0; leftPosition < left.length() && rightPosition < right.length(); leftPosition += codePointLeft > 0 ? SliceUtf8.lengthOfCodePoint((int)codePointLeft) : -codePointLeft, rightPosition += codePointRight > 0 ? SliceUtf8.lengthOfCodePoint((int)codePointRight) : -codePointRight) {
            codePointLeft = SliceUtf8.tryGetCodePointAt((Slice)left, (int)leftPosition);
            if (codePointLeft == (codePointRight = SliceUtf8.tryGetCodePointAt((Slice)right, (int)rightPosition))) continue;
            ++distance;
        }
        Failures.checkCondition(leftPosition == left.length() && rightPosition == right.length(), (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "The input strings to hamming_distance function must have the same length", new Object[0]);
        return distance;
    }

    @Description(value="transforms the string to normalized form")
    @ScalarFunction
    @LiteralParameters(value={"x", "y"})
    @SqlType(value="varchar")
    public static Slice normalize(@SqlType(value="varchar(x)") Slice slice, @SqlType(value="varchar(y)") Slice form) {
        Normalizer.Form targetForm;
        try {
            targetForm = Normalizer.Form.valueOf(form.toStringUtf8());
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Normalization form must be one of [NFD, NFC, NFKD, NFKC]");
        }
        return Slices.utf8Slice((String)Normalizer.normalize(slice.toStringUtf8(), targetForm));
    }

    @Description(value="decodes the UTF-8 encoded string")
    @ScalarFunction
    @SqlType(value="varchar")
    public static Slice fromUtf8(@SqlType(value="varbinary") Slice slice) {
        return SliceUtf8.fixInvalidUtf8((Slice)slice);
    }

    @Description(value="decodes the UTF-8 encoded string")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varchar")
    public static Slice fromUtf8(@SqlType(value="varbinary") Slice slice, @SqlType(value="varchar(x)") Slice replacementCharacter) {
        OptionalInt replacementCodePoint;
        int count = SliceUtf8.countCodePoints((Slice)replacementCharacter);
        if (count > 1) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Replacement character string must empty or a single character");
        }
        if (count == 1) {
            try {
                replacementCodePoint = OptionalInt.of(SliceUtf8.getCodePointAt((Slice)replacementCharacter, (int)0));
            }
            catch (InvalidUtf8Exception e) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid replacement character");
            }
        } else {
            replacementCodePoint = OptionalInt.empty();
        }
        return SliceUtf8.fixInvalidUtf8((Slice)slice, (OptionalInt)replacementCodePoint);
    }

    @Description(value="decodes the UTF-8 encoded string")
    @ScalarFunction
    @SqlType(value="varchar")
    public static Slice fromUtf8(@SqlType(value="varbinary") Slice slice, @SqlType(value="bigint") long replacementCodePoint) {
        if (replacementCodePoint > 0x10FFFFL || Character.getType((int)replacementCodePoint) == 19) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid replacement character");
        }
        return SliceUtf8.fixInvalidUtf8((Slice)slice, (OptionalInt)OptionalInt.of((int)replacementCodePoint));
    }

    @Description(value="encodes the string to UTF-8")
    @ScalarFunction
    @LiteralParameters(value={"x"})
    @SqlType(value="varbinary")
    public static Slice toUtf8(@SqlType(value="varchar(x)") Slice slice) {
        return slice;
    }

    @Description(value="concatenates given character strings")
    @ScalarFunction
    @LiteralParameters(value={"x", "y", "u"})
    @Constraint(variable="u", expression="x + y")
    @SqlType(value="char(u)")
    public static Slice concat(@LiteralParameter(value="x") Long x, @SqlType(value="char(x)") Slice left, @SqlType(value="char(y)") Slice right) {
        int rightLength = right.length();
        if (rightLength == 0) {
            return left;
        }
        Slice paddedLeft = Chars.padSpaces((Slice)left, (int)x.intValue());
        int leftLength = paddedLeft.length();
        Slice result = Slices.allocate((int)(leftLength + rightLength));
        result.setBytes(0, paddedLeft);
        result.setBytes(leftLength, right);
        return result;
    }
}

