/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.server;

import com.facebook.presto.server.JavaVersion;
import com.google.common.base.StandardSystemProperty;
import com.google.common.collect.ImmutableList;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryManagerMXBean;
import java.nio.ByteOrder;
import java.util.List;
import java.util.Locale;
import java.util.OptionalLong;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.joda.time.DateTime;

final class PrestoSystemRequirements {
    private static final int MIN_FILE_DESCRIPTORS = 4096;
    private static final int RECOMMENDED_FILE_DESCRIPTORS = 8192;

    private PrestoSystemRequirements() {
    }

    public static void verifyJvmRequirements() {
        String vendor = StandardSystemProperty.JAVA_VENDOR.value();
        if (!"Oracle Corporation".equals(vendor)) {
            PrestoSystemRequirements.failRequirement("Presto requires an Oracle or OpenJDK JVM (found %s)", vendor);
        }
        PrestoSystemRequirements.verifyJavaVersion();
        String dataModel = System.getProperty("sun.arch.data.model");
        if (!"64".equals(dataModel)) {
            PrestoSystemRequirements.failRequirement("Presto requires a 64-bit JVM (found %s)", dataModel);
        }
        String osName = StandardSystemProperty.OS_NAME.value();
        String osArch = StandardSystemProperty.OS_ARCH.value();
        if ("Linux".equals(osName)) {
            if (!"amd64".equals(osArch)) {
                PrestoSystemRequirements.failRequirement("Presto requires x86-64 or amd64 on Linux (found %s)", osArch);
            }
        } else if ("Mac OS X".equals(osName)) {
            if (!"x86_64".equals(osArch)) {
                PrestoSystemRequirements.failRequirement("Presto requires x86_64 on Mac OS X (found %s)", osArch);
            }
        } else {
            PrestoSystemRequirements.failRequirement("Presto requires Linux or Mac OS X (found %s)", osName);
        }
        if (!ByteOrder.nativeOrder().equals(ByteOrder.LITTLE_ENDIAN)) {
            PrestoSystemRequirements.failRequirement("Presto requires a little endian platform (found %s)", ByteOrder.nativeOrder());
        }
        PrestoSystemRequirements.verifyUsingG1Gc();
        PrestoSystemRequirements.verifyFileDescriptor();
        PrestoSystemRequirements.verifySlice();
    }

    private static void verifyJavaVersion() {
        JavaVersion version;
        String javaVersion = StandardSystemProperty.JAVA_VERSION.value();
        if (javaVersion == null) {
            PrestoSystemRequirements.failRequirement("Java version not defined", new Object[0]);
        }
        if ((version = JavaVersion.parse(javaVersion)).getMajor() == 8 && version.getUpdate().isPresent() && version.getUpdate().getAsInt() >= 92) {
            return;
        }
        if (version.getMajor() >= 9) {
            return;
        }
        PrestoSystemRequirements.failRequirement("Presto requires Java 8u92+ (found %s)", javaVersion);
    }

    private static void verifyUsingG1Gc() {
        try {
            List garbageCollectors = (List)ManagementFactory.getGarbageCollectorMXBeans().stream().map(MemoryManagerMXBean::getName).collect(ImmutableList.toImmutableList());
            if (garbageCollectors.stream().noneMatch(name -> name.toUpperCase(Locale.US).startsWith("G1 "))) {
                PrestoSystemRequirements.warnRequirement("Current garbage collectors are %s. Presto recommends the G1 garbage collector.", garbageCollectors);
            }
        }
        catch (RuntimeException e) {
            PrestoSystemRequirements.failRequirement("Cannot read garbage collector information: %s", e);
        }
    }

    private static void verifyFileDescriptor() {
        OptionalLong maxFileDescriptorCount = PrestoSystemRequirements.getMaxFileDescriptorCount();
        if (!maxFileDescriptorCount.isPresent()) {
            PrestoSystemRequirements.failRequirement("Cannot read OS file descriptor limit", new Object[0]);
        }
        if (maxFileDescriptorCount.getAsLong() < 4096L) {
            PrestoSystemRequirements.failRequirement("Presto requires at least %s file descriptors (found %s)", 4096, maxFileDescriptorCount.getAsLong());
        }
        if (maxFileDescriptorCount.getAsLong() < 8192L) {
            PrestoSystemRequirements.warnRequirement("Current OS file descriptor limit is %s. Presto recommends at least %s", maxFileDescriptorCount.getAsLong(), 8192);
        }
    }

    private static OptionalLong getMaxFileDescriptorCount() {
        try {
            MBeanServer mbeanServer = ManagementFactory.getPlatformMBeanServer();
            Object maxFileDescriptorCount = mbeanServer.getAttribute(ObjectName.getInstance("java.lang:type=OperatingSystem"), "MaxFileDescriptorCount");
            return OptionalLong.of(((Number)maxFileDescriptorCount).longValue());
        }
        catch (Exception e) {
            return OptionalLong.empty();
        }
    }

    private static void verifySlice() {
        Slice slice = Slices.wrappedBuffer((byte[])new byte[5]);
        slice.setByte(4, 222);
        slice.setByte(3, 173);
        slice.setByte(2, 190);
        slice.setByte(1, 239);
        if (slice.getInt(1) != -559038737) {
            PrestoSystemRequirements.failRequirement("Slice library produced an unexpected result", new Object[0]);
        }
    }

    public static void verifySystemTimeIsReasonable() {
        int currentYear = DateTime.now().year().get();
        if (currentYear < 2015) {
            PrestoSystemRequirements.failRequirement("Presto requires the system time to be current (found year %s)", currentYear);
        }
    }

    private static void failRequirement(String format, Object ... args) {
        System.err.println(String.format(format, args));
        System.exit(100);
    }

    private static void warnRequirement(String format, Object ... args) {
        System.err.println("WARNING: " + String.format(format, args));
    }
}

