/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.operator.scalar;

import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.function.Description;
import com.facebook.presto.spi.function.LiteralParameters;
import com.facebook.presto.spi.function.ScalarFunction;
import com.facebook.presto.spi.function.SqlType;
import com.facebook.presto.util.Failures;
import com.google.common.hash.Hashing;
import com.google.common.io.BaseEncoding;
import com.google.common.primitives.Ints;
import io.airlift.slice.Slice;
import io.airlift.slice.Slices;
import io.airlift.slice.XxHash64;
import java.util.Base64;
import java.util.zip.CRC32;

public final class VarbinaryFunctions {
    private VarbinaryFunctions() {
    }

    @Description(value="length of the given binary")
    @ScalarFunction
    @SqlType(value="bigint")
    public static long length(@SqlType(value="varbinary") Slice slice) {
        return slice.length();
    }

    @Description(value="encode binary data as base64")
    @ScalarFunction
    @SqlType(value="varchar")
    public static Slice toBase64(@SqlType(value="varbinary") Slice slice) {
        return Slices.wrappedBuffer((byte[])Base64.getEncoder().encode(slice.getBytes()));
    }

    @Description(value="decode base64 encoded binary data")
    @ScalarFunction(value="from_base64")
    @LiteralParameters(value={"x"})
    @SqlType(value="varbinary")
    public static Slice fromBase64Varchar(@SqlType(value="varchar(x)") Slice slice) {
        try {
            return Slices.wrappedBuffer((byte[])Base64.getDecoder().decode(slice.getBytes()));
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
        }
    }

    @Description(value="decode base64 encoded binary data")
    @ScalarFunction(value="from_base64")
    @SqlType(value="varbinary")
    public static Slice fromBase64Varbinary(@SqlType(value="varbinary") Slice slice) {
        try {
            return Slices.wrappedBuffer((byte[])Base64.getDecoder().decode(slice.getBytes()));
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
        }
    }

    @Description(value="encode binary data as base64 using the URL safe alphabet")
    @ScalarFunction(value="to_base64url")
    @SqlType(value="varchar")
    public static Slice toBase64Url(@SqlType(value="varbinary") Slice slice) {
        return Slices.wrappedBuffer((byte[])Base64.getUrlEncoder().encode(slice.getBytes()));
    }

    @Description(value="decode URL safe base64 encoded binary data")
    @ScalarFunction(value="from_base64url")
    @LiteralParameters(value={"x"})
    @SqlType(value="varbinary")
    public static Slice fromBase64UrlVarchar(@SqlType(value="varchar(x)") Slice slice) {
        try {
            return Slices.wrappedBuffer((byte[])Base64.getUrlDecoder().decode(slice.getBytes()));
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
        }
    }

    @Description(value="decode URL safe base64 encoded binary data")
    @ScalarFunction(value="from_base64url")
    @SqlType(value="varbinary")
    public static Slice fromBase64UrlVarbinary(@SqlType(value="varbinary") Slice slice) {
        try {
            return Slices.wrappedBuffer((byte[])Base64.getUrlDecoder().decode(slice.getBytes()));
        }
        catch (IllegalArgumentException e) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, (Throwable)e);
        }
    }

    @Description(value="encode binary data as hex")
    @ScalarFunction
    @SqlType(value="varchar")
    public static Slice toHex(@SqlType(value="varbinary") Slice slice) {
        return Slices.utf8Slice((String)BaseEncoding.base16().encode(slice.getBytes()));
    }

    @Description(value="decode hex encoded binary data")
    @ScalarFunction(value="from_hex")
    @LiteralParameters(value={"x"})
    @SqlType(value="varbinary")
    public static Slice fromHexVarchar(@SqlType(value="varchar(x)") Slice slice) {
        if (slice.length() % 2 != 0) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "invalid input length " + slice.length());
        }
        byte[] result = new byte[slice.length() / 2];
        for (int i = 0; i < slice.length(); i += 2) {
            result[i / 2] = (byte)(VarbinaryFunctions.hexDigitCharToInt(slice.getByte(i)) << 4 | VarbinaryFunctions.hexDigitCharToInt(slice.getByte(i + 1)));
        }
        return Slices.wrappedBuffer((byte[])result);
    }

    @Description(value="encode value as a 64-bit 2's complement big endian varbinary")
    @ScalarFunction(value="to_big_endian_64")
    @SqlType(value="varbinary")
    public static Slice toBigEndian64(@SqlType(value="bigint") long value) {
        Slice slice = Slices.allocate((int)8);
        slice.setLong(0, Long.reverseBytes(value));
        return slice;
    }

    @Description(value="decode bigint value from a 64-bit 2's complement big endian varbinary")
    @ScalarFunction(value="from_big_endian_64")
    @SqlType(value="bigint")
    public static long fromBigEndian64(@SqlType(value="varbinary") Slice slice) {
        if (slice.length() != 8) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "expected 8-byte input, but got instead: " + slice.length());
        }
        return Long.reverseBytes(slice.getLong(0));
    }

    @Description(value="encode value as a 32-bit 2's complement big endian varbinary")
    @ScalarFunction(value="to_big_endian_32")
    @SqlType(value="varbinary")
    public static Slice toBigEndian32(@SqlType(value="integer") long value) {
        Slice slice = Slices.allocate((int)4);
        slice.setInt(0, Integer.reverseBytes((int)value));
        return slice;
    }

    @Description(value="decode bigint value from a 32-bit 2's complement big endian varbinary")
    @ScalarFunction(value="from_big_endian_32")
    @SqlType(value="integer")
    public static long fromBigEndian32(@SqlType(value="varbinary") Slice slice) {
        if (slice.length() != 4) {
            throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "expected 4-byte input, but got instead: " + slice.length());
        }
        return Integer.reverseBytes(slice.getInt(0));
    }

    @Description(value="encode value as a big endian varbinary according to IEEE 754 single-precision floating-point format")
    @ScalarFunction(value="to_ieee754_32")
    @SqlType(value="varbinary")
    public static Slice toIEEE754Binary32(@SqlType(value="real") long value) {
        Slice slice = Slices.allocate((int)4);
        slice.setInt(0, Integer.reverseBytes((int)value));
        return slice;
    }

    @Description(value="decode the 32-bit big-endian binary in IEEE 754 single-precision floating-point format")
    @ScalarFunction(value="from_ieee754_32")
    @SqlType(value="real")
    public static long fromIEEE754Binary32(@SqlType(value="varbinary") Slice slice) {
        Failures.checkCondition(slice.length() == 4, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Input floating-point value must be exactly 4 bytes long", new Object[0]);
        return Integer.reverseBytes(slice.getInt(0));
    }

    @Description(value="encode value as a big endian varbinary according to IEEE 754 double-precision floating-point format")
    @ScalarFunction(value="to_ieee754_64")
    @SqlType(value="varbinary")
    public static Slice toIEEE754Binary64(@SqlType(value="double") double value) {
        Slice slice = Slices.allocate((int)8);
        slice.setLong(0, Long.reverseBytes(Double.doubleToLongBits(value)));
        return slice;
    }

    @Description(value="decode the 64-bit big-endian binary in IEEE 754 double-precision floating-point format")
    @ScalarFunction(value="from_ieee754_64")
    @SqlType(value="double")
    public static double fromIEEE754Binary64(@SqlType(value="varbinary") Slice slice) {
        Failures.checkCondition(slice.length() == 8, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Input floating-point value must be exactly 8 bytes long", new Object[0]);
        return Double.longBitsToDouble(Long.reverseBytes(slice.getLong(0)));
    }

    @Description(value="compute md5 hash")
    @ScalarFunction
    @SqlType(value="varbinary")
    public static Slice md5(@SqlType(value="varbinary") Slice slice) {
        return Slices.wrappedBuffer((byte[])Hashing.md5().hashBytes(slice.getBytes()).asBytes());
    }

    @Description(value="compute sha1 hash")
    @ScalarFunction
    @SqlType(value="varbinary")
    public static Slice sha1(@SqlType(value="varbinary") Slice slice) {
        return Slices.wrappedBuffer((byte[])Hashing.sha1().hashBytes(slice.getBytes()).asBytes());
    }

    @Description(value="compute sha256 hash")
    @ScalarFunction
    @SqlType(value="varbinary")
    public static Slice sha256(@SqlType(value="varbinary") Slice slice) {
        return Slices.wrappedBuffer((byte[])Hashing.sha256().hashBytes(slice.getBytes()).asBytes());
    }

    @Description(value="compute sha512 hash")
    @ScalarFunction
    @SqlType(value="varbinary")
    public static Slice sha512(@SqlType(value="varbinary") Slice slice) {
        return Slices.wrappedBuffer((byte[])Hashing.sha512().hashBytes(slice.getBytes()).asBytes());
    }

    private static int hexDigitCharToInt(byte b) {
        if (b >= 48 && b <= 57) {
            return b - 48;
        }
        if (b >= 97 && b <= 102) {
            return b - 97 + 10;
        }
        if (b >= 65 && b <= 70) {
            return b - 65 + 10;
        }
        throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "invalid hex character: " + (char)b);
    }

    @Description(value="compute xxhash64 hash")
    @ScalarFunction
    @SqlType(value="varbinary")
    public static Slice xxhash64(@SqlType(value="varbinary") Slice slice) {
        Slice hash = Slices.allocate((int)8);
        hash.setLong(0, Long.reverseBytes(XxHash64.hash((Slice)slice)));
        return hash;
    }

    @Description(value="decode hex encoded binary data")
    @ScalarFunction(value="from_hex")
    @SqlType(value="varbinary")
    public static Slice fromHexVarbinary(@SqlType(value="varbinary") Slice slice) {
        return VarbinaryFunctions.fromHexVarchar(slice);
    }

    @Description(value="compute CRC-32")
    @ScalarFunction
    @SqlType(value="bigint")
    public static long crc32(@SqlType(value="varbinary") Slice slice) {
        CRC32 crc32 = new CRC32();
        crc32.update(slice.toByteBuffer());
        return crc32.getValue();
    }

    @Description(value="suffix starting at given index")
    @ScalarFunction
    @SqlType(value="varbinary")
    public static Slice substr(@SqlType(value="varbinary") Slice slice, @SqlType(value="bigint") long start) {
        return VarbinaryFunctions.substr(slice, start, (long)slice.length() - start + 1L);
    }

    @Description(value="substring of given length starting at an index")
    @ScalarFunction
    @SqlType(value="varbinary")
    public static Slice substr(@SqlType(value="varbinary") Slice slice, @SqlType(value="bigint") long start, @SqlType(value="bigint") long length) {
        if (start == 0L || length <= 0L || slice.length() == 0) {
            return Slices.EMPTY_SLICE;
        }
        int startByte = Ints.saturatedCast((long)start);
        int byteLength = Ints.saturatedCast((long)length);
        if (startByte > 0) {
            int indexStart = startByte - 1;
            if (indexStart >= slice.length()) {
                return Slices.EMPTY_SLICE;
            }
            int indexEnd = indexStart + byteLength;
            if (indexEnd > slice.length()) {
                indexEnd = slice.length();
            }
            return slice.slice(indexStart, indexEnd - indexStart);
        }
        if ((startByte += slice.length()) < 0) {
            return Slices.EMPTY_SLICE;
        }
        int indexStart = startByte;
        int indexEnd = indexStart + byteLength;
        if (indexEnd > slice.length()) {
            indexEnd = slice.length();
        }
        return slice.slice(indexStart, indexEnd - indexStart);
    }

    private static Slice pad(Slice inputSlice, long targetLength, Slice padSlice, int paddingOffset) {
        Failures.checkCondition(0L <= targetLength && targetLength <= Integer.MAX_VALUE, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Target length must be in the range [0..2147483647]", new Object[0]);
        Failures.checkCondition(padSlice.length() > 0, (ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Padding bytes must not be empty", new Object[0]);
        int inputLength = inputSlice.length();
        int resultLength = (int)targetLength;
        if (inputLength == resultLength) {
            return inputSlice;
        }
        if (inputLength > resultLength) {
            return inputSlice.slice(0, resultLength);
        }
        Slice buffer = Slices.allocate((int)resultLength);
        int fillLength = resultLength - inputLength;
        int startPointOfExistingText = (paddingOffset + fillLength) % resultLength;
        buffer.setBytes(startPointOfExistingText, inputSlice);
        int byteIndex = paddingOffset;
        for (int i = 0; i < fillLength / padSlice.length(); ++i) {
            buffer.setBytes(byteIndex, padSlice);
            byteIndex += padSlice.length();
        }
        buffer.setBytes(byteIndex, padSlice.getBytes(0, paddingOffset + fillLength - byteIndex));
        return buffer;
    }

    @Description(value="pads a varbinary on the left")
    @ScalarFunction(value="lpad")
    @SqlType(value="varbinary")
    public static Slice leftPad(@SqlType(value="varbinary") Slice inputSlice, @SqlType(value="bigint") long targetLength, @SqlType(value="varbinary") Slice padBytes) {
        return VarbinaryFunctions.pad(inputSlice, targetLength, padBytes, 0);
    }

    @Description(value="pads a varbinary on the right")
    @ScalarFunction(value="rpad")
    @SqlType(value="varbinary")
    public static Slice rightPad(@SqlType(value="varbinary") Slice inputSlice, @SqlType(value="bigint") long targetLength, @SqlType(value="varbinary") Slice padBytes) {
        return VarbinaryFunctions.pad(inputSlice, targetLength, padBytes, inputSlice.length());
    }
}

