/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.server.security;

import com.facebook.presto.server.security.LdapConfig;
import com.facebook.presto.server.security.util.jndi.JndiUtils;
import com.google.common.base.CharMatcher;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.Throwables;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.ByteStreams;
import io.airlift.http.client.HttpStatus;
import io.airlift.log.Logger;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.security.Principal;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nonnull;
import javax.inject.Inject;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletInputStream;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.servlet.http.HttpServletResponse;

public class LdapFilter
implements Filter {
    private static final Logger log = Logger.get(LdapFilter.class);
    private static final String BASIC_AUTHENTICATION_PREFIX = "Basic ";
    private static final String LDAP_CONTEXT_FACTORY = "com.sun.jndi.ldap.LdapCtxFactory";
    private final String ldapUrl;
    private final String userBindSearchPattern;
    private final Optional<String> groupAuthorizationSearchPattern;
    private final Optional<String> userBaseDistinguishedName;
    private final Map<String, String> basicEnvironment;
    private final LoadingCache<Credentials, Principal> authenticationCache;

    @Inject
    public LdapFilter(LdapConfig serverConfig) {
        this.ldapUrl = Objects.requireNonNull(serverConfig.getLdapUrl(), "ldapUrl is null");
        this.userBindSearchPattern = Objects.requireNonNull(serverConfig.getUserBindSearchPattern(), "userBindSearchPattern is null");
        this.groupAuthorizationSearchPattern = Optional.ofNullable(serverConfig.getGroupAuthorizationSearchPattern());
        this.userBaseDistinguishedName = Optional.ofNullable(serverConfig.getUserBaseDistinguishedName());
        if (this.groupAuthorizationSearchPattern.isPresent()) {
            Preconditions.checkState((boolean)this.userBaseDistinguishedName.isPresent(), (Object)"Base distinguished name (DN) for user is null");
        }
        ImmutableMap environment = ImmutableMap.builder().put((Object)"java.naming.factory.initial", (Object)LDAP_CONTEXT_FACTORY).put((Object)"java.naming.provider.url", (Object)this.ldapUrl).build();
        LdapFilter.checkEnvironment((Map<String, String>)environment);
        this.basicEnvironment = environment;
        this.authenticationCache = CacheBuilder.newBuilder().expireAfterWrite(serverConfig.getLdapCacheTtl().toMillis(), TimeUnit.MILLISECONDS).build((CacheLoader)new CacheLoader<Credentials, Principal>(){

            public Principal load(@Nonnull Credentials key) throws AuthenticationException {
                return LdapFilter.this.authenticate(key.getUser(), key.getPassword());
            }
        });
    }

    private static void checkEnvironment(Map<String, String> environment) {
        try {
            LdapFilter.closeContext(LdapFilter.createDirContext(environment));
        }
        catch (NamingException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    private static InitialDirContext createDirContext(Map<String, String> environment) throws NamingException {
        return JndiUtils.getInitialDirContext(environment);
    }

    private static void closeContext(InitialDirContext context) {
        if (context != null) {
            try {
                context.close();
            }
            catch (NamingException namingException) {
                // empty catch block
            }
        }
    }

    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain nextFilter) throws IOException, ServletException {
        if (!servletRequest.isSecure()) {
            nextFilter.doFilter(servletRequest, servletResponse);
            return;
        }
        HttpServletRequest request = (HttpServletRequest)servletRequest;
        HttpServletResponse response = (HttpServletResponse)servletResponse;
        try {
            String header = request.getHeader("Authorization");
            Credentials credentials = LdapFilter.getCredentials(header);
            final Principal principal = this.getPrincipal(credentials);
            nextFilter.doFilter((ServletRequest)new HttpServletRequestWrapper(request){

                public Principal getUserPrincipal() {
                    return principal;
                }
            }, servletResponse);
        }
        catch (AuthenticationException e) {
            log.debug((Throwable)e, "LDAP authentication failed");
            LdapFilter.processAuthenticationException(e, request, response);
        }
    }

    private Principal getPrincipal(Credentials credentials) throws AuthenticationException {
        try {
            return (Principal)this.authenticationCache.get((Object)credentials);
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause != null) {
                Throwables.throwIfInstanceOf((Throwable)cause, AuthenticationException.class);
            }
            throw Throwables.propagate((Throwable)cause);
        }
    }

    private static void processAuthenticationException(AuthenticationException e, HttpServletRequest request, HttpServletResponse response) throws IOException {
        if (e.getStatus() == HttpStatus.UNAUTHORIZED) {
            LdapFilter.skipRequestBody(request);
            response.setHeader("WWW-Authenticate", "Basic realm=\"presto\"");
        }
        response.sendError(e.getStatus().code(), e.getMessage());
    }

    private static void skipRequestBody(HttpServletRequest request) throws IOException {
        try (ServletInputStream inputStream = request.getInputStream();){
            ByteStreams.copy((InputStream)inputStream, (OutputStream)ByteStreams.nullOutputStream());
        }
    }

    private static Credentials getCredentials(String header) throws AuthenticationException {
        if (header == null) {
            throw new AuthenticationException(HttpStatus.UNAUTHORIZED, "Unauthorized");
        }
        if (!header.startsWith(BASIC_AUTHENTICATION_PREFIX)) {
            throw new AuthenticationException(HttpStatus.BAD_REQUEST, "Basic authentication is expected");
        }
        String base64EncodedCredentials = header.substring(BASIC_AUTHENTICATION_PREFIX.length());
        String credentials = LdapFilter.decodeCredentials(base64EncodedCredentials);
        List parts = Splitter.on((char)':').limit(2).splitToList((CharSequence)credentials);
        if (parts.size() != 2 || parts.stream().anyMatch(String::isEmpty)) {
            throw new AuthenticationException(HttpStatus.BAD_REQUEST, "Malformed decoded credentials");
        }
        return new Credentials((String)parts.get(0), (String)parts.get(1));
    }

    private static String decodeCredentials(String base64EncodedCredentials) throws AuthenticationException {
        byte[] bytes;
        try {
            bytes = Base64.getDecoder().decode(base64EncodedCredentials);
        }
        catch (IllegalArgumentException e) {
            throw new AuthenticationException(HttpStatus.BAD_REQUEST, "Invalid base64 encoded credentials");
        }
        return new String(bytes, StandardCharsets.UTF_8);
    }

    private Principal authenticate(String user, String password) throws AuthenticationException {
        Map<String, String> environment = this.createEnvironment(user, password);
        InitialDirContext context = null;
        try {
            context = LdapFilter.createDirContext(environment);
            this.checkForGroupMembership(user, context);
            log.debug("Authentication successful for user %s", new Object[]{user});
            LdapPrincipal ldapPrincipal = new LdapPrincipal(user);
            return ldapPrincipal;
        }
        catch (javax.naming.AuthenticationException e) {
            String formattedAsciiMessage = String.format("Invalid credentials: %s", CharMatcher.JAVA_ISO_CONTROL.removeFrom((CharSequence)e.getMessage()));
            log.debug("Authentication failed for user [%s]. %s", new Object[]{user, e.getMessage()});
            throw new AuthenticationException(HttpStatus.UNAUTHORIZED, formattedAsciiMessage, e);
        }
        catch (NamingException e) {
            log.debug("Authentication failed", new Object[]{e.getMessage()});
            throw new AuthenticationException(HttpStatus.INTERNAL_SERVER_ERROR, "Authentication failed", e);
        }
        finally {
            LdapFilter.closeContext(context);
        }
    }

    private Map<String, String> createEnvironment(String user, String password) {
        return ImmutableMap.builder().putAll(this.basicEnvironment).put((Object)"java.naming.security.authentication", (Object)"simple").put((Object)"java.naming.security.principal", (Object)this.createPrincipal(user)).put((Object)"java.naming.security.credentials", (Object)password).build();
    }

    private String createPrincipal(String user) {
        return this.replaceUser(this.userBindSearchPattern, user);
    }

    private String replaceUser(String pattern, String user) {
        return pattern.replaceAll("\\$\\{USER\\}", user);
    }

    private void checkForGroupMembership(String user, DirContext context) throws AuthenticationException {
        boolean authorized;
        if (!this.groupAuthorizationSearchPattern.isPresent()) {
            return;
        }
        String searchFilter = this.replaceUser(this.groupAuthorizationSearchPattern.get(), user);
        SearchControls searchControls = new SearchControls();
        searchControls.setSearchScope(2);
        NamingEnumeration<SearchResult> search = null;
        try {
            search = context.search(this.userBaseDistinguishedName.get(), searchFilter, searchControls);
            authorized = search.hasMoreElements();
        }
        catch (NamingException e) {
            log.debug("Authentication failed", new Object[]{e.getMessage()});
            throw new AuthenticationException(HttpStatus.INTERNAL_SERVER_ERROR, "Authentication failed", e);
        }
        finally {
            if (search != null) {
                try {
                    search.close();
                }
                catch (NamingException namingException) {}
            }
        }
        if (!authorized) {
            String message = String.format("Unauthorized user: User %s not a member of the authorized group", user);
            log.debug("Authorization failed for user. " + message);
            throw new AuthenticationException(HttpStatus.UNAUTHORIZED, message);
        }
        log.debug("Authorization succeeded for user %s", new Object[]{user});
    }

    public void init(FilterConfig filterConfig) {
    }

    public void destroy() {
    }

    private static class AuthenticationException
    extends Exception {
        private final HttpStatus status;

        private AuthenticationException(HttpStatus status, String message) {
            this(status, message, (Throwable)null);
        }

        private AuthenticationException(HttpStatus status, String message, Throwable cause) {
            super(message, cause);
            Objects.requireNonNull(message, "message is null");
            this.status = Objects.requireNonNull(status, "status is null");
        }

        public HttpStatus getStatus() {
            return this.status;
        }
    }

    private static class Credentials {
        private final String user;
        private final String password;

        private Credentials(String user, String password) {
            this.user = Objects.requireNonNull(user);
            this.password = Objects.requireNonNull(password);
        }

        public String getUser() {
            return this.user;
        }

        public String getPassword() {
            return this.password;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Credentials that = (Credentials)o;
            return Objects.equals(this.user, that.user) && Objects.equals(this.password, that.password);
        }

        public int hashCode() {
            return Objects.hash(this.user, this.password);
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("user", (Object)this.user).add("password", (Object)this.password).toString();
        }
    }

    private static final class LdapPrincipal
    implements Principal {
        private final String name;

        private LdapPrincipal(String name) {
            this.name = Objects.requireNonNull(name, "name is null");
        }

        @Override
        public String getName() {
            return this.name;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            LdapPrincipal that = (LdapPrincipal)o;
            return Objects.equals(this.name, that.name);
        }

        @Override
        public int hashCode() {
            return Objects.hash(this.name);
        }

        @Override
        public String toString() {
            return this.name;
        }
    }
}

