/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.operator;

import com.facebook.presto.ScheduledSplit;
import com.facebook.presto.TaskSource;
import com.facebook.presto.metadata.Split;
import com.facebook.presto.operator.DeleteOperator;
import com.facebook.presto.operator.DriverContext;
import com.facebook.presto.operator.Operator;
import com.facebook.presto.operator.SourceOperator;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.Page;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.spi.UpdatablePageSource;
import com.facebook.presto.sql.planner.plan.PlanNodeId;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.base.VerifyException;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.SettableFuture;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;
import javax.annotation.concurrent.GuardedBy;

public class Driver
implements Closeable {
    private static final Logger log = Logger.get(Driver.class);
    private final DriverContext driverContext;
    private final List<Operator> operators;
    private final Optional<SourceOperator> sourceOperator;
    private final Optional<DeleteOperator> deleteOperator;
    private final AtomicReference<TaskSource> newTaskSource = new AtomicReference();
    private final AtomicReference<State> state = new AtomicReference<State>(State.ALIVE);
    private final DriverLock exclusiveLock = new DriverLock();
    @GuardedBy(value="exclusiveLock")
    private TaskSource currentTaskSource;

    public Driver(DriverContext driverContext, Operator firstOperator, Operator ... otherOperators) {
        this(Objects.requireNonNull(driverContext, "driverContext is null"), (List<Operator>)ImmutableList.builder().add((Object)Objects.requireNonNull(firstOperator, "firstOperator is null")).add((Object[])Objects.requireNonNull(otherOperators, "otherOperators is null")).build());
    }

    public Driver(DriverContext driverContext, List<Operator> operators) {
        this.driverContext = Objects.requireNonNull(driverContext, "driverContext is null");
        this.operators = ImmutableList.copyOf((Collection)Objects.requireNonNull(operators, "operators is null"));
        Preconditions.checkArgument((!operators.isEmpty() ? 1 : 0) != 0, (Object)"There must be at least one operator");
        Optional<SourceOperator> sourceOperator = Optional.empty();
        Optional<Object> deleteOperator = Optional.empty();
        for (Operator operator2 : operators) {
            if (operator2 instanceof SourceOperator) {
                Preconditions.checkArgument((!sourceOperator.isPresent() ? 1 : 0) != 0, (Object)"There must be at most one SourceOperator");
                sourceOperator = Optional.of((SourceOperator)operator2);
                continue;
            }
            if (!(operator2 instanceof DeleteOperator)) continue;
            Preconditions.checkArgument((!deleteOperator.isPresent() ? 1 : 0) != 0, (Object)"There must be at most one DeleteOperator");
            deleteOperator = Optional.of((DeleteOperator)operator2);
        }
        this.sourceOperator = sourceOperator;
        this.deleteOperator = deleteOperator;
        this.currentTaskSource = sourceOperator.map(operator -> new TaskSource(operator.getSourceId(), (Set<ScheduledSplit>)ImmutableSet.of(), false)).orElse(null);
    }

    public DriverContext getDriverContext() {
        return this.driverContext;
    }

    public Optional<PlanNodeId> getSourceId() {
        return this.sourceOperator.map(SourceOperator::getSourceId);
    }

    @Override
    public void close() {
        if (!this.state.compareAndSet(State.ALIVE, State.NEED_DESTRUCTION)) {
            return;
        }
        this.exclusiveLock.interruptCurrentOwner();
        this.tryWithLock(() -> Boolean.TRUE);
    }

    public boolean isFinished() {
        this.checkLockNotHeld("Can not check finished status while holding the driver lock");
        Optional<Boolean> result = this.tryWithLock(this::isFinishedInternal);
        return result.orElseGet(() -> this.state.get() != State.ALIVE || this.driverContext.isDone());
    }

    @GuardedBy(value="exclusiveLock")
    private boolean isFinishedInternal() {
        boolean finished;
        this.checkLockHeld("Lock must be held to call isFinishedInternal");
        boolean bl = finished = this.state.get() != State.ALIVE || this.driverContext.isDone() || this.operators.get(this.operators.size() - 1).isFinished();
        if (finished) {
            this.state.compareAndSet(State.ALIVE, State.NEED_DESTRUCTION);
        }
        return finished;
    }

    public void updateSource(TaskSource source) {
        this.checkLockNotHeld("Can not update sources while holding the driver lock");
        if (!this.sourceOperator.isPresent() || !this.sourceOperator.get().getSourceId().equals(source.getPlanNodeId())) {
            return;
        }
        this.newTaskSource.updateAndGet(current -> current == null ? source : current.update(source));
        this.tryWithLock(() -> Boolean.TRUE);
    }

    @GuardedBy(value="exclusiveLock")
    private void processNewSources() {
        this.checkLockHeld("Lock must be held to call processNewSources");
        if (this.state.get() != State.ALIVE) {
            return;
        }
        TaskSource source = this.newTaskSource.getAndSet(null);
        if (source == null) {
            return;
        }
        TaskSource newSource = this.currentTaskSource.update(source);
        if (newSource == this.currentTaskSource) {
            return;
        }
        Sets.SetView newSplits = Sets.difference(newSource.getSplits(), this.currentTaskSource.getSplits());
        SourceOperator sourceOperator = this.sourceOperator.orElseThrow(VerifyException::new);
        for (ScheduledSplit newSplit : newSplits) {
            Split split = newSplit.getSplit();
            Supplier<Optional<UpdatablePageSource>> pageSource = sourceOperator.addSplit(split);
            this.deleteOperator.ifPresent(deleteOperator -> deleteOperator.setPageSource(pageSource));
        }
        if (newSource.isNoMoreSplits()) {
            sourceOperator.noMoreSplits();
        }
        this.currentTaskSource = newSource;
    }

    public ListenableFuture<?> processFor(Duration duration) {
        this.checkLockNotHeld("Can not process for a duration while holding the driver lock");
        Objects.requireNonNull(duration, "duration is null");
        long maxRuntime = duration.roundTo(TimeUnit.NANOSECONDS);
        Optional<ListenableFuture> result = this.tryWithLock(100L, TimeUnit.MILLISECONDS, () -> {
            this.driverContext.startProcessTimer();
            try {
                long start = System.nanoTime();
                do {
                    ListenableFuture<?> future;
                    if ((future = this.processInternal()).isDone()) continue;
                    ListenableFuture<?> listenableFuture = future;
                    return listenableFuture;
                } while (System.nanoTime() - start < maxRuntime && !this.isFinishedInternal());
            }
            finally {
                this.driverContext.recordProcessed();
            }
            return Operator.NOT_BLOCKED;
        });
        return result.orElse(Operator.NOT_BLOCKED);
    }

    public ListenableFuture<?> process() {
        this.checkLockNotHeld("Can not process while holding the driver lock");
        Optional<ListenableFuture> result = this.tryWithLock(100L, TimeUnit.MILLISECONDS, this::processInternal);
        return result.orElse(Operator.NOT_BLOCKED);
    }

    @GuardedBy(value="exclusiveLock")
    private ListenableFuture<?> processInternal() {
        this.checkLockHeld("Lock must be held to call processInternal");
        try {
            this.processNewSources();
            if (this.operators.size() == 1) {
                if (this.driverContext.isDone()) {
                    return Operator.NOT_BLOCKED;
                }
                Operator current = this.operators.get(0);
                ListenableFuture<?> blocked = Driver.isBlocked(current);
                if (!blocked.isDone()) {
                    current.getOperatorContext().recordBlocked(blocked);
                    return blocked;
                }
                current.getOperatorContext().startIntervalTimer();
                current.finish();
                current.getOperatorContext().recordFinish();
                return Operator.NOT_BLOCKED;
            }
            boolean movedPage = false;
            for (int i = 0; i < this.operators.size() - 1 && !this.driverContext.isDone(); ++i) {
                Operator current = this.operators.get(i);
                Operator next = this.operators.get(i + 1);
                if (!Driver.isBlocked(current).isDone() || !Driver.isBlocked(next).isDone()) continue;
                if (!current.isFinished() && next.needsInput()) {
                    current.getOperatorContext().startIntervalTimer();
                    Page page = current.getOutput();
                    current.getOperatorContext().recordGetOutput(page);
                    if (page != null && page.getPositionCount() != 0) {
                        next.getOperatorContext().startIntervalTimer();
                        next.addInput(page);
                        next.getOperatorContext().recordAddInput(page);
                        movedPage = true;
                    }
                    if (current instanceof SourceOperator) {
                        movedPage = true;
                    }
                }
                if (!current.isFinished()) continue;
                next.getOperatorContext().startIntervalTimer();
                next.finish();
                next.getOperatorContext().recordFinish();
            }
            if (!movedPage) {
                ArrayList<Operator> blockedOperators = new ArrayList<Operator>();
                ArrayList blockedFutures = new ArrayList();
                for (Operator operator : this.operators) {
                    ListenableFuture<?> blocked = Driver.isBlocked(operator);
                    if (blocked.isDone()) continue;
                    blockedOperators.add(operator);
                    blockedFutures.add(blocked);
                }
                if (!blockedFutures.isEmpty()) {
                    ListenableFuture<?> blocked = Driver.firstFinishedFuture(blockedFutures);
                    this.driverContext.recordBlocked(blocked);
                    for (Operator operator : blockedOperators) {
                        operator.getOperatorContext().recordBlocked(blocked);
                    }
                    return blocked;
                }
            }
            return Operator.NOT_BLOCKED;
        }
        catch (Throwable t) {
            List<StackTraceElement> interrupterStack = this.exclusiveLock.getInterrupterStack();
            if (interrupterStack == null) {
                this.driverContext.failed(t);
                throw t;
            }
            Exception exception = new Exception("Interrupted By");
            exception.setStackTrace((StackTraceElement[])interrupterStack.stream().toArray(StackTraceElement[]::new));
            PrestoException newException = new PrestoException((ErrorCodeSupplier)StandardErrorCode.GENERIC_INTERNAL_ERROR, "Driver was interrupted", (Throwable)exception);
            newException.addSuppressed(t);
            this.driverContext.failed(newException);
            throw newException;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @GuardedBy(value="exclusiveLock")
    private void destroyIfNecessary() {
        this.checkLockHeld("Lock must be held to call destroyIfNecessary");
        if (!this.state.compareAndSet(State.NEED_DESTRUCTION, State.DESTROYED)) {
            return;
        }
        boolean wasInterrupted = Thread.interrupted();
        Throwable inFlightException = null;
        try {
            for (Operator operator : this.operators) {
                try {
                    operator.close();
                }
                catch (InterruptedException t) {
                    wasInterrupted = true;
                }
                catch (Throwable t) {
                    inFlightException = Driver.addSuppressedException(inFlightException, t, "Error closing operator %s for task %s", operator.getOperatorContext().getOperatorId(), this.driverContext.getTaskId());
                }
                try {
                    operator.getOperatorContext().setMemoryReservation(0L);
                }
                catch (Throwable t) {
                    inFlightException = Driver.addSuppressedException(inFlightException, t, "Error freeing memory for operator %s for task %s", operator.getOperatorContext().getOperatorId(), this.driverContext.getTaskId());
                }
                try {
                    operator.getOperatorContext().closeSystemMemoryContext();
                }
                catch (Throwable t) {
                    inFlightException = Driver.addSuppressedException(inFlightException, t, "Error freeing system memory for operator %s for task %s", operator.getOperatorContext().getOperatorId(), this.driverContext.getTaskId());
                }
            }
            if (this.driverContext.getMemoryUsage() > 0L) {
                log.error("Driver still has memory reserved after freeing all operator memory. Freeing it.");
            }
            if (this.driverContext.getSystemMemoryUsage() > 0L) {
                log.error("Driver still has system memory reserved after freeing all operator memory. Freeing it.");
            }
            this.driverContext.freeMemory(this.driverContext.getMemoryUsage());
            this.driverContext.freeSystemMemory(this.driverContext.getSystemMemoryUsage());
            this.driverContext.finished();
        }
        catch (Throwable t) {
            inFlightException = Driver.addSuppressedException(inFlightException, t, "Error destroying driver for task %s", this.driverContext.getTaskId());
        }
        finally {
            if (wasInterrupted) {
                Thread.currentThread().interrupt();
            }
        }
        if (inFlightException != null) {
            Throwables.throwIfUnchecked((Throwable)inFlightException);
            throw new RuntimeException(inFlightException);
        }
    }

    private static ListenableFuture<?> isBlocked(Operator operator) {
        ListenableFuture<?> blocked = operator.isBlocked();
        if (blocked.isDone()) {
            blocked = operator.getOperatorContext().isWaitingForMemory();
        }
        return blocked;
    }

    private static Throwable addSuppressedException(Throwable inFlightException, Throwable newException, String message, Object ... args) {
        if (newException instanceof Error) {
            if (inFlightException == null) {
                inFlightException = newException;
            } else if (inFlightException != newException) {
                inFlightException.addSuppressed(newException);
            }
        } else {
            log.error(newException, message, args);
        }
        return inFlightException;
    }

    private synchronized void checkLockNotHeld(String message) {
        Preconditions.checkState((!this.exclusiveLock.isHeldByCurrentThread() ? 1 : 0) != 0, (Object)message);
    }

    @GuardedBy(value="exclusiveLock")
    private synchronized void checkLockHeld(String message) {
        Preconditions.checkState((boolean)this.exclusiveLock.isHeldByCurrentThread(), (Object)message);
    }

    private static ListenableFuture<?> firstFinishedFuture(List<ListenableFuture<?>> futures) {
        SettableFuture result = SettableFuture.create();
        ListeningExecutorService executor = MoreExecutors.newDirectExecutorService();
        for (ListenableFuture<?> future : futures) {
            future.addListener(() -> result.set(null), (Executor)executor);
        }
        return result;
    }

    private <T> Optional<T> tryWithLock(Supplier<T> task) {
        return this.tryWithLock(0L, TimeUnit.MILLISECONDS, task);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T> Optional<T> tryWithLock(long timeout, TimeUnit unit, Supplier<T> task) {
        Optional<T> result;
        this.checkLockNotHeld("Lock can not be reacquired");
        boolean acquired = false;
        try {
            acquired = this.exclusiveLock.tryLock(timeout, unit);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        if (!acquired) {
            return Optional.empty();
        }
        try {
            result = Optional.of(task.get());
        }
        finally {
            try {
                try {
                    this.processNewSources();
                }
                finally {
                    this.destroyIfNecessary();
                }
            }
            finally {
                this.exclusiveLock.unlock();
            }
        }
        while (this.newTaskSource.get() != null && this.state.get() == State.ALIVE && this.exclusiveLock.tryLock()) {
            try {
                try {
                    this.processNewSources();
                }
                finally {
                    this.destroyIfNecessary();
                }
            }
            finally {
                this.exclusiveLock.unlock();
            }
        }
        return result;
    }

    private static class DriverLock {
        private final ReentrantLock lock = new ReentrantLock();
        @GuardedBy(value="this")
        private Thread currentOwner;
        @GuardedBy(value="this")
        private List<StackTraceElement> interrupterStack;

        private DriverLock() {
        }

        public boolean isHeldByCurrentThread() {
            return this.lock.isHeldByCurrentThread();
        }

        public boolean tryLock() {
            Preconditions.checkState((!this.lock.isHeldByCurrentThread() ? 1 : 0) != 0, (Object)"Lock is not reentrant");
            boolean acquired = this.lock.tryLock();
            if (acquired) {
                this.setOwner();
            }
            return acquired;
        }

        public boolean tryLock(long timeout, TimeUnit unit) throws InterruptedException {
            Preconditions.checkState((!this.lock.isHeldByCurrentThread() ? 1 : 0) != 0, (Object)"Lock is not reentrant");
            boolean acquired = this.lock.tryLock(timeout, unit);
            if (acquired) {
                this.setOwner();
            }
            return acquired;
        }

        private synchronized void setOwner() {
            Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread(), (Object)"Current thread does not hold lock");
            this.currentOwner = Thread.currentThread();
        }

        public synchronized void unlock() {
            Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread(), (Object)"Current thread does not hold lock");
            this.currentOwner = null;
            this.lock.unlock();
        }

        public synchronized List<StackTraceElement> getInterrupterStack() {
            return this.interrupterStack;
        }

        public synchronized void interruptCurrentOwner() {
            if (this.interrupterStack == null) {
                this.interrupterStack = ImmutableList.copyOf((Object[])Thread.currentThread().getStackTrace());
            }
            if (this.currentOwner != null) {
                this.currentOwner.interrupt();
            }
        }
    }

    private static enum State {
        ALIVE,
        NEED_DESTRUCTION,
        DESTROYED;

    }
}

