/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.execution;

import com.facebook.presto.Session;
import com.facebook.presto.SystemSessionProperties;
import com.facebook.presto.event.query.QueryMonitor;
import com.facebook.presto.execution.AbandonedException;
import com.facebook.presto.execution.FailedQueryExecution;
import com.facebook.presto.execution.LocationFactory;
import com.facebook.presto.execution.QueryExecution;
import com.facebook.presto.execution.QueryId;
import com.facebook.presto.execution.QueryInfo;
import com.facebook.presto.execution.QueryManager;
import com.facebook.presto.execution.QueryManagerConfig;
import com.facebook.presto.execution.QueryQueueManager;
import com.facebook.presto.execution.QueryState;
import com.facebook.presto.execution.SqlQueryManagerStats;
import com.facebook.presto.execution.StageId;
import com.facebook.presto.memory.ClusterMemoryManager;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.PrestoException;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.sql.parser.ParsingException;
import com.facebook.presto.sql.parser.SqlParser;
import com.facebook.presto.sql.tree.Statement;
import com.facebook.presto.util.ImmutableCollectors;
import com.google.common.base.Preconditions;
import io.airlift.concurrent.ThreadPoolExecutorMBean;
import io.airlift.concurrent.Threads;
import io.airlift.log.Logger;
import io.airlift.units.Duration;
import java.net.URI;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.annotation.PreDestroy;
import javax.annotation.concurrent.ThreadSafe;
import javax.inject.Inject;
import org.joda.time.DateTime;
import org.joda.time.ReadableInstant;
import org.weakref.jmx.Flatten;
import org.weakref.jmx.Managed;
import org.weakref.jmx.Nested;

@ThreadSafe
public class SqlQueryManager
implements QueryManager {
    private static final Logger log = Logger.get(SqlQueryManager.class);
    private final SqlParser sqlParser;
    private final ExecutorService queryExecutor;
    private final ThreadPoolExecutorMBean queryExecutorMBean;
    private final QueryQueueManager queueManager;
    private final ClusterMemoryManager memoryManager;
    private final int maxQueryHistory;
    private final Duration maxQueryAge;
    private final ConcurrentMap<QueryId, QueryExecution> queries = new ConcurrentHashMap<QueryId, QueryExecution>();
    private final Queue<QueryExecution> expirationQueue = new LinkedBlockingQueue<QueryExecution>();
    private final Duration clientTimeout;
    private final ScheduledExecutorService queryManagementExecutor;
    private final ThreadPoolExecutorMBean queryManagementExecutorMBean;
    private final QueryMonitor queryMonitor;
    private final LocationFactory locationFactory;
    private final Map<Class<? extends Statement>, QueryExecution.QueryExecutionFactory<?>> executionFactories;
    private final SqlQueryManagerStats stats = new SqlQueryManagerStats();

    @Inject
    public SqlQueryManager(SqlParser sqlParser, QueryManagerConfig config, QueryMonitor queryMonitor, QueryQueueManager queueManager, ClusterMemoryManager memoryManager, LocationFactory locationFactory, Map<Class<? extends Statement>, QueryExecution.QueryExecutionFactory<?>> executionFactories) {
        this.sqlParser = Objects.requireNonNull(sqlParser, "sqlParser is null");
        this.executionFactories = Objects.requireNonNull(executionFactories, "executionFactories is null");
        this.queryExecutor = Executors.newCachedThreadPool(Threads.threadsNamed((String)"query-scheduler-%s"));
        this.queryExecutorMBean = new ThreadPoolExecutorMBean((ThreadPoolExecutor)this.queryExecutor);
        Objects.requireNonNull(config, "config is null");
        this.queueManager = Objects.requireNonNull(queueManager, "queueManager is null");
        this.memoryManager = Objects.requireNonNull(memoryManager, "memoryManager is null");
        this.queryMonitor = Objects.requireNonNull(queryMonitor, "queryMonitor is null");
        this.locationFactory = Objects.requireNonNull(locationFactory, "locationFactory is null");
        this.maxQueryAge = config.getMaxQueryAge();
        this.maxQueryHistory = config.getMaxQueryHistory();
        this.clientTimeout = config.getClientTimeout();
        this.queryManagementExecutor = Executors.newScheduledThreadPool(config.getQueryManagerExecutorPoolSize(), Threads.threadsNamed((String)"query-management-%s"));
        this.queryManagementExecutorMBean = new ThreadPoolExecutorMBean((ThreadPoolExecutor)((Object)this.queryManagementExecutor));
        this.queryManagementExecutor.scheduleWithFixedDelay(new Runnable(){

            @Override
            public void run() {
                try {
                    SqlQueryManager.this.failAbandonedQueries();
                }
                catch (Throwable e) {
                    log.warn(e, "Error cancelling abandoned queries");
                }
                try {
                    SqlQueryManager.this.enforceMemoryLimits();
                }
                catch (Throwable e) {
                    log.warn(e, "Error enforcing memory limits");
                }
                try {
                    SqlQueryManager.this.enforceQueryMaxRunTimeLimits();
                }
                catch (Throwable e) {
                    log.warn(e, "Error enforcing query timeout limits");
                }
                try {
                    SqlQueryManager.this.removeExpiredQueries();
                }
                catch (Throwable e) {
                    log.warn(e, "Error removing expired queries");
                }
                try {
                    SqlQueryManager.this.pruneExpiredQueries();
                }
                catch (Throwable e) {
                    log.warn(e, "Error pruning expired queries");
                }
            }
        }, 1L, 1L, TimeUnit.SECONDS);
    }

    @PreDestroy
    public void stop() {
        boolean queryCancelled = false;
        for (QueryExecution queryExecution : this.queries.values()) {
            QueryInfo queryInfo = queryExecution.getQueryInfo();
            if (queryInfo.getState().isDone()) continue;
            log.info("Server shutting down. Query %s has been cancelled", new Object[]{queryExecution.getQueryInfo().getQueryId()});
            queryExecution.fail(new PrestoException((ErrorCodeSupplier)StandardErrorCode.SERVER_SHUTTING_DOWN, "Server is shutting down. Query " + queryInfo.getQueryId() + " has been cancelled"));
            queryCancelled = true;
        }
        if (queryCancelled) {
            try {
                TimeUnit.SECONDS.sleep(5L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        this.queryManagementExecutor.shutdownNow();
        this.queryExecutor.shutdownNow();
    }

    @Override
    public List<QueryInfo> getAllQueryInfo() {
        return (List)this.queries.values().stream().map(queryExecution -> {
            try {
                return queryExecution.getQueryInfo();
            }
            catch (RuntimeException ignored) {
                return null;
            }
        }).filter(Objects::nonNull).collect(ImmutableCollectors.toImmutableList());
    }

    @Override
    public Duration waitForStateChange(QueryId queryId, QueryState currentState, Duration maxWait) throws InterruptedException {
        Objects.requireNonNull(queryId, "queryId is null");
        Objects.requireNonNull(maxWait, "maxWait is null");
        QueryExecution query = (QueryExecution)this.queries.get(queryId);
        if (query == null) {
            return maxWait;
        }
        return query.waitForStateChange(currentState, maxWait);
    }

    @Override
    public QueryInfo getQueryInfo(QueryId queryId) {
        Objects.requireNonNull(queryId, "queryId is null");
        QueryExecution query = (QueryExecution)this.queries.get(queryId);
        if (query == null) {
            throw new NoSuchElementException();
        }
        return query.getQueryInfo();
    }

    @Override
    public Optional<QueryState> getQueryState(QueryId queryId) {
        Objects.requireNonNull(queryId, "queryId is null");
        return Optional.ofNullable(this.queries.get(queryId)).map(QueryExecution::getState);
    }

    @Override
    public void recordHeartbeat(QueryId queryId) {
        Objects.requireNonNull(queryId, "queryId is null");
        QueryExecution query = (QueryExecution)this.queries.get(queryId);
        if (query == null) {
            return;
        }
        query.recordHeartbeat();
    }

    @Override
    public QueryInfo createQuery(Session session, String query) {
        QueryExecution.QueryExecutionFactory<?> queryExecutionFactory;
        Statement statement;
        Objects.requireNonNull(query, "query is null");
        Preconditions.checkArgument((!query.isEmpty() ? 1 : 0) != 0, (Object)"query must not be empty string");
        QueryId queryId = session.getQueryId();
        try {
            statement = this.sqlParser.createStatement(query);
            queryExecutionFactory = this.executionFactories.get(statement.getClass());
            if (queryExecutionFactory == null) {
                throw new PrestoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Unsupported statement type: " + statement.getClass().getSimpleName());
            }
        }
        catch (PrestoException | ParsingException e) {
            URI self = this.locationFactory.createQueryLocation(queryId);
            FailedQueryExecution execution = new FailedQueryExecution(queryId, query, session, self, this.queryExecutor, e);
            this.queries.put(queryId, execution);
            this.queryMonitor.createdEvent(execution.getQueryInfo());
            this.queryMonitor.completionEvent(execution.getQueryInfo());
            this.stats.queryFinished(execution.getQueryInfo());
            this.expirationQueue.add(execution);
            return execution.getQueryInfo();
        }
        Object queryExecution = queryExecutionFactory.createQueryExecution(queryId, query, session, statement);
        this.queryMonitor.createdEvent(queryExecution.getQueryInfo());
        queryExecution.addStateChangeListener(newValue -> {
            if (newValue.isDone()) {
                QueryInfo info = queryExecution.getQueryInfo();
                this.stats.queryFinished(info);
                this.queryMonitor.completionEvent(info);
                this.expirationQueue.add((QueryExecution)queryExecution);
            }
        });
        this.queries.put(queryId, (QueryExecution)queryExecution);
        if (!this.queueManager.submit((QueryExecution)queryExecution, this.queryExecutor, this.stats)) {
            queryExecution.fail(new PrestoException((ErrorCodeSupplier)StandardErrorCode.QUERY_QUEUE_FULL, "Too many queued queries!"));
        }
        return queryExecution.getQueryInfo();
    }

    @Override
    public void cancelQuery(QueryId queryId) {
        Objects.requireNonNull(queryId, "queryId is null");
        log.debug("Cancel query %s", new Object[]{queryId});
        QueryExecution query = (QueryExecution)this.queries.get(queryId);
        if (query != null) {
            query.fail(new PrestoException((ErrorCodeSupplier)StandardErrorCode.USER_CANCELED, "Query was canceled"));
        }
    }

    @Override
    public void cancelStage(StageId stageId) {
        Objects.requireNonNull(stageId, "stageId is null");
        log.debug("Cancel stage %s", new Object[]{stageId});
        QueryExecution query = (QueryExecution)this.queries.get(stageId.getQueryId());
        if (query != null) {
            query.cancelStage(stageId);
        }
    }

    @Managed
    @Flatten
    public SqlQueryManagerStats getStats() {
        return this.stats;
    }

    @Managed(description="Query scheduler executor")
    @Nested
    public ThreadPoolExecutorMBean getExecutor() {
        return this.queryExecutorMBean;
    }

    @Managed(description="Query garbage collector executor")
    @Nested
    public ThreadPoolExecutorMBean getManagementExecutor() {
        return this.queryManagementExecutorMBean;
    }

    public void enforceMemoryLimits() {
        this.memoryManager.process((Iterable)this.queries.values().stream().filter(query -> query.getQueryInfo().getState() == QueryState.RUNNING).collect(ImmutableCollectors.toImmutableList()));
    }

    public void enforceQueryMaxRunTimeLimits() {
        for (QueryExecution query : this.queries.values()) {
            if (query.getQueryInfo().getState().isDone()) continue;
            Duration queryMaxRunTime = SystemSessionProperties.getQueryMaxRunTime(query.getSession());
            DateTime executionStartTime = query.getQueryInfo().getQueryStats().getCreateTime();
            if (!executionStartTime.plus(queryMaxRunTime.toMillis()).isBeforeNow()) continue;
            query.fail(new PrestoException((ErrorCodeSupplier)StandardErrorCode.EXCEEDED_TIME_LIMIT, "Query exceeded maximum time limit of " + queryMaxRunTime));
        }
    }

    private void pruneExpiredQueries() {
        if (this.expirationQueue.size() <= this.maxQueryHistory) {
            return;
        }
        int count = 0;
        for (QueryExecution query : this.expirationQueue) {
            if (this.expirationQueue.size() - count <= this.maxQueryHistory) break;
            query.pruneInfo();
            ++count;
        }
    }

    private void removeExpiredQueries() {
        DateTime timeHorizon = DateTime.now().minus(this.maxQueryAge.toMillis());
        while (this.expirationQueue.size() > this.maxQueryHistory) {
            QueryInfo queryInfo = this.expirationQueue.peek().getQueryInfo();
            if (queryInfo.getQueryStats().getEndTime().isAfter((ReadableInstant)timeHorizon)) {
                return;
            }
            QueryId queryId = queryInfo.getQueryId();
            log.debug("Remove query %s", new Object[]{queryId});
            this.queries.remove(queryId);
            this.expirationQueue.remove();
        }
    }

    public void failAbandonedQueries() {
        for (QueryExecution queryExecution : this.queries.values()) {
            QueryInfo queryInfo = queryExecution.getQueryInfo();
            if (queryInfo.getState().isDone() || !this.isAbandoned(queryExecution)) continue;
            log.info("Failing abandoned query %s", new Object[]{queryExecution.getQueryInfo().getQueryId()});
            queryExecution.fail((Throwable)((Object)new AbandonedException("Query " + queryInfo.getQueryId(), queryInfo.getQueryStats().getLastHeartbeat(), DateTime.now())));
        }
    }

    private boolean isAbandoned(QueryExecution query) {
        DateTime oldestAllowedHeartbeat = DateTime.now().minus(this.clientTimeout.toMillis());
        DateTime lastHeartbeat = query.getQueryInfo().getQueryStats().getLastHeartbeat();
        return lastHeartbeat != null && lastHeartbeat.isBefore((ReadableInstant)oldestAllowedHeartbeat);
    }

    static void addCompletionCallback(QueryExecution queryExecution, Runnable callback) {
        AtomicBoolean taskExecuted = new AtomicBoolean();
        queryExecution.addStateChangeListener(newValue -> {
            if (newValue.isDone() && taskExecuted.compareAndSet(false, true)) {
                callback.run();
            }
        });
        if (queryExecution.getQueryInfo().getState().isDone() && taskExecuted.compareAndSet(false, true)) {
            callback.run();
        }
    }
}

