/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.execution;

import com.facebook.presto.execution.NodeSchedulerConfig;
import com.facebook.presto.execution.NodeTaskMap;
import com.facebook.presto.execution.RemoteTask;
import com.facebook.presto.execution.ResettableRandomizedIterator;
import com.facebook.presto.metadata.Split;
import com.facebook.presto.spi.ErrorCodeSupplier;
import com.facebook.presto.spi.HostAddress;
import com.facebook.presto.spi.Node;
import com.facebook.presto.spi.NodeManager;
import com.facebook.presto.spi.StandardErrorCode;
import com.facebook.presto.util.Failures;
import com.facebook.presto.util.ImmutableCollectors;
import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSetMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.SetMultimap;
import com.google.common.net.InetAddresses;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import javax.inject.Inject;
import org.weakref.jmx.Managed;

public class NodeScheduler {
    private final String coordinatorNodeId;
    private final NodeManager nodeManager;
    private final AtomicLong scheduleLocal = new AtomicLong();
    private final AtomicLong scheduleRack = new AtomicLong();
    private final AtomicLong scheduleRandom = new AtomicLong();
    private final int minCandidates;
    private final boolean locationAwareScheduling;
    private final boolean includeCoordinator;
    private final int maxSplitsPerNode;
    private final int maxSplitsPerNodePerTaskWhenFull;
    private final NodeTaskMap nodeTaskMap;
    private final boolean doubleScheduling;

    @Inject
    public NodeScheduler(NodeManager nodeManager, NodeSchedulerConfig config, NodeTaskMap nodeTaskMap) {
        this.nodeManager = nodeManager;
        this.coordinatorNodeId = nodeManager.getCurrentNode().getNodeIdentifier();
        this.minCandidates = config.getMinCandidates();
        this.locationAwareScheduling = config.isLocationAwareSchedulingEnabled();
        this.includeCoordinator = config.isIncludeCoordinator();
        this.doubleScheduling = config.isMultipleTasksPerNodeEnabled();
        this.maxSplitsPerNode = config.getMaxSplitsPerNode();
        this.maxSplitsPerNodePerTaskWhenFull = config.getMaxPendingSplitsPerNodePerTask();
        this.nodeTaskMap = (NodeTaskMap)Preconditions.checkNotNull((Object)nodeTaskMap, (Object)"nodeTaskMap is null");
        Preconditions.checkArgument((this.maxSplitsPerNode > this.maxSplitsPerNodePerTaskWhenFull ? 1 : 0) != 0, (Object)"maxSplitsPerNode must be > maxSplitsPerNodePerTaskWhenFull");
    }

    @Managed
    public long getScheduleLocal() {
        return this.scheduleLocal.get();
    }

    @Managed
    public long getScheduleRack() {
        return this.scheduleRack.get();
    }

    @Managed
    public long getScheduleRandom() {
        return this.scheduleRandom.get();
    }

    @Managed
    public void reset() {
        this.scheduleLocal.set(0L);
        this.scheduleRack.set(0L);
        this.scheduleRandom.set(0L);
    }

    public NodeSelector createNodeSelector(String dataSourceName) {
        Supplier nodeMap = Suppliers.memoizeWithExpiration(() -> {
            ImmutableSetMultimap.Builder byHostAndPort = ImmutableSetMultimap.builder();
            ImmutableSetMultimap.Builder byHost = ImmutableSetMultimap.builder();
            ImmutableSetMultimap.Builder byRack = ImmutableSetMultimap.builder();
            Set nodes = dataSourceName != null ? this.nodeManager.getActiveDatasourceNodes(dataSourceName) : this.nodeManager.getActiveNodes();
            for (Node node : nodes) {
                try {
                    byHostAndPort.put((Object)node.getHostAndPort(), (Object)node);
                    InetAddress host = InetAddress.getByName(node.getHttpUri().getHost());
                    byHost.put((Object)host, (Object)node);
                    byRack.put((Object)Rack.of(host), (Object)node);
                }
                catch (UnknownHostException unknownHostException) {}
            }
            return new NodeMap((SetMultimap<HostAddress, Node>)byHostAndPort.build(), (SetMultimap<InetAddress, Node>)byHost.build(), (SetMultimap<Rack, Node>)byRack.build());
        }, (long)5L, (TimeUnit)TimeUnit.SECONDS);
        return new NodeSelector((Supplier<NodeMap>)nodeMap);
    }

    private static class Rack {
        private final int id;

        public static Rack of(InetAddress address) {
            int id = InetAddresses.coerceToInteger((InetAddress)address) & 0xFFFFFF00;
            return new Rack(id);
        }

        private Rack(int id) {
            this.id = id;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Rack rack = (Rack)o;
            return this.id == rack.id;
        }

        public int hashCode() {
            return this.id;
        }
    }

    private static class NodeMap {
        private final SetMultimap<HostAddress, Node> nodesByHostAndPort;
        private final SetMultimap<InetAddress, Node> nodesByHost;
        private final SetMultimap<Rack, Node> nodesByRack;

        public NodeMap(SetMultimap<HostAddress, Node> nodesByHostAndPort, SetMultimap<InetAddress, Node> nodesByHost, SetMultimap<Rack, Node> nodesByRack) {
            this.nodesByHostAndPort = nodesByHostAndPort;
            this.nodesByHost = nodesByHost;
            this.nodesByRack = nodesByRack;
        }

        private SetMultimap<HostAddress, Node> getNodesByHostAndPort() {
            return this.nodesByHostAndPort;
        }

        public SetMultimap<InetAddress, Node> getNodesByHost() {
            return this.nodesByHost;
        }

        public SetMultimap<Rack, Node> getNodesByRack() {
            return this.nodesByRack;
        }
    }

    public class NodeSelector {
        private final AtomicReference<Supplier<NodeMap>> nodeMap;

        public NodeSelector(Supplier<NodeMap> nodeMap) {
            this.nodeMap = new AtomicReference<Supplier<NodeMap>>(nodeMap);
        }

        public void lockDownNodes() {
            this.nodeMap.set((Supplier<NodeMap>)Suppliers.ofInstance((Object)this.nodeMap.get().get()));
        }

        public List<Node> allNodes() {
            return ImmutableList.copyOf((Collection)((NodeMap)this.nodeMap.get().get()).getNodesByHostAndPort().values());
        }

        public Node selectCurrentNode() {
            return NodeScheduler.this.nodeManager.getCurrentNode();
        }

        public List<Node> selectRandomNodes(int limit) {
            return this.selectNodes(limit, this.randomizedNodes());
        }

        private List<Node> selectNodes(int limit, Iterator<Node> candidates) {
            Preconditions.checkArgument((limit > 0 ? 1 : 0) != 0, (Object)"limit must be at least 1");
            ArrayList<Node> selected = new ArrayList<Node>(limit);
            while (selected.size() < limit && candidates.hasNext()) {
                selected.add(candidates.next());
            }
            if (NodeScheduler.this.doubleScheduling && !selected.isEmpty()) {
                int uniqueNodes = selected.size();
                int i = 0;
                while (selected.size() < limit) {
                    if (i >= uniqueNodes) {
                        i = 0;
                    }
                    selected.add((Node)selected.get(i));
                    ++i;
                }
            }
            return selected;
        }

        public Multimap<Node, Split> computeAssignments(Set<Split> splits, Iterable<RemoteTask> existingTasks) {
            HashMultimap assignment = HashMultimap.create();
            HashMap<Node, Integer> assignmentCount = new HashMap<Node, Integer>();
            for (Node node : ((NodeMap)this.nodeMap.get().get()).getNodesByHostAndPort().values()) {
                assignmentCount.put(node, 0);
            }
            HashMap<Node, Integer> splitCountByNode = new HashMap<Node, Integer>();
            HashMap<String, Integer> queuedSplitCountByNode = new HashMap<String, Integer>();
            for (RemoteTask task : existingTasks) {
                String nodeId = task.getNodeId();
                queuedSplitCountByNode.put(nodeId, queuedSplitCountByNode.getOrDefault(nodeId, 0) + task.getQueuedPartitionedSplitCount());
            }
            ResettableRandomizedIterator<Node> randomCandidates = null;
            if (!NodeScheduler.this.locationAwareScheduling) {
                randomCandidates = this.randomizedNodes();
            }
            for (Split split : splits) {
                List<Node> candidateNodes;
                if (NodeScheduler.this.locationAwareScheduling || !split.isRemotelyAccessible()) {
                    candidateNodes = this.selectCandidateNodes((NodeMap)this.nodeMap.get().get(), split);
                } else {
                    randomCandidates.reset();
                    candidateNodes = this.selectNodes(NodeScheduler.this.minCandidates, randomCandidates);
                }
                Failures.checkCondition(!candidateNodes.isEmpty(), (ErrorCodeSupplier)StandardErrorCode.NO_NODES_AVAILABLE, "No nodes available to run query", new Object[0]);
                for (Node node : candidateNodes) {
                    if (splitCountByNode.containsKey(node)) continue;
                    splitCountByNode.put(node, NodeScheduler.this.nodeTaskMap.getPartitionedSplitsOnNode(node));
                }
                Node chosenNode = null;
                int min = Integer.MAX_VALUE;
                for (Node node : candidateNodes) {
                    int totalSplitCount = assignmentCount.getOrDefault(node, 0) + (Integer)splitCountByNode.get(node);
                    if (totalSplitCount >= min || totalSplitCount >= NodeScheduler.this.maxSplitsPerNode) continue;
                    chosenNode = node;
                    min = totalSplitCount;
                }
                if (chosenNode == null) {
                    for (Node node : candidateNodes) {
                        int assignedSplitCount = assignmentCount.getOrDefault(node, 0);
                        int queuedSplitCount = queuedSplitCountByNode.getOrDefault(node.getNodeIdentifier(), 0);
                        int totalSplitCount = queuedSplitCount + assignedSplitCount;
                        if (totalSplitCount >= min || totalSplitCount >= NodeScheduler.this.maxSplitsPerNodePerTaskWhenFull) continue;
                        chosenNode = node;
                        min = totalSplitCount;
                    }
                }
                if (chosenNode == null) continue;
                assignment.put(chosenNode, (Object)split);
                assignmentCount.put(chosenNode, assignmentCount.getOrDefault(chosenNode, 0) + 1);
            }
            return assignment;
        }

        private ResettableRandomizedIterator<Node> randomizedNodes() {
            ImmutableList nodes = ((NodeMap)this.nodeMap.get().get()).getNodesByHostAndPort().values().stream().filter(node -> NodeScheduler.this.includeCoordinator || !NodeScheduler.this.coordinatorNodeId.equals(node.getNodeIdentifier())).collect(ImmutableCollectors.toImmutableList());
            return new ResettableRandomizedIterator<Node>((Collection<Node>)nodes);
        }

        private List<Node> selectCandidateNodes(NodeMap nodeMap, Split split) {
            InetAddress address;
            LinkedHashSet<Node> chosen = new LinkedHashSet<Node>(NodeScheduler.this.minCandidates);
            String coordinatorIdentifier = NodeScheduler.this.nodeManager.getCurrentNode().getNodeIdentifier();
            for (HostAddress hint : split.getAddresses()) {
                nodeMap.getNodesByHostAndPort().get((Object)hint).stream().filter(node -> NodeScheduler.this.includeCoordinator || !coordinatorIdentifier.equals(node.getNodeIdentifier())).filter(chosen::add).forEach(node -> NodeScheduler.this.scheduleLocal.incrementAndGet());
                try {
                    address = hint.toInetAddress();
                }
                catch (UnknownHostException e) {
                    continue;
                }
                if (hint.hasPort()) {
                    if (!split.isRemotelyAccessible()) continue;
                }
                nodeMap.getNodesByHost().get((Object)address).stream().filter(node -> NodeScheduler.this.includeCoordinator || !coordinatorIdentifier.equals(node.getNodeIdentifier())).filter(chosen::add).forEach(node -> NodeScheduler.this.scheduleLocal.incrementAndGet());
            }
            if (split.isRemotelyAccessible() && chosen.size() < NodeScheduler.this.minCandidates) {
                for (HostAddress hint : split.getAddresses()) {
                    try {
                        address = hint.toInetAddress();
                    }
                    catch (UnknownHostException e) {
                        continue;
                    }
                    for (Node node2 : nodeMap.getNodesByRack().get((Object)Rack.of(address))) {
                        if (!NodeScheduler.this.includeCoordinator && coordinatorIdentifier.equals(node2.getNodeIdentifier())) continue;
                        if (chosen.add(node2)) {
                            NodeScheduler.this.scheduleRack.incrementAndGet();
                        }
                        if (chosen.size() != NodeScheduler.this.minCandidates) continue;
                        break;
                    }
                    if (chosen.size() != NodeScheduler.this.minCandidates) continue;
                    break;
                }
            }
            if (split.isRemotelyAccessible() && chosen.size() < NodeScheduler.this.minCandidates) {
                ResettableRandomizedIterator<Node> randomizedIterator = this.randomizedNodes();
                while (randomizedIterator.hasNext()) {
                    Node node3 = randomizedIterator.next();
                    if (chosen.add(node3)) {
                        NodeScheduler.this.scheduleRandom.incrementAndGet();
                    }
                    if (chosen.size() != NodeScheduler.this.minCandidates) continue;
                    break;
                }
            }
            if (chosen.isEmpty() && !NodeScheduler.this.includeCoordinator) {
                HostAddress coordinatorHostAddress = NodeScheduler.this.nodeManager.getCurrentNode().getHostAndPort();
                if (split.getAddresses().stream().anyMatch(host -> this.canSplitRunOnHost(split, coordinatorHostAddress, (HostAddress)host))) {
                    chosen.add(NodeScheduler.this.nodeManager.getCurrentNode());
                }
            }
            return ImmutableList.copyOf(chosen);
        }

        private boolean canSplitRunOnHost(Split split, HostAddress coordinatorHost, HostAddress host) {
            if (host.equals((Object)coordinatorHost)) {
                return true;
            }
            return (!host.hasPort() || split.isRemotelyAccessible()) && host.getHostText().equals(coordinatorHost.getHostText());
        }
    }
}

