/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.jdbc.internal.airlift.stats.cardinality;

import com.facebook.presto.jdbc.internal.airlift.slice.BasicSliceInput;
import com.facebook.presto.jdbc.internal.airlift.slice.DynamicSliceOutput;
import com.facebook.presto.jdbc.internal.airlift.slice.SizeOf;
import com.facebook.presto.jdbc.internal.airlift.slice.Slice;
import com.facebook.presto.jdbc.internal.airlift.stats.cardinality.BiasCorrection;
import com.facebook.presto.jdbc.internal.airlift.stats.cardinality.Format;
import com.facebook.presto.jdbc.internal.airlift.stats.cardinality.HllInstance;
import com.facebook.presto.jdbc.internal.airlift.stats.cardinality.Utils;
import com.facebook.presto.jdbc.internal.guava.base.Preconditions;
import com.facebook.presto.jdbc.internal.jol.info.ClassLayout;
import javax.annotation.concurrent.NotThreadSafe;

@NotThreadSafe
final class DenseHll
implements HllInstance {
    private static final double LINEAR_COUNTING_MIN_EMPTY_BUCKETS = 0.4;
    private static final int BITS_PER_BUCKET = 4;
    private static final int MAX_DELTA = 15;
    private static final int BUCKET_MASK = 15;
    private static final int DENSE_INSTANCE_SIZE = ClassLayout.parseClass(DenseHll.class).instanceSize();
    private final byte indexBitLength;
    private byte baseline;
    private short baselineCount;
    private final byte[] deltas;
    private short overflowBucket = (short)-1;
    private byte overflowValue;

    public DenseHll(int indexBitLength) {
        Preconditions.checkArgument(indexBitLength >= 1 && indexBitLength <= 16, "indexBitLength is out of range");
        int numberOfBuckets = 1 << indexBitLength;
        this.indexBitLength = (byte)indexBitLength;
        this.baselineCount = (short)numberOfBuckets;
        this.deltas = new byte[numberOfBuckets * 4 / 8];
    }

    public DenseHll(Slice serialized) {
        BasicSliceInput input = serialized.getInput();
        Preconditions.checkArgument(input.readByte() == Format.DENSE_V1.getTag(), "invalid format tag");
        this.indexBitLength = input.readByte();
        Preconditions.checkArgument(this.indexBitLength >= 1 && this.indexBitLength <= 16, "indexBitLength is out of range");
        this.baseline = input.readByte();
        this.deltas = new byte[Utils.numberOfBuckets(this.indexBitLength) / 2];
        input.readBytes(this.deltas);
        this.overflowBucket = input.readShort();
        this.overflowValue = input.readByte();
        this.baselineCount = 0;
        for (int i = 0; i < Utils.numberOfBuckets(this.indexBitLength); ++i) {
            if (this.getDelta(i) != 0) continue;
            this.baselineCount = (short)(this.baselineCount + 1);
        }
        Preconditions.checkArgument(!input.isReadable(), "input is too big");
    }

    public static boolean canDeserialize(Slice serialized) {
        return serialized.getByte(0) == Format.DENSE_V1.getTag();
    }

    @Override
    public void insertHash(long hash) {
        int index = Utils.computeIndex(hash, this.indexBitLength);
        int value = Utils.computeValue(hash, this.indexBitLength);
        this.insert(index, value);
    }

    @Override
    public int estimatedInMemorySize() {
        return (int)((long)DENSE_INSTANCE_SIZE + SizeOf.sizeOf(this.deltas));
    }

    @Override
    public int getIndexBitLength() {
        return this.indexBitLength;
    }

    @Override
    public long cardinality() {
        int numberOfBuckets = Utils.numberOfBuckets(this.indexBitLength);
        if (this.baseline == 0 && (double)this.baselineCount > 0.4 * (double)numberOfBuckets) {
            return Math.round(Utils.linearCounting(this.baselineCount, numberOfBuckets));
        }
        double sum = 0.0;
        for (int i = 0; i < numberOfBuckets; ++i) {
            int value = this.getValue(i);
            sum += 1.0 / (double)(1L << value);
        }
        double estimate = Utils.alpha(this.indexBitLength) * (double)numberOfBuckets * (double)numberOfBuckets / sum;
        estimate = this.correctBias(estimate);
        return Math.round(estimate);
    }

    private double correctBias(double rawEstimate) {
        double bias;
        double[] estimates = BiasCorrection.RAW_ESTIMATES[this.indexBitLength - 4];
        if (rawEstimate < estimates[0] || rawEstimate > estimates[estimates.length - 1]) {
            return rawEstimate;
        }
        double[] biases = BiasCorrection.BIAS[this.indexBitLength - 4];
        int position = this.search(rawEstimate, estimates);
        if (position >= 0) {
            bias = biases[position];
        } else {
            int insertionPoint = -(position + 1);
            double x0 = estimates[insertionPoint - 1];
            double y0 = biases[insertionPoint - 1];
            double x1 = estimates[insertionPoint];
            double y1 = biases[insertionPoint];
            bias = (rawEstimate - x0) * (y1 - y0) / (x1 - x0) + y0;
        }
        return rawEstimate - bias;
    }

    private int search(double rawEstimate, double[] estimateCurve) {
        int low = 0;
        int high = estimateCurve.length - 1;
        while (low <= high) {
            int middle = low + high >>> 1;
            double middleValue = estimateCurve[middle];
            if (rawEstimate > middleValue) {
                low = middle + 1;
                continue;
            }
            if (rawEstimate < middleValue) {
                high = middle - 1;
                continue;
            }
            return middle;
        }
        return -(low + 1);
    }

    public void insert(int index, int value) {
        int delta = value - this.baseline;
        int oldDelta = this.getDelta(index);
        if (delta <= oldDelta) {
            return;
        }
        if (delta > 15) {
            int overflow = delta - 15;
            if (overflow > this.overflowValue) {
                this.overflowBucket = (short)index;
                this.overflowValue = (byte)overflow;
            }
            delta = 15;
        }
        this.setDelta(index, delta);
        if (oldDelta == 0) {
            this.baselineCount = (short)(this.baselineCount - 1);
            this.adjustBaselineIfNeeded();
        }
    }

    @Override
    public Slice serialize() {
        int size = this.estimatedSerializedSize();
        return new DynamicSliceOutput(size).appendByte(Format.DENSE_V1.getTag()).appendByte(this.indexBitLength).appendByte(this.baseline).appendBytes(this.deltas).appendShort(this.overflowBucket).appendByte(this.overflowValue).slice();
    }

    @Override
    public DenseHll toDense() {
        return this;
    }

    @Override
    public int estimatedSerializedSize() {
        return 3 + Utils.numberOfBuckets(this.indexBitLength) * 1 / 2 + 2 + 1;
    }

    private void setDelta(int bucket, int value) {
        int slot = DenseHll.bucketToSlot(bucket);
        byte clearMask = (byte)(15 << DenseHll.shiftForBucket(bucket));
        int n = slot;
        this.deltas[n] = (byte)(this.deltas[n] & ~clearMask);
        byte setMask = (byte)(value << DenseHll.shiftForBucket(bucket));
        int n2 = slot;
        this.deltas[n2] = (byte)(this.deltas[n2] | setMask);
    }

    private int getDelta(int bucket) {
        int slot = DenseHll.bucketToSlot(bucket);
        return this.deltas[slot] >> DenseHll.shiftForBucket(bucket) & 0xF;
    }

    private int getValue(int bucket) {
        int result = this.baseline + this.getDelta(bucket);
        if (bucket == this.overflowBucket) {
            result += this.overflowValue;
        }
        return result;
    }

    private void adjustBaselineIfNeeded() {
        while (this.baselineCount == 0) {
            this.baseline = (byte)(this.baseline + 1);
            for (int bucket = 0; bucket < Utils.numberOfBuckets(this.indexBitLength); ++bucket) {
                if (this.overflowBucket == bucket && this.overflowValue > 0) {
                    this.overflowValue = (byte)(this.overflowValue - 1);
                    continue;
                }
                if (this.overflowBucket == bucket) {
                    this.overflowBucket = (short)-1;
                }
                int newDelta = this.getDelta(bucket) - 1;
                this.setDelta(bucket, newDelta);
                if (newDelta != 0) continue;
                this.baselineCount = (short)(this.baselineCount + 1);
            }
        }
    }

    public DenseHll mergeWith(DenseHll other) {
        int newBaseline = Math.max(this.baseline, other.baseline);
        int newBaselineCount = 0;
        int newOverflowBucket = -1;
        int newOverflowValue = 0;
        int numberOfBuckets = 1 << this.indexBitLength;
        for (int i = 0; i < numberOfBuckets; ++i) {
            int value = Math.max(this.getValue(i), other.getValue(i));
            int delta = value - newBaseline;
            if (delta == 0) {
                ++newBaselineCount;
            } else if (delta > 15) {
                int overflow = delta - 15;
                if (overflow > newOverflowValue) {
                    newOverflowBucket = i;
                    newOverflowValue = overflow;
                }
                delta = 15;
            }
            this.setDelta(i, delta);
        }
        this.baseline = (byte)newBaseline;
        this.baselineCount = (short)newBaselineCount;
        this.overflowBucket = (short)newOverflowBucket;
        this.overflowValue = (byte)newOverflowValue;
        return this;
    }

    public static int estimatedInMemorySize(int indexBitLength) {
        return (int)((long)DENSE_INSTANCE_SIZE + SizeOf.sizeOfByteArray(Utils.numberOfBuckets(indexBitLength) / 2));
    }

    private static int bucketToSlot(int bucket) {
        return bucket >> 1;
    }

    private static int shiftForBucket(int bucket) {
        return (~bucket & 1) << 2;
    }

    @Override
    public void verify() {
        int zeroDeltas = 0;
        for (int i = 0; i < Utils.numberOfBuckets(this.indexBitLength); ++i) {
            if (this.getDelta(i) != 0) continue;
            ++zeroDeltas;
        }
        Preconditions.checkState(zeroDeltas == this.baselineCount, "baselineCount (%s) doesn't match number of zero deltas (%s)", this.baselineCount, zeroDeltas);
        if (this.overflowBucket != -1) {
            Preconditions.checkState(this.getDelta(this.overflowBucket) == 15, "delta in bucket %s is less than MAX_DELTA (%s < %s) even though there's an associated overflow entry", this.overflowBucket, this.getDelta(this.overflowBucket), 15);
        }
    }
}

