/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.jdbc.internal.spi.page;

import com.facebook.presto.jdbc.internal.common.Page;
import com.facebook.presto.jdbc.internal.common.block.BlockEncodingSerde;
import com.facebook.presto.jdbc.internal.io.airlift.slice.DynamicSliceOutput;
import com.facebook.presto.jdbc.internal.io.airlift.slice.SizeOf;
import com.facebook.presto.jdbc.internal.io.airlift.slice.Slice;
import com.facebook.presto.jdbc.internal.io.airlift.slice.SliceOutput;
import com.facebook.presto.jdbc.internal.io.airlift.slice.Slices;
import com.facebook.presto.jdbc.internal.javax.annotation.concurrent.NotThreadSafe;
import com.facebook.presto.jdbc.internal.spi.page.PageCodecMarker;
import com.facebook.presto.jdbc.internal.spi.page.PageCompressor;
import com.facebook.presto.jdbc.internal.spi.page.PageDecompressor;
import com.facebook.presto.jdbc.internal.spi.page.PagesSerdeUtil;
import com.facebook.presto.jdbc.internal.spi.page.SerializedPage;
import com.facebook.presto.jdbc.internal.spi.spiller.SpillCipher;
import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.Optional;
import sun.misc.Unsafe;

@NotThreadSafe
public class PagesSerde {
    private static final double MINIMUM_COMPRESSION_RATIO = 0.8;
    private final BlockEncodingSerde blockEncodingSerde;
    private final Optional<PageCompressor> compressor;
    private final Optional<PageDecompressor> decompressor;
    private final Optional<SpillCipher> spillCipher;
    private final boolean checksumEnabled;
    private byte[] compressionBuffer;

    public PagesSerde(BlockEncodingSerde blockEncodingSerde, Optional<PageCompressor> compressor, Optional<PageDecompressor> decompressor, Optional<SpillCipher> spillCipher) {
        this(blockEncodingSerde, compressor, decompressor, spillCipher, false);
    }

    public PagesSerde(BlockEncodingSerde blockEncodingSerde, Optional<PageCompressor> compressor, Optional<PageDecompressor> decompressor, Optional<SpillCipher> spillCipher, boolean checksumEnabled) {
        this.blockEncodingSerde = Objects.requireNonNull(blockEncodingSerde, "blockEncodingSerde is null");
        PagesSerde.checkArgument(compressor.isPresent() == decompressor.isPresent(), "compressor and decompressor must both be present or both be absent");
        this.compressor = Objects.requireNonNull(compressor, "compressor is null");
        this.decompressor = Objects.requireNonNull(decompressor, "decompressor is null");
        this.spillCipher = Objects.requireNonNull(spillCipher, "spillCipher is null");
        PagesSerde.checkState(!spillCipher.isPresent() || !spillCipher.get().isDestroyed(), "spillCipher is already destroyed");
        this.checksumEnabled = checksumEnabled;
    }

    public SerializedPage serialize(Page page) {
        DynamicSliceOutput serializationBuffer = new DynamicSliceOutput(Math.toIntExact(page.getSizeInBytes() + 4L));
        PagesSerdeUtil.writeRawPage(page, serializationBuffer, this.blockEncodingSerde);
        return this.wrapSlice(((SliceOutput)serializationBuffer).slice(), page.getPositionCount());
    }

    public SerializedPage serialize(Slice slice, int positionCount) {
        PagesSerde.checkArgument(slice.isCompact(), "slice is not compact");
        return this.wrapSlice(slice, positionCount);
    }

    public Page deserialize(SerializedPage serializedPage) {
        PagesSerde.checkArgument(serializedPage != null, "serializedPage is null");
        Slice slice = serializedPage.getSlice();
        if (PageCodecMarker.ENCRYPTED.isSet(serializedPage.getPageCodecMarkers())) {
            PagesSerde.checkState(this.spillCipher.isPresent(), "Page is encrypted, but spill cipher is missing");
            slice = Slices.wrappedBuffer(this.spillCipher.get().decrypt(slice.toByteBuffer()));
        }
        if (PageCodecMarker.COMPRESSED.isSet(serializedPage.getPageCodecMarkers())) {
            PagesSerde.checkState(this.decompressor.isPresent(), "Page is compressed, but decompressor is missing");
            int uncompressedSize = serializedPage.getUncompressedSizeInBytes();
            ByteBuffer decompressionBuffer = ByteBuffer.allocate(uncompressedSize);
            this.decompressor.get().decompress(slice.toByteBuffer(), decompressionBuffer);
            decompressionBuffer.flip();
            PagesSerde.checkState(decompressionBuffer.remaining() == uncompressedSize, "page size changed after decompression into decompressionBuffer");
            slice = Slices.wrappedBuffer(decompressionBuffer);
        }
        return PagesSerdeUtil.readRawPage(serializedPage.getPositionCount(), slice.getInput(), this.blockEncodingSerde);
    }

    public long getSizeInBytes() {
        return this.compressionBuffer == null ? 0L : (long)this.compressionBuffer.length;
    }

    public long getRetainedSizeInBytes() {
        return SizeOf.sizeOf(this.compressionBuffer);
    }

    private SerializedPage wrapSlice(Slice slice, int positionCount) {
        int uncompressedSize = slice.length();
        byte markers = PageCodecMarker.none();
        if (this.compressor.isPresent()) {
            int maxCompressedSize = this.compressor.get().maxCompressedLength(uncompressedSize);
            this.compressionBuffer = PagesSerde.ensureCapacity(this.compressionBuffer, maxCompressedSize);
            int compressedSize = this.compressor.get().compress((byte[])slice.getBase(), (int)(slice.getAddress() - (long)Unsafe.ARRAY_BYTE_BASE_OFFSET), uncompressedSize, this.compressionBuffer, 0, maxCompressedSize);
            if ((double)compressedSize / (double)uncompressedSize <= 0.8) {
                slice = Slices.copyOf(Slices.wrappedBuffer(this.compressionBuffer, 0, compressedSize));
                markers = PageCodecMarker.COMPRESSED.set(markers);
            }
        }
        if (this.spillCipher.isPresent()) {
            slice = Slices.wrappedBuffer(this.spillCipher.get().encrypt(slice.toByteBuffer()));
            markers = PageCodecMarker.ENCRYPTED.set(markers);
        } else if (!slice.isCompact()) {
            slice = Slices.copyOf(slice);
        }
        if (this.checksumEnabled) {
            markers = PageCodecMarker.CHECKSUMMED.set(markers);
        }
        return new SerializedPage(slice, markers, positionCount, uncompressedSize);
    }

    private static void checkArgument(boolean condition, String message) {
        if (!condition) {
            throw new IllegalArgumentException(message);
        }
    }

    private static void checkState(boolean test, String message) {
        if (!test) {
            throw new IllegalStateException(message);
        }
    }

    private static byte[] ensureCapacity(byte[] buffer, int capacity) {
        if (buffer == null || buffer.length < capacity) {
            return new byte[capacity];
        }
        return buffer;
    }
}

