/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.jdbc.internal.airlift.stats;

import com.facebook.presto.jdbc.internal.airlift.slice.BasicSliceInput;
import com.facebook.presto.jdbc.internal.airlift.slice.DynamicSliceOutput;
import com.facebook.presto.jdbc.internal.airlift.slice.SizeOf;
import com.facebook.presto.jdbc.internal.airlift.slice.Slice;
import com.facebook.presto.jdbc.internal.airlift.slice.SliceInput;
import com.facebook.presto.jdbc.internal.airlift.slice.SliceOutput;
import com.facebook.presto.jdbc.internal.guava.annotations.VisibleForTesting;
import com.facebook.presto.jdbc.internal.guava.base.Preconditions;
import com.facebook.presto.jdbc.internal.guava.base.Ticker;
import com.facebook.presto.jdbc.internal.guava.collect.ImmutableList;
import com.facebook.presto.jdbc.internal.guava.collect.ImmutableListMultimap;
import com.facebook.presto.jdbc.internal.guava.collect.Iterators;
import com.facebook.presto.jdbc.internal.guava.collect.Multimaps;
import com.facebook.presto.jdbc.internal.guava.collect.Ordering;
import com.facebook.presto.jdbc.internal.guava.collect.PeekingIterator;
import com.facebook.presto.jdbc.internal.guava.util.concurrent.AtomicDouble;
import com.facebook.presto.jdbc.internal.jol.info.ClassLayout;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.concurrent.NotThreadSafe;

@NotThreadSafe
public class QuantileDigest {
    private static final int MAX_BITS = 64;
    private static final int QUANTILE_DIGEST_SIZE = ClassLayout.parseClass(QuantileDigest.class).instanceSize();
    static final long RESCALE_THRESHOLD_SECONDS = 50L;
    static final double ZERO_WEIGHT_THRESHOLD = 1.0E-5;
    private static final int INITIAL_CAPACITY = 1;
    private final double maxError;
    private final Ticker ticker;
    private final double alpha;
    private long landmarkInSeconds;
    private double weightedCount;
    private long max = Long.MIN_VALUE;
    private long min = Long.MAX_VALUE;
    private int root = -1;
    private int nextNode = 0;
    private double[] counts;
    private byte[] levels;
    private long[] values;
    private int[] lefts;
    private int[] rights;
    private int freeCount;
    private int firstFree = -1;

    public QuantileDigest(double maxError) {
        this(maxError, 0.0);
    }

    public QuantileDigest(double maxError, double alpha) {
        this(maxError, alpha, alpha == 0.0 ? QuantileDigest.noOpTicker() : Ticker.systemTicker());
    }

    @VisibleForTesting
    QuantileDigest(double maxError, double alpha, Ticker ticker) {
        Preconditions.checkArgument(maxError >= 0.0 && maxError <= 1.0, "maxError must be in range [0, 1]");
        Preconditions.checkArgument(alpha >= 0.0 && alpha < 1.0, "alpha must be in range [0, 1)");
        this.maxError = maxError;
        this.alpha = alpha;
        this.ticker = ticker;
        this.landmarkInSeconds = TimeUnit.NANOSECONDS.toSeconds(ticker.read());
        this.counts = new double[1];
        this.levels = new byte[1];
        this.values = new long[1];
        this.lefts = new int[1];
        this.rights = new int[1];
        Arrays.fill(this.lefts, -1);
        Arrays.fill(this.rights, -1);
    }

    public QuantileDigest(QuantileDigest other) {
        this.maxError = other.maxError;
        this.alpha = other.alpha;
        this.ticker = this.alpha == 0.0 ? QuantileDigest.noOpTicker() : Ticker.systemTicker();
        this.landmarkInSeconds = other.landmarkInSeconds;
        this.weightedCount = other.weightedCount;
        this.max = other.max;
        this.min = other.min;
        this.root = other.root;
        this.nextNode = other.nextNode;
        this.counts = (double[])other.counts.clone();
        this.levels = (byte[])other.levels.clone();
        this.values = (long[])other.values.clone();
        this.lefts = (int[])other.lefts.clone();
        this.rights = (int[])other.rights.clone();
        this.freeCount = other.freeCount;
        this.firstFree = other.firstFree;
    }

    public QuantileDigest(Slice serialized) {
        int i;
        BasicSliceInput input = new BasicSliceInput(serialized);
        this.maxError = ((SliceInput)input).readDouble();
        this.alpha = ((SliceInput)input).readDouble();
        this.ticker = this.alpha == 0.0 ? QuantileDigest.noOpTicker() : Ticker.systemTicker();
        this.min = ((SliceInput)input).readLong();
        this.max = ((SliceInput)input).readLong();
        int nodeCount = ((SliceInput)input).readInt();
        int numberOfLevels = 64 - Long.numberOfLeadingZeros(this.min ^ this.max) + 1;
        double k = (double)(3 * numberOfLevels) / this.maxError;
        Preconditions.checkArgument((double)nodeCount <= 2.0 * k, "Too many nodes in deserialized tree. Possible corruption");
        this.counts = new double[nodeCount];
        for (i = 0; i < nodeCount; ++i) {
            double count = ((SliceInput)input).readDouble();
            this.weightedCount += count;
            this.counts[i] = count;
        }
        this.levels = new byte[nodeCount];
        for (i = 0; i < nodeCount; ++i) {
            this.levels[i] = ((SliceInput)input).readByte();
        }
        this.values = new long[nodeCount];
        for (i = 0; i < nodeCount; ++i) {
            this.values[i] = ((SliceInput)input).readLong();
        }
        int[] stack = new int[(Integer.highestOneBit(nodeCount - 1) << 1) + 1];
        int top = -1;
        this.lefts = new int[nodeCount];
        this.rights = new int[nodeCount];
        int node = 0;
        while (node < nodeCount) {
            byte flags = ((SliceInput)input).readByte();
            this.rights[node] = (flags & 2) != 0 ? stack[top--] : -1;
            this.lefts[node] = (flags & 1) != 0 ? stack[top--] : -1;
            stack[++top] = node++;
        }
        Preconditions.checkArgument(nodeCount == 0 || top == 0, "Tree is corrupted. Expected a single root node");
        this.root = nodeCount - 1;
        this.nextNode = nodeCount;
    }

    public double getMaxError() {
        return this.maxError;
    }

    public double getAlpha() {
        return this.alpha;
    }

    public void add(long value) {
        this.add(value, 1L);
    }

    public void add(long value, long count) {
        Preconditions.checkArgument(count > 0L, "count must be > 0");
        boolean needsCompression = false;
        double weight = count;
        if (this.alpha > 0.0) {
            long nowInSeconds = TimeUnit.NANOSECONDS.toSeconds(this.ticker.read());
            if (nowInSeconds - this.landmarkInSeconds >= 50L) {
                this.rescale(nowInSeconds);
                needsCompression = true;
            }
            weight = this.weight(nowInSeconds) * (double)count;
        }
        this.max = Math.max(this.max, value);
        this.min = Math.min(this.min, value);
        double previousCount = this.weightedCount;
        this.insert(QuantileDigest.longToBits(value), weight);
        int compressionFactor = this.calculateCompressionFactor();
        if (needsCompression || (long)previousCount / (long)compressionFactor != (long)this.weightedCount / (long)compressionFactor) {
            this.compress();
        }
    }

    public void merge(QuantileDigest other) {
        this.rescaleToCommonLandmark(this, other);
        this.root = this.merge(this.root, other, other.root);
        this.max = Math.max(this.max, other.max);
        this.min = Math.min(this.min, other.min);
        this.compress();
    }

    public List<Long> getQuantilesLowerBound(List<Double> quantiles) {
        Preconditions.checkArgument(Ordering.natural().isOrdered(quantiles), "quantiles must be sorted in increasing order");
        for (double quantile : quantiles) {
            Preconditions.checkArgument(quantile >= 0.0 && quantile <= 1.0, "quantile must be between [0,1]");
        }
        ImmutableList<Double> reversedQuantiles = ImmutableList.copyOf(quantiles).reverse();
        final ImmutableList.Builder builder = ImmutableList.builder();
        final PeekingIterator iterator = Iterators.peekingIterator(reversedQuantiles.iterator());
        this.postOrderTraversal(this.root, new Callback(){
            private double sum;

            @Override
            public boolean process(int node) {
                this.sum += QuantileDigest.this.counts[node];
                while (iterator.hasNext() && this.sum > (1.0 - (Double)iterator.peek()) * QuantileDigest.this.weightedCount) {
                    iterator.next();
                    long value = Math.max(QuantileDigest.this.lowerBound(node), QuantileDigest.this.min);
                    builder.add((Object)value);
                }
                return iterator.hasNext();
            }
        }, TraversalOrder.REVERSE);
        while (iterator.hasNext()) {
            builder.add((Object)this.min);
            iterator.next();
        }
        return ((ImmutableList)builder.build()).reverse();
    }

    public List<Long> getQuantilesUpperBound(List<Double> quantiles) {
        Preconditions.checkArgument(Ordering.natural().isOrdered(quantiles), "quantiles must be sorted in increasing order");
        for (double quantile : quantiles) {
            Preconditions.checkArgument(quantile >= 0.0 && quantile <= 1.0, "quantile must be between [0,1]");
        }
        final ImmutableList.Builder builder = ImmutableList.builder();
        final PeekingIterator<Double> iterator = Iterators.peekingIterator(quantiles.iterator());
        this.postOrderTraversal(this.root, new Callback(){
            private double sum = 0.0;

            @Override
            public boolean process(int node) {
                this.sum += QuantileDigest.this.counts[node];
                while (iterator.hasNext() && this.sum > (Double)iterator.peek() * QuantileDigest.this.weightedCount) {
                    iterator.next();
                    long value = Math.min(QuantileDigest.this.upperBound(node), QuantileDigest.this.max);
                    builder.add((Object)value);
                }
                return iterator.hasNext();
            }
        });
        while (iterator.hasNext()) {
            builder.add((Object)this.max);
            iterator.next();
        }
        return builder.build();
    }

    public List<Long> getQuantiles(List<Double> quantiles) {
        return this.getQuantilesUpperBound(quantiles);
    }

    public long getQuantile(double quantile) {
        return this.getQuantiles(ImmutableList.of(Double.valueOf(quantile))).get(0);
    }

    public long getQuantileLowerBound(double quantile) {
        return this.getQuantilesLowerBound(ImmutableList.of(Double.valueOf(quantile))).get(0);
    }

    public long getQuantileUpperBound(double quantile) {
        return this.getQuantilesUpperBound(ImmutableList.of(Double.valueOf(quantile))).get(0);
    }

    public double getCount() {
        return this.weightedCount / this.weight(TimeUnit.NANOSECONDS.toSeconds(this.ticker.read()));
    }

    public List<Bucket> getHistogram(List<Long> bucketUpperBounds) {
        Preconditions.checkArgument(Ordering.natural().isOrdered(bucketUpperBounds), "buckets must be sorted in increasing order");
        ImmutableList.Builder builder = ImmutableList.builder();
        PeekingIterator<Long> iterator = Iterators.peekingIterator(bucketUpperBounds.iterator());
        AtomicDouble sum = new AtomicDouble();
        AtomicDouble lastSum = new AtomicDouble();
        AtomicDouble bucketWeightedSum = new AtomicDouble();
        double normalizationFactor = this.weight(TimeUnit.NANOSECONDS.toSeconds(this.ticker.read()));
        this.postOrderTraversal(this.root, node -> {
            while (iterator.hasNext() && (Long)iterator.peek() <= this.upperBound(node)) {
                double bucketCount = sum.get() - lastSum.get();
                Bucket bucket = new Bucket(bucketCount / normalizationFactor, bucketWeightedSum.get() / bucketCount);
                builder.add(bucket);
                lastSum.set(sum.get());
                bucketWeightedSum.set(0.0);
                iterator.next();
            }
            bucketWeightedSum.addAndGet((double)this.middle(node) * this.counts[node]);
            sum.addAndGet(this.counts[node]);
            return iterator.hasNext();
        });
        while (iterator.hasNext()) {
            double bucketCount = sum.get() - lastSum.get();
            Bucket bucket = new Bucket(bucketCount / normalizationFactor, bucketWeightedSum.get() / bucketCount);
            builder.add(bucket);
            iterator.next();
        }
        return builder.build();
    }

    public long getMin() {
        AtomicLong chosen = new AtomicLong(this.min);
        this.postOrderTraversal(this.root, node -> {
            if (this.counts[node] >= 1.0E-5) {
                chosen.set(this.lowerBound(node));
                return false;
            }
            return true;
        }, TraversalOrder.FORWARD);
        return Math.max(this.min, chosen.get());
    }

    public long getMax() {
        AtomicLong chosen = new AtomicLong(this.max);
        this.postOrderTraversal(this.root, node -> {
            if (this.counts[node] >= 1.0E-5) {
                chosen.set(this.upperBound(node));
                return false;
            }
            return true;
        }, TraversalOrder.REVERSE);
        return Math.min(this.max, chosen.get());
    }

    public int estimatedInMemorySizeInBytes() {
        return (int)((long)QUANTILE_DIGEST_SIZE + SizeOf.sizeOf(this.counts) + SizeOf.sizeOf(this.levels) + SizeOf.sizeOf(this.values) + SizeOf.sizeOf(this.lefts) + SizeOf.sizeOf(this.rights));
    }

    public int estimatedSerializedSizeInBytes() {
        int nodeSize = 18;
        return 36 + this.getNodeCount() * nodeSize;
    }

    public Slice serialize() {
        this.compress();
        DynamicSliceOutput output = new DynamicSliceOutput(this.estimatedSerializedSizeInBytes());
        ((SliceOutput)output).writeDouble(this.maxError);
        ((SliceOutput)output).writeDouble(this.alpha);
        ((SliceOutput)output).writeLong(this.min);
        ((SliceOutput)output).writeLong(this.max);
        ((SliceOutput)output).writeInt(this.getNodeCount());
        final int[] nodes = new int[this.getNodeCount()];
        this.postOrderTraversal(this.root, new Callback(){
            int index = 0;

            @Override
            public boolean process(int node) {
                nodes[this.index++] = node;
                return true;
            }
        });
        for (int node : nodes) {
            ((SliceOutput)output).writeDouble(this.counts[node]);
        }
        for (int node : nodes) {
            ((SliceOutput)output).writeByte(this.levels[node]);
        }
        for (int node : nodes) {
            ((SliceOutput)output).writeLong(this.values[node]);
        }
        for (int node : nodes) {
            byte flags = 0;
            if (this.lefts[node] != -1) {
                flags = (byte)(flags | 1);
            }
            if (this.rights[node] != -1) {
                flags = (byte)(flags | 2);
            }
            ((SliceOutput)output).writeByte(flags);
        }
        return ((SliceOutput)output).slice();
    }

    @VisibleForTesting
    int getNodeCount() {
        return this.nextNode - this.freeCount;
    }

    @VisibleForTesting
    void compress() {
        double bound = Math.floor(this.weightedCount / (double)this.calculateCompressionFactor());
        this.postOrderTraversal(this.root, node -> {
            boolean shouldCompress;
            int left = this.lefts[node];
            int right = this.rights[node];
            if (left == -1 && right == -1) {
                return true;
            }
            double leftCount = left == -1 ? 0.0 : this.counts[left];
            double rightCount = right == -1 ? 0.0 : this.counts[right];
            boolean bl = shouldCompress = this.counts[node] + leftCount + rightCount < bound;
            if (left != -1 && (shouldCompress || leftCount < 1.0E-5)) {
                this.lefts[node] = this.tryRemove(left);
                int n = node;
                this.counts[n] = this.counts[n] + leftCount;
            }
            if (right != -1 && (shouldCompress || rightCount < 1.0E-5)) {
                this.rights[node] = this.tryRemove(right);
                int n = node;
                this.counts[n] = this.counts[n] + rightCount;
            }
            return true;
        });
        if (this.root != -1 && this.counts[this.root] < 1.0E-5) {
            this.root = this.tryRemove(this.root);
        }
    }

    private double weight(long timestamp) {
        return Math.exp(this.alpha * (double)(timestamp - this.landmarkInSeconds));
    }

    private void rescale(long newLandmarkInSeconds) {
        double factor = Math.exp(-this.alpha * (double)(newLandmarkInSeconds - this.landmarkInSeconds));
        this.weightedCount *= factor;
        int i = 0;
        while (i < this.nextNode) {
            int n = i++;
            this.counts[n] = this.counts[n] * factor;
        }
        this.landmarkInSeconds = newLandmarkInSeconds;
    }

    private int calculateCompressionFactor() {
        if (this.root == -1) {
            return 1;
        }
        return Math.max((int)((double)(this.levels[this.root] + 1) / this.maxError), 1);
    }

    private void insert(long value, double count) {
        if (count < 1.0E-5) {
            return;
        }
        long lastBranch = 0L;
        int parent = -1;
        int current = this.root;
        while (true) {
            if (current == -1) {
                this.setChild(parent, lastBranch, this.createLeaf(value, count));
                return;
            }
            long currentValue = this.values[current];
            byte currentLevel = this.levels[current];
            if (!QuantileDigest.inSameSubtree(value, currentValue, currentLevel)) {
                this.setChild(parent, lastBranch, this.makeSiblings(current, this.createLeaf(value, count)));
                return;
            }
            if (currentLevel == 0 && currentValue == value) {
                int n = current;
                this.counts[n] = this.counts[n] + count;
                this.weightedCount += count;
                return;
            }
            long branch = value & this.getBranchMask(currentLevel);
            parent = current;
            lastBranch = branch;
            if (branch == 0L) {
                current = this.lefts[current];
                continue;
            }
            current = this.rights[current];
        }
    }

    private void setChild(int parent, long branch, int child) {
        if (parent == -1) {
            this.root = child;
        } else if (branch == 0L) {
            this.lefts[parent] = child;
        } else {
            this.rights[parent] = child;
        }
    }

    private int makeSiblings(int first, int second) {
        long firstValue = this.values[first];
        long secondValue = this.values[second];
        int parentLevel = 64 - Long.numberOfLeadingZeros(firstValue ^ secondValue);
        int parent = this.createNode(firstValue, parentLevel, 0.0);
        long branch = firstValue & this.getBranchMask(this.levels[parent]);
        if (branch == 0L) {
            this.lefts[parent] = first;
            this.rights[parent] = second;
        } else {
            this.lefts[parent] = second;
            this.rights[parent] = first;
        }
        return parent;
    }

    private int createLeaf(long value, double count) {
        return this.createNode(value, 0, count);
    }

    private int createNode(long value, int level, double count) {
        int node = this.popFree();
        if (node == -1) {
            if (this.nextNode == this.counts.length) {
                int newSize = this.counts.length + Math.min(this.counts.length, this.calculateCompressionFactor() / 5 + 1);
                this.counts = Arrays.copyOf(this.counts, newSize);
                this.levels = Arrays.copyOf(this.levels, newSize);
                this.values = Arrays.copyOf(this.values, newSize);
                this.lefts = Arrays.copyOf(this.lefts, newSize);
                this.rights = Arrays.copyOf(this.rights, newSize);
            }
            node = this.nextNode++;
        }
        this.weightedCount += count;
        this.values[node] = value;
        this.levels[node] = (byte)level;
        this.counts[node] = count;
        this.lefts[node] = -1;
        this.rights[node] = -1;
        return node;
    }

    private int merge(int node, QuantileDigest other, int otherNode) {
        if (otherNode == -1) {
            return node;
        }
        if (node == -1) {
            return this.copyRecursive(other, otherNode);
        }
        if (!QuantileDigest.inSameSubtree(this.values[node], other.values[otherNode], Math.max(this.levels[node], other.levels[otherNode]))) {
            return this.makeSiblings(node, this.copyRecursive(other, otherNode));
        }
        if (this.levels[node] > other.levels[otherNode]) {
            long branch = other.values[otherNode] & this.getBranchMask(this.levels[node]);
            if (branch == 0L) {
                int left;
                this.lefts[node] = left = this.merge(this.lefts[node], other, otherNode);
            } else {
                int right;
                this.rights[node] = right = this.merge(this.rights[node], other, otherNode);
            }
            return node;
        }
        if (this.levels[node] < other.levels[otherNode]) {
            int right;
            int left;
            long branch = this.values[node] & this.getBranchMask(other.levels[otherNode]);
            if (branch == 0L) {
                left = this.merge(node, other, other.lefts[otherNode]);
                right = this.copyRecursive(other, other.rights[otherNode]);
            } else {
                left = this.copyRecursive(other, other.lefts[otherNode]);
                right = this.merge(node, other, other.rights[otherNode]);
            }
            int result = this.createNode(other.values[otherNode], other.levels[otherNode], other.counts[otherNode]);
            this.lefts[result] = left;
            this.rights[result] = right;
            return result;
        }
        this.weightedCount += other.counts[otherNode];
        int n = node;
        this.counts[n] = this.counts[n] + other.counts[otherNode];
        int left = this.merge(this.lefts[node], other, other.lefts[otherNode]);
        int right = this.merge(this.rights[node], other, other.rights[otherNode]);
        this.lefts[node] = left;
        this.rights[node] = right;
        return node;
    }

    private static boolean inSameSubtree(long bitsA, long bitsB, int level) {
        return level == 64 || bitsA >>> level == bitsB >>> level;
    }

    private int copyRecursive(QuantileDigest other, int otherNode) {
        if (otherNode == -1) {
            return otherNode;
        }
        int node = this.createNode(other.values[otherNode], other.levels[otherNode], other.counts[otherNode]);
        if (other.lefts[otherNode] != -1) {
            int left;
            this.lefts[node] = left = this.copyRecursive(other, other.lefts[otherNode]);
        }
        if (other.rights[otherNode] != -1) {
            int right;
            this.rights[node] = right = this.copyRecursive(other, other.rights[otherNode]);
        }
        return node;
    }

    private int tryRemove(int node) {
        Preconditions.checkArgument(node != -1, "node is -1");
        int left = this.lefts[node];
        int right = this.rights[node];
        if (left == -1 && right == -1) {
            this.remove(node);
            return -1;
        }
        if (left != -1 && right != -1) {
            this.counts[node] = 0.0;
            return node;
        }
        this.remove(node);
        if (left != -1) {
            return left;
        }
        return right;
    }

    private void remove(int node) {
        if (node == this.nextNode - 1) {
            --this.nextNode;
        } else {
            this.pushFree(node);
        }
        if (node == this.root) {
            this.root = -1;
        }
    }

    private void pushFree(int node) {
        this.lefts[node] = this.firstFree;
        this.firstFree = node;
        ++this.freeCount;
    }

    private int popFree() {
        int node = this.firstFree;
        if (node == -1) {
            return node;
        }
        this.firstFree = this.lefts[this.firstFree];
        --this.freeCount;
        return node;
    }

    private void postOrderTraversal(int node, Callback callback) {
        this.postOrderTraversal(node, callback, TraversalOrder.FORWARD);
    }

    private void postOrderTraversal(int node, Callback callback, TraversalOrder order) {
        if (order == TraversalOrder.FORWARD) {
            this.postOrderTraversal(node, callback, this.lefts, this.rights);
        } else {
            this.postOrderTraversal(node, callback, this.rights, this.lefts);
        }
    }

    private boolean postOrderTraversal(int node, Callback callback, int[] lefts, int[] rights) {
        if (node == -1) {
            return false;
        }
        int first = lefts[node];
        int second = rights[node];
        if (first != -1 && !this.postOrderTraversal(first, callback, lefts, rights)) {
            return false;
        }
        if (second != -1 && !this.postOrderTraversal(second, callback, lefts, rights)) {
            return false;
        }
        return callback.process(node);
    }

    public double getConfidenceFactor() {
        return this.computeMaxPathWeight(this.root) * 1.0 / this.weightedCount;
    }

    @VisibleForTesting
    boolean equivalent(QuantileDigest other) {
        return this.getNodeCount() == other.getNodeCount() && this.min == other.min && this.max == other.max && this.weightedCount == other.weightedCount && this.alpha == other.alpha;
    }

    private void rescaleToCommonLandmark(QuantileDigest one, QuantileDigest two) {
        long targetLandmark;
        long nowInSeconds = TimeUnit.NANOSECONDS.toSeconds(this.ticker.read());
        if (nowInSeconds - (targetLandmark = Math.max(one.landmarkInSeconds, two.landmarkInSeconds)) >= 50L) {
            targetLandmark = nowInSeconds;
        }
        if (targetLandmark != one.landmarkInSeconds) {
            one.rescale(targetLandmark);
        }
        if (targetLandmark != two.landmarkInSeconds) {
            two.rescale(targetLandmark);
        }
    }

    private double computeMaxPathWeight(int node) {
        if (node == -1 || this.levels[node] == 0) {
            return 0.0;
        }
        double leftMaxWeight = this.computeMaxPathWeight(this.lefts[node]);
        double rightMaxWeight = this.computeMaxPathWeight(this.rights[node]);
        return Math.max(leftMaxWeight, rightMaxWeight) + this.counts[node];
    }

    @VisibleForTesting
    void validate() {
        AtomicDouble sum = new AtomicDouble();
        AtomicInteger nodeCount = new AtomicInteger();
        Set<Integer> freeSlots = this.computeFreeList();
        Preconditions.checkState(freeSlots.size() == this.freeCount, "Free count (%s) doesn't match actual free slots: %s", this.freeCount, freeSlots.size());
        if (this.root != -1) {
            this.validateStructure(this.root, freeSlots);
            this.postOrderTraversal(this.root, node -> {
                sum.addAndGet(this.counts[node]);
                nodeCount.incrementAndGet();
                return true;
            });
        }
        Preconditions.checkState(Math.abs(sum.get() - this.weightedCount) < 1.0E-5, "Computed weight (%s) doesn't match summary (%s)", (Object)sum.get(), (Object)this.weightedCount);
        Preconditions.checkState(nodeCount.get() == this.getNodeCount(), "Actual node count (%s) doesn't match summary (%s)", nodeCount.get(), this.getNodeCount());
    }

    private void validateStructure(int node, Set<Integer> freeNodes) {
        Preconditions.checkState(this.levels[node] >= 0);
        Preconditions.checkState(!freeNodes.contains(node), "Node is in list of free slots: %s", node);
        if (this.lefts[node] != -1) {
            this.validateBranchStructure(node, this.lefts[node], this.rights[node], true);
            this.validateStructure(this.lefts[node], freeNodes);
        }
        if (this.rights[node] != -1) {
            this.validateBranchStructure(node, this.rights[node], this.lefts[node], false);
            this.validateStructure(this.rights[node], freeNodes);
        }
    }

    private void validateBranchStructure(int parent, int child, int otherChild, boolean isLeft) {
        Preconditions.checkState(this.levels[child] < this.levels[parent], "Child level (%s) should be smaller than parent level (%s)", (int)this.levels[child], (int)this.levels[parent]);
        long branch = this.values[child] & 1L << this.levels[parent] - 1;
        Preconditions.checkState(branch == 0L && isLeft || branch != 0L && !isLeft, "Value of child node is inconsistent with its branch");
        Preconditions.checkState(this.counts[parent] > 0.0 || this.counts[child] > 0.0 || otherChild != -1, "Found a linear chain of zero-weight nodes");
    }

    private Set<Integer> computeFreeList() {
        HashSet<Integer> freeSlots = new HashSet<Integer>();
        int index = this.firstFree;
        while (index != -1) {
            freeSlots.add(index);
            index = this.lefts[index];
        }
        return freeSlots;
    }

    public String toGraphviz() {
        StringBuilder builder = new StringBuilder();
        builder.append("digraph QuantileDigest {\n").append("\tgraph [ordering=\"out\"];");
        ArrayList nodes = new ArrayList();
        this.postOrderTraversal(this.root, node -> {
            nodes.add(node);
            return true;
        });
        ImmutableListMultimap nodesByLevel = Multimaps.index(nodes, input -> this.levels[input]);
        for (Map.Entry entry : nodesByLevel.asMap().entrySet()) {
            builder.append("\tsubgraph level_" + entry.getKey() + " {\n").append("\t\trank = same;\n");
            Iterator iterator = entry.getValue().iterator();
            while (iterator.hasNext()) {
                int node2 = (Integer)iterator.next();
                if (this.levels[node2] == 0) {
                    builder.append(String.format("\t\t%s [label=\"%s:[%s]@%s\\n%s\", shape=rect, style=filled,color=%s];\n", QuantileDigest.idFor(node2), node2, this.lowerBound(node2), this.levels[node2], this.counts[node2], this.counts[node2] > 0.0 ? "salmon2" : "white"));
                    continue;
                }
                builder.append(String.format("\t\t%s [label=\"%s:[%s..%s]@%s\\n%s\", shape=rect, style=filled,color=%s];\n", QuantileDigest.idFor(node2), node2, this.lowerBound(node2), this.upperBound(node2), this.levels[node2], this.counts[node2], this.counts[node2] > 0.0 ? "salmon2" : "white"));
            }
            builder.append("\t}\n");
        }
        Iterator iterator = nodes.iterator();
        while (iterator.hasNext()) {
            int node3 = (Integer)((Object)iterator.next());
            if (this.lefts[node3] != -1) {
                builder.append(String.format("\t%s -> %s [style=\"%s\"];\n", QuantileDigest.idFor(node3), QuantileDigest.idFor(this.lefts[node3]), this.levels[node3] - this.levels[this.lefts[node3]] == 1 ? "solid" : "dotted"));
            }
            if (this.rights[node3] == -1) continue;
            builder.append(String.format("\t%s -> %s [style=\"%s\"];\n", QuantileDigest.idFor(node3), QuantileDigest.idFor(this.rights[node3]), this.levels[node3] - this.levels[this.rights[node3]] == 1 ? "solid" : "dotted"));
        }
        builder.append("}\n");
        return builder.toString();
    }

    private static String idFor(int node) {
        return String.format("node_%x", node);
    }

    private static long longToBits(long value) {
        return value ^ Long.MIN_VALUE;
    }

    private static long bitsToLong(long bits) {
        return bits ^ Long.MIN_VALUE;
    }

    private long getBranchMask(byte level) {
        return 1L << level - 1;
    }

    private long upperBound(int node) {
        long mask = 0L;
        if (this.levels[node] > 0) {
            mask = -1L >>> 64 - this.levels[node];
        }
        return QuantileDigest.bitsToLong(this.values[node] | mask);
    }

    private long lowerBound(int node) {
        long mask = 0L;
        if (this.levels[node] > 0) {
            mask = -1L >>> 64 - this.levels[node];
        }
        return QuantileDigest.bitsToLong(this.values[node] & (mask ^ 0xFFFFFFFFFFFFFFFFL));
    }

    private long middle(int node) {
        long lower = this.lowerBound(node);
        long upper = this.upperBound(node);
        return lower + (upper - lower) / 2L;
    }

    private static Ticker noOpTicker() {
        return new Ticker(){

            @Override
            public long read() {
                return 0L;
            }
        };
    }

    private static class Flags {
        public static final int HAS_LEFT = 1;
        public static final int HAS_RIGHT = 2;

        private Flags() {
        }
    }

    private static interface Callback {
        public boolean process(int var1);
    }

    public static class Bucket {
        private double count;
        private double mean;

        public Bucket(double count, double mean) {
            this.count = count;
            this.mean = mean;
        }

        public double getCount() {
            return this.count;
        }

        public double getMean() {
            return this.mean;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Bucket bucket = (Bucket)o;
            if (Double.compare(bucket.count, this.count) != 0) {
                return false;
            }
            return Double.compare(bucket.mean, this.mean) == 0;
        }

        public int hashCode() {
            long temp = this.count != 0.0 ? Double.doubleToLongBits(this.count) : 0L;
            int result = (int)(temp ^ temp >>> 32);
            temp = this.mean != 0.0 ? Double.doubleToLongBits(this.mean) : 0L;
            result = 31 * result + (int)(temp ^ temp >>> 32);
            return result;
        }

        public String toString() {
            return String.format("[count: %f, mean: %f]", this.count, this.mean);
        }
    }

    private static enum TraversalOrder {
        FORWARD,
        REVERSE;

    }
}

