/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.iceberg;

import com.facebook.presto.common.block.Block;
import com.facebook.presto.common.block.BlockBuilder;
import com.facebook.presto.common.type.BigintType;
import com.facebook.presto.common.type.DateType;
import com.facebook.presto.common.type.DecimalType;
import com.facebook.presto.common.type.Decimals;
import com.facebook.presto.common.type.IntegerType;
import com.facebook.presto.common.type.Type;
import com.facebook.presto.common.type.VarbinaryType;
import com.facebook.presto.common.type.VarcharType;
import io.airlift.slice.Murmur3Hash32;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceUtf8;
import io.airlift.slice.Slices;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Objects;
import java.util.function.Function;
import java.util.function.IntUnaryOperator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.iceberg.PartitionField;
import org.joda.time.DateTimeField;
import org.joda.time.chrono.ISOChronology;

public final class PartitionTransforms {
    private static final Pattern BUCKET_PATTERN = Pattern.compile("bucket\\[(\\d+)]");
    private static final Pattern TRUNCATE_PATTERN = Pattern.compile("truncate\\[(\\d+)]");
    private static final DateTimeField YEAR_FIELD = ISOChronology.getInstanceUTC().year();
    private static final DateTimeField MONTH_FIELD = ISOChronology.getInstanceUTC().monthOfYear();
    private static final DateTimeField DAY_OF_YEAR_FIELD = ISOChronology.getInstanceUTC().dayOfYear();
    private static final DateTimeField DAY_OF_MONTH_FIELD = ISOChronology.getInstanceUTC().dayOfMonth();

    private PartitionTransforms() {
    }

    public static ColumnTransform getColumnTransform(PartitionField field, Type type) {
        String transform = field.transform().toString();
        if (transform.equals("identity")) {
            return new ColumnTransform(type, Function.identity());
        }
        Matcher matcher = BUCKET_PATTERN.matcher(transform);
        if (matcher.matches()) {
            int count = Integer.parseInt(matcher.group(1));
            if (type.equals(IntegerType.INTEGER)) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketInteger(block, count));
            }
            if (type.equals(BigintType.BIGINT)) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketBigint(block, count));
            }
            if (Decimals.isShortDecimal((Type)type)) {
                DecimalType decimal = (DecimalType)type;
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketShortDecimal(decimal, block, count));
            }
            if (Decimals.isLongDecimal((Type)type)) {
                DecimalType decimal = (DecimalType)type;
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketLongDecimal(decimal, block, count));
            }
            if (type.equals(DateType.DATE)) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketDate(block, count));
            }
            if (type instanceof VarcharType) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketVarchar(block, count));
            }
            if (type.equals(VarbinaryType.VARBINARY)) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.bucketVarbinary(block, count));
            }
            throw new UnsupportedOperationException("Unsupported type for 'bucket': " + field);
        }
        matcher = TRUNCATE_PATTERN.matcher(transform);
        if (matcher.matches()) {
            int width = Integer.parseInt(matcher.group(1));
            if (type.equals(IntegerType.INTEGER)) {
                return new ColumnTransform((Type)IntegerType.INTEGER, block -> PartitionTransforms.truncateInteger(block, width));
            }
            if (type.equals(BigintType.BIGINT)) {
                return new ColumnTransform((Type)BigintType.BIGINT, block -> PartitionTransforms.truncateBigint(block, width));
            }
            if (Decimals.isShortDecimal((Type)type)) {
                DecimalType decimal = (DecimalType)type;
                return new ColumnTransform(type, block -> PartitionTransforms.truncateShortDecimal(decimal, block, width));
            }
            if (Decimals.isLongDecimal((Type)type)) {
                DecimalType decimal = (DecimalType)type;
                return new ColumnTransform(type, block -> PartitionTransforms.truncateLongDecimal(decimal, block, width));
            }
            if (type instanceof VarcharType) {
                return new ColumnTransform((Type)VarcharType.VARCHAR, block -> PartitionTransforms.truncateVarchar(block, width));
            }
            if (type.equals(VarbinaryType.VARBINARY)) {
                return new ColumnTransform((Type)VarbinaryType.VARBINARY, block -> PartitionTransforms.truncateVarbinary(block, width));
            }
            throw new UnsupportedOperationException("Unsupported type for 'truncate': " + field);
        }
        throw new UnsupportedOperationException("Unsupported partition transform: " + field);
    }

    private static Block bucketInteger(Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> PartitionTransforms.bucketHash(IntegerType.INTEGER.getLong(block, position)));
    }

    private static Block bucketBigint(Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> PartitionTransforms.bucketHash(BigintType.BIGINT.getLong(block, position)));
    }

    private static Block bucketShortDecimal(DecimalType decimal, Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> {
            BigDecimal value = Decimals.readBigDecimal((DecimalType)decimal, (Block)block, (int)position);
            return PartitionTransforms.bucketHash(Slices.wrappedBuffer((byte[])value.unscaledValue().toByteArray()));
        });
    }

    private static Block bucketLongDecimal(DecimalType decimal, Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> {
            BigDecimal value = Decimals.readBigDecimal((DecimalType)decimal, (Block)block, (int)position);
            return PartitionTransforms.bucketHash(Slices.wrappedBuffer((byte[])value.unscaledValue().toByteArray()));
        });
    }

    private static Block bucketDate(Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> PartitionTransforms.bucketHash(DateType.DATE.getLong(block, position)));
    }

    private static Block bucketVarchar(Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> PartitionTransforms.bucketHash(VarcharType.VARCHAR.getSlice(block, position)));
    }

    private static Block bucketVarbinary(Block block, int count) {
        return PartitionTransforms.bucketBlock(block, count, position -> PartitionTransforms.bucketHash(VarcharType.VARCHAR.getSlice(block, position)));
    }

    private static Block bucketBlock(Block block, int count, IntUnaryOperator hasher) {
        BlockBuilder builder = IntegerType.INTEGER.createFixedSizeBlockBuilder(block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            int hash = hasher.applyAsInt(position);
            int bucket = (hash & Integer.MAX_VALUE) % count;
            IntegerType.INTEGER.writeLong(builder, (long)bucket);
        }
        return builder.build();
    }

    private static int bucketHash(long value) {
        return Murmur3Hash32.hash((long)value);
    }

    private static int bucketHash(Slice value) {
        return Murmur3Hash32.hash((Slice)value);
    }

    private static Block truncateInteger(Block block, int width) {
        BlockBuilder builder = IntegerType.INTEGER.createFixedSizeBlockBuilder(block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            long value = IntegerType.INTEGER.getLong(block, position);
            value -= (value % (long)width + (long)width) % (long)width;
            IntegerType.INTEGER.writeLong(builder, value);
        }
        return builder.build();
    }

    private static Block truncateBigint(Block block, int width) {
        BlockBuilder builder = BigintType.BIGINT.createFixedSizeBlockBuilder(block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            long value = BigintType.BIGINT.getLong(block, position);
            value -= (value % (long)width + (long)width) % (long)width;
            BigintType.BIGINT.writeLong(builder, value);
        }
        return builder.build();
    }

    private static Block truncateShortDecimal(DecimalType type, Block block, int width) {
        BigInteger unscaledWidth = BigInteger.valueOf(width);
        BlockBuilder builder = type.createBlockBuilder(null, block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            BigDecimal value = Decimals.readBigDecimal((DecimalType)type, (Block)block, (int)position);
            value = PartitionTransforms.truncateDecimal(value, unscaledWidth);
            type.writeLong(builder, Decimals.encodeShortScaledValue((BigDecimal)value, (int)type.getScale()));
        }
        return builder.build();
    }

    private static Block truncateLongDecimal(DecimalType type, Block block, int width) {
        BigInteger unscaledWidth = BigInteger.valueOf(width);
        BlockBuilder builder = type.createBlockBuilder(null, block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            BigDecimal value = Decimals.readBigDecimal((DecimalType)type, (Block)block, (int)position);
            value = PartitionTransforms.truncateDecimal(value, unscaledWidth);
            type.writeSlice(builder, Decimals.encodeScaledValue((BigDecimal)value, (int)type.getScale()));
        }
        return builder.build();
    }

    private static BigDecimal truncateDecimal(BigDecimal value, BigInteger unscaledWidth) {
        BigDecimal remainder = new BigDecimal(value.unscaledValue().remainder(unscaledWidth).add(unscaledWidth).remainder(unscaledWidth), value.scale());
        return value.subtract(remainder);
    }

    private static Block truncateVarchar(Block block, int max) {
        BlockBuilder builder = VarcharType.VARCHAR.createBlockBuilder(null, block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            Slice value = VarcharType.VARCHAR.getSlice(block, position);
            value = PartitionTransforms.truncateVarchar(value, max);
            VarcharType.VARCHAR.writeSlice(builder, value);
        }
        return builder.build();
    }

    private static Slice truncateVarchar(Slice value, int max) {
        if (value.length() <= max) {
            return value;
        }
        int end = SliceUtf8.offsetOfCodePoint((Slice)value, (int)0, (int)max);
        if (end < 0) {
            return value;
        }
        return value.slice(0, end);
    }

    private static Block truncateVarbinary(Block block, int max) {
        BlockBuilder builder = VarbinaryType.VARBINARY.createBlockBuilder(null, block.getPositionCount());
        for (int position = 0; position < block.getPositionCount(); ++position) {
            if (block.isNull(position)) {
                builder.appendNull();
                continue;
            }
            Slice value = VarbinaryType.VARBINARY.getSlice(block, position);
            if (value.length() > max) {
                value = value.slice(0, max);
            }
            VarbinaryType.VARBINARY.writeSlice(builder, value);
        }
        return builder.build();
    }

    public static class ColumnTransform {
        private final Type type;
        private final Function<Block, Block> transform;

        public ColumnTransform(Type type, Function<Block, Block> transform) {
            this.type = Objects.requireNonNull(type, "resultType is null");
            this.transform = Objects.requireNonNull(transform, "transform is null");
        }

        public Type getType() {
            return this.type;
        }

        public Function<Block, Block> getTransform() {
            return this.transform;
        }
    }
}

