/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.presto.geospatial;

import com.facebook.presto.geospatial.Rectangle;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.ComparisonChain;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import java.util.ArrayDeque;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;

public class KdbTree {
    private static final int MAX_LEVELS = 10000;
    private final Node root;
    private static final SplitDimension BY_X = new SplitDimension(){
        private final Comparator<Rectangle> comparator = (first, second) -> ComparisonChain.start().compare(first.getXMin(), second.getXMin()).compare(first.getYMin(), second.getYMin()).result();

        @Override
        public Comparator<Rectangle> getComparator() {
            return this.comparator;
        }

        @Override
        public double getValue(Rectangle rectangle) {
            return rectangle.getXMin();
        }

        @Override
        public SplitResult<Rectangle> split(Rectangle rectangle, double x) {
            Preconditions.checkArgument((rectangle.getXMin() < x && x < rectangle.getXMax() ? 1 : 0) != 0);
            return new SplitResult<Rectangle>(new Rectangle(rectangle.getXMin(), rectangle.getYMin(), x, rectangle.getYMax()), new Rectangle(x, rectangle.getYMin(), rectangle.getXMax(), rectangle.getYMax()));
        }
    };
    private static final SplitDimension BY_Y = new SplitDimension(){
        private final Comparator<Rectangle> comparator = (first, second) -> ComparisonChain.start().compare(first.getYMin(), second.getYMin()).compare(first.getXMin(), second.getXMin()).result();

        @Override
        public Comparator<Rectangle> getComparator() {
            return this.comparator;
        }

        @Override
        public double getValue(Rectangle rectangle) {
            return rectangle.getYMin();
        }

        @Override
        public SplitResult<Rectangle> split(Rectangle rectangle, double y) {
            Preconditions.checkArgument((rectangle.getYMin() < y && y < rectangle.getYMax() ? 1 : 0) != 0);
            return new SplitResult<Rectangle>(new Rectangle(rectangle.getXMin(), rectangle.getYMin(), rectangle.getXMax(), y), new Rectangle(rectangle.getXMin(), y, rectangle.getXMax(), rectangle.getYMax()));
        }
    };

    @JsonCreator
    public KdbTree(@JsonProperty(value="root") Node root) {
        this.root = Objects.requireNonNull(root, "root is null");
    }

    @JsonProperty
    public Node getRoot() {
        return this.root;
    }

    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KdbTree)) {
            return false;
        }
        KdbTree other = (KdbTree)obj;
        return this.root.equals(other.root);
    }

    public int hashCode() {
        return Objects.hash(this.root);
    }

    public Map<Integer, Rectangle> getLeaves() {
        return this.findLeaves((Predicate<Node>)((Predicate)node -> true));
    }

    public Map<Integer, Rectangle> findIntersectingLeaves(Rectangle envelope) {
        return this.findLeaves((Predicate<Node>)((Predicate)node -> ((Node)node).extent.getXMin() <= envelope.getXMax() && ((Node)node).extent.getXMax() > envelope.getXMin() && ((Node)node).extent.getYMin() <= envelope.getYMax() && ((Node)node).extent.getYMax() > envelope.getYMin()));
    }

    private Map<Integer, Rectangle> findLeaves(Predicate<Node> predicate) {
        ImmutableMap.Builder leaves = ImmutableMap.builder();
        ArrayDeque<Node> deque = new ArrayDeque<Node>();
        deque.push(this.root);
        while (!deque.isEmpty()) {
            Node node = (Node)deque.pop();
            if (!predicate.apply((Object)node)) continue;
            if (node.leafId.isPresent()) {
                leaves.put((Object)node.leafId.getAsInt(), (Object)node.extent);
                continue;
            }
            deque.push(node.getLeft().get());
            deque.push(node.getRight().get());
        }
        return leaves.build();
    }

    public static KdbTree buildKdbTree(int maxItemsPerNode, List<Rectangle> items) {
        Preconditions.checkArgument((maxItemsPerNode > 0 ? 1 : 0) != 0, (Object)"maxItemsPerNode must be > 0");
        Objects.requireNonNull(items, "items is null");
        return new KdbTree(KdbTree.buildKdbTreeNode(maxItemsPerNode, 0, Rectangle.getUniverseRectangle(), items, new LeafIdAllocator()));
    }

    private static Node buildKdbTreeNode(int maxItemsPerNode, int level, Rectangle extent, List<Rectangle> items, LeafIdAllocator leafIdAllocator) {
        Preconditions.checkArgument((maxItemsPerNode > 0 ? 1 : 0) != 0, (Object)"maxItemsPerNode must be > 0");
        Preconditions.checkArgument((level >= 0 ? 1 : 0) != 0, (Object)"level must be >= 0");
        Preconditions.checkArgument((level <= 10000 ? 1 : 0) != 0, (Object)"level must be <= 10,000");
        Objects.requireNonNull(extent, "extent is null");
        Objects.requireNonNull(items, "items is null");
        if (items.size() <= maxItemsPerNode || level == 10000) {
            return Node.newLeaf(extent, leafIdAllocator.next());
        }
        boolean splitVertically = KdbTree.shouldSplitVertically(extent);
        Optional<SplitResult<Node>> splitResult = KdbTree.trySplit(splitVertically ? BY_X : BY_Y, maxItemsPerNode, level, extent, items, leafIdAllocator);
        if (!splitResult.isPresent()) {
            splitResult = KdbTree.trySplit(splitVertically ? BY_Y : BY_X, maxItemsPerNode, level, extent, items, leafIdAllocator);
        }
        if (!splitResult.isPresent()) {
            return Node.newLeaf(extent, leafIdAllocator.next());
        }
        return Node.newInternal(extent, splitResult.get().getLeft(), splitResult.get().getRight());
    }

    private static boolean shouldSplitVertically(Rectangle extent) {
        int numHorizontalInfinities = 0;
        if (extent.getXMax() == Double.POSITIVE_INFINITY) {
            ++numHorizontalInfinities;
        }
        if (extent.getXMin() == Double.NEGATIVE_INFINITY) {
            ++numHorizontalInfinities;
        }
        int numVerticalInfinities = 0;
        if (extent.getYMax() == Double.POSITIVE_INFINITY) {
            ++numVerticalInfinities;
        }
        if (extent.getYMin() == Double.NEGATIVE_INFINITY) {
            ++numVerticalInfinities;
        }
        if (numHorizontalInfinities == numVerticalInfinities) {
            return extent.getWidth() >= extent.getHeight();
        }
        return numHorizontalInfinities > numVerticalInfinities;
    }

    private static Optional<SplitResult<Node>> trySplit(SplitDimension splitDimension, int maxItemsPerNode, int level, Rectangle extent, List<Rectangle> items, LeafIdAllocator leafIdAllocator) {
        int splitIndex;
        Preconditions.checkArgument((items.size() > 1 ? 1 : 0) != 0, (Object)"Number of items to split must be > 1");
        ImmutableList sortedItems = ImmutableList.sortedCopyOf(splitDimension.getComparator(), items);
        int middleIndex = (sortedItems.size() - 1) / 2;
        Rectangle middleEnvelope = (Rectangle)sortedItems.get(middleIndex);
        double splitValue = splitDimension.getValue(middleEnvelope);
        for (splitIndex = middleIndex; splitIndex < sortedItems.size() && splitDimension.getValue((Rectangle)sortedItems.get(splitIndex)) == splitValue; ++splitIndex) {
        }
        if (splitIndex == sortedItems.size()) {
            return Optional.empty();
        }
        splitValue = (splitValue + splitDimension.getValue((Rectangle)sortedItems.get(splitIndex))) / 2.0;
        SplitResult<Rectangle> childExtents = splitDimension.split(extent, splitValue);
        return Optional.of(new SplitResult(KdbTree.buildKdbTreeNode(maxItemsPerNode, level + 1, childExtents.getLeft(), sortedItems.subList(0, splitIndex), leafIdAllocator), KdbTree.buildKdbTreeNode(maxItemsPerNode, level + 1, childExtents.getRight(), sortedItems.subList(splitIndex, sortedItems.size()), leafIdAllocator)));
    }

    private static final class SplitResult<T> {
        private final T left;
        private final T right;

        private SplitResult(T left, T right) {
            this.left = Objects.requireNonNull(left, "left is null");
            this.right = Objects.requireNonNull(right, "right is null");
        }

        public T getLeft() {
            return this.left;
        }

        public T getRight() {
            return this.right;
        }
    }

    private static final class LeafIdAllocator {
        private int nextId;

        private LeafIdAllocator() {
        }

        public int next() {
            return this.nextId++;
        }
    }

    private static interface SplitDimension {
        public Comparator<Rectangle> getComparator();

        public double getValue(Rectangle var1);

        public SplitResult<Rectangle> split(Rectangle var1, double var2);
    }

    public static final class Node {
        private final Rectangle extent;
        private final OptionalInt leafId;
        private final Optional<Node> left;
        private final Optional<Node> right;

        public static Node newLeaf(Rectangle extent, int leafId) {
            return new Node(extent, OptionalInt.of(leafId), Optional.empty(), Optional.empty());
        }

        public static Node newInternal(Rectangle extent, Node left, Node right) {
            return new Node(extent, OptionalInt.empty(), Optional.of(left), Optional.of(right));
        }

        @JsonCreator
        public Node(@JsonProperty(value="extent") Rectangle extent, @JsonProperty(value="leafId") OptionalInt leafId, @JsonProperty(value="left") Optional<Node> left, @JsonProperty(value="right") Optional<Node> right) {
            this.extent = Objects.requireNonNull(extent, "extent is null");
            this.leafId = Objects.requireNonNull(leafId, "leafId is null");
            this.left = Objects.requireNonNull(left, "left is null");
            this.right = Objects.requireNonNull(right, "right is null");
            if (leafId.isPresent()) {
                Preconditions.checkArgument((leafId.getAsInt() >= 0 ? 1 : 0) != 0, (Object)"leafId must be >= 0");
                Preconditions.checkArgument((!left.isPresent() ? 1 : 0) != 0, (Object)"Leaf node cannot have left child");
                Preconditions.checkArgument((!right.isPresent() ? 1 : 0) != 0, (Object)"Leaf node cannot have right child");
            } else {
                Preconditions.checkArgument((boolean)left.isPresent(), (Object)"Intermediate node must have left child");
                Preconditions.checkArgument((boolean)right.isPresent(), (Object)"Intermediate node must have right child");
            }
        }

        @JsonProperty
        public Rectangle getExtent() {
            return this.extent;
        }

        @JsonProperty
        public OptionalInt getLeafId() {
            return this.leafId;
        }

        @JsonProperty
        public Optional<Node> getLeft() {
            return this.left;
        }

        @JsonProperty
        public Optional<Node> getRight() {
            return this.right;
        }

        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            if (!(obj instanceof Node)) {
                return false;
            }
            Node other = (Node)obj;
            return this.extent.equals(other.extent) && Objects.equals(this.leafId, other.leafId) && Objects.equals(this.left, other.left) && Objects.equals(this.right, other.right);
        }

        public int hashCode() {
            return Objects.hash(this.extent, this.leafId, this.left, this.right);
        }
    }
}

