/*
 * Copyright (c) 2014-present, Facebook, Inc. All rights reserved.
 *
 * You are hereby granted a non-exclusive, worldwide, royalty-free license to use,
 * copy, modify, and distribute this software in source code or binary form for use
 * in connection with the web services and APIs provided by Facebook.
 *
 * As with any software that integrates with the Facebook platform, your use of
 * this software is subject to the Facebook Developer Principles and Policies
 * [http://developers.facebook.com/policy/]. This copyright notice shall be
 * included in all copies or substantial portions of the software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.facebook.marketing.internal;

import android.content.Context;

import com.facebook.GraphRequest;
import com.facebook.FacebookSdk;
import com.facebook.HttpMethod;
import com.facebook.internal.AttributionIdentifiers;

import android.os.Bundle;

import org.json.JSONArray;
import org.json.JSONObject;


import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.Locale;

import androidx.annotation.Nullable;
import android.util.Log;

public class RemoteConfigManager {
    private static final String TAG = RemoteConfigManager.class.getCanonicalName();

    private static final String SAMPLING_ENDPOINT_PATH =
            "%s/button_auto_detection_device_selection";
    private static final String SAMPLING_RESULT_FIELD = "is_selected";
    final private static Map<String, RemoteConfig> remoteConfigs =
            new ConcurrentHashMap<>();

    public static void loadRemoteConfigAsync() {
        FacebookSdk.getExecutor().execute(new Runnable() {
            @Override
            public void run() {
                final Context context = FacebookSdk.getApplicationContext();
                final String applicationId = FacebookSdk.getApplicationId();
                AttributionIdentifiers identifiers =
                        AttributionIdentifiers.getAttributionIdentifiers(context);
                final String advertiserId = identifiers != null ?
                        identifiers.getAndroidAdvertiserId() : null;
                if (advertiserId != null) {
                    JSONObject resultJSON =
                            getRemoteConfigQueryResponse(applicationId);
                    if (resultJSON != null) {
                        parseRemoteConfigFromJSON(applicationId, resultJSON);
                    }
                }
            }
        });
    }

    @Nullable
    public static RemoteConfig getRemoteConfigWithoutQuery(final String applicationId) {
        return applicationId != null ? remoteConfigs.get(applicationId) : null;
    }

    @Nullable
    private static JSONObject getRemoteConfigQueryResponse(
            String applicationId) {
        try {
            String publishUrl = String.format(Locale.US, SAMPLING_ENDPOINT_PATH, applicationId);
            Bundle publishParams = new Bundle();
            publishParams.putString("fields", SAMPLING_RESULT_FIELD);
            GraphRequest request = new GraphRequest(
                    null, publishUrl, publishParams, HttpMethod.GET, null);
            request.setSkipClientToken(true);
            return request.executeAndWait().getJSONObject();
        } catch (Exception e) {
            Log.e(TAG, "fail to request button sampling api", e);
            return new JSONObject();
        }
    }

    private static void parseRemoteConfigFromJSON(
            String applicationId,
            JSONObject jsonObject) {
        JSONArray data = jsonObject.optJSONArray("data");
        if (data != null && data.length() > 0) {
            JSONObject result = data.optJSONObject(0);
            if (result != null) {
                boolean enableButtonIndexing =
                        result.optBoolean(SAMPLING_RESULT_FIELD, false);
                RemoteConfig remoteConfig = new RemoteConfig(enableButtonIndexing);
                remoteConfigs.put(applicationId, remoteConfig);
            }
        }
    }
}
