/**
 * Copyright (c) 2014-present, Facebook, Inc. All rights reserved.
 *
 * You are hereby granted a non-exclusive, worldwide, royalty-free license to use,
 * copy, modify, and distribute this software in source code or binary form for use
 * in connection with the web services and APIs provided by Facebook.
 *
 * As with any software that integrates with the Facebook platform, your use of
 * this software is subject to the Facebook Developer Principles and Policies
 * [http://developers.facebook.com/policy/]. This copyright notice shall be
 * included in all copies or substantial portions of the software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.facebook.marketing.internal;

import android.content.Context;
import android.os.Bundle;

import com.facebook.FacebookSdk;
import com.facebook.appevents.AppEventsLogger;
import com.facebook.appevents.InternalAppEventsLogger;

public final class MarketingLogger {
    // App Indexing internal events
    static final String EVENT_NAME_CODELESS_DEBUG = "fb_codeless_debug";

    static final String CODELESS_ACTION_SDK_INITIALIZED = "sdk_initialized";
    static final String CODELESS_ACTION_GESTURE_TRIGGERED = "gesture_triggered";
    static final String CODELESS_ACTION_SESSION_READY = "session_ready";
    static final String CODELESS_ACTION_INDEXING_START = "indexing_start";
    static final String CODELESS_ACTION_INDEXING_COMPLETE = "indexing_complete";
    static final String CODELESS_ACTION_INDEXING_CANCELLED = "indexing_cancelled";

    static final String ACTIVITY_NAME = "_activity_name";
    static final String CODELESS_ACTION_KEY = "_codeless_action";

    private final InternalAppEventsLogger logger;

    public MarketingLogger(Context context, String applicationId) {
        logger = new InternalAppEventsLogger(context, applicationId);
    }

    public void logCodelessInitialized() {
        if (!FacebookSdk.getCodelessDebugLogEnabled()) {
            return;
        }
        Bundle params = new Bundle();
        params.putString(CODELESS_ACTION_KEY, CODELESS_ACTION_SDK_INITIALIZED);
        logger.logEventImplicitly(EVENT_NAME_CODELESS_DEBUG, params);
    }

    public void logGestureTriggered() {
        if (!FacebookSdk.getCodelessDebugLogEnabled()) {
            return;
        }
        Bundle params = new Bundle();
        params.putString(CODELESS_ACTION_KEY, CODELESS_ACTION_GESTURE_TRIGGERED);
        logger.logEventImplicitly(EVENT_NAME_CODELESS_DEBUG, params);
    }

    public void logSessionReady() {
        if (!FacebookSdk.getCodelessDebugLogEnabled()) {
            return;
        }
        Bundle params = new Bundle();
        params.putString(CODELESS_ACTION_KEY, CODELESS_ACTION_SESSION_READY);
        logger.logEventImplicitly(EVENT_NAME_CODELESS_DEBUG, params);
    }

    public void logIndexingStart(String activityName) {
        if (!FacebookSdk.getCodelessDebugLogEnabled()) {
            return;
        }
        Bundle params = new Bundle();
        params.putString(CODELESS_ACTION_KEY, CODELESS_ACTION_INDEXING_START);
        params.putString(ACTIVITY_NAME, activityName);
        logger.logEventImplicitly(EVENT_NAME_CODELESS_DEBUG, params);
    }

    public void logIndexingComplete(String activityName) {
        if (!FacebookSdk.getCodelessDebugLogEnabled()) {
            return;
        }
        Bundle params = new Bundle();
        params.putString(CODELESS_ACTION_KEY, CODELESS_ACTION_INDEXING_COMPLETE);
        params.putString(ACTIVITY_NAME, activityName);
        logger.logEventImplicitly(EVENT_NAME_CODELESS_DEBUG, params);
    }

    public void logIndexingCancelled(String activityName) {
        if (!FacebookSdk.getCodelessDebugLogEnabled()) {
            return;
        }
        Bundle params = new Bundle();
        params.putString(CODELESS_ACTION_KEY, CODELESS_ACTION_INDEXING_CANCELLED);
        params.putString(ACTIVITY_NAME, activityName);
        logger.logEventImplicitly(EVENT_NAME_CODELESS_DEBUG, params);
    }
}
