/*
 * Copyright (c) 2014-present, Facebook, Inc. All rights reserved.
 *
 * You are hereby granted a non-exclusive, worldwide, royalty-free license to use,
 * copy, modify, and distribute this software in source code or binary form for use
 * in connection with the web services and APIs provided by Facebook.
 *
 * As with any software that integrates with the Facebook platform, your use of
 * this software is subject to the Facebook Developer Principles and Policies
 * [http://developers.facebook.com/policy/]. This copyright notice shall be
 * included in all copies or substantial portions of the software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.facebook.marketing.internal;

import android.app.Activity;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import androidx.annotation.Nullable;
import android.util.DisplayMetrics;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.widget.AdapterView;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.ListView;
import android.widget.TextView;

import com.facebook.FacebookException;
import com.facebook.FacebookSdk;
import com.facebook.appevents.codeless.internal.ViewHierarchy;
import com.facebook.internal.Utility;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;
import java.util.WeakHashMap;


public class ButtonIndexer {
    private static final String TAG = ButtonIndexer.class.getCanonicalName();

    private final Handler uiThreadHandler;
    private Set<Activity> activitiesSet;
    private Set<ViewProcessor> viewProcessors;
    private HashSet<String> listenerSet;
    private HashMap<Integer, HashSet<String>> activityToListenerMap;

    private static ButtonIndexer buttonIndexer = null;

    private ButtonIndexer() {
        uiThreadHandler = new Handler(Looper.getMainLooper());
        activitiesSet = Collections.newSetFromMap(
                new WeakHashMap<Activity, Boolean>());
        viewProcessors = new HashSet<>();
        listenerSet = new HashSet<>();
        this.activityToListenerMap = new HashMap<>();
    }

    public static synchronized ButtonIndexer getInstance() {
        if (buttonIndexer == null) {
            buttonIndexer = new ButtonIndexer();
        }
        return buttonIndexer;
    }

    public void add(Activity activity) {
        if (Thread.currentThread() != Looper.getMainLooper().getThread()) {
            throw new FacebookException("Can't add activity to ButtonIndexer on non-UI thread");
        }
        activitiesSet.add(activity);
        listenerSet.clear();
        if (activityToListenerMap.containsKey(activity.hashCode())) {
            listenerSet = activityToListenerMap.get(activity.hashCode());
        }
        startTracking();
    }

    public void remove(Activity activity) {
        if (Thread.currentThread() != Looper.getMainLooper().getThread()) {
            throw new FacebookException(
                    "Can't remove activity from ButtonIndexer on non-UI thread"
            );
        }
        activitiesSet.remove(activity);
        viewProcessors.clear();
        this.activityToListenerMap.put(activity.hashCode(), (HashSet<String>) listenerSet.clone());
        listenerSet.clear();
    }

    public void destroy(Activity activity) {
        this.activityToListenerMap.remove(activity.hashCode());
    }

    private void startTracking() {
        if (Thread.currentThread() == Looper.getMainLooper().getThread()) {
            processViews();
        } else {
            uiThreadHandler.post(new Runnable() {
                @Override
                public void run() {
                    processViews();
                }
            });

        }
    }

    private void processViews() {
        for (Activity activity : this.activitiesSet) {
            if (activity != null) {
                final View rootView = activity.getWindow().getDecorView().getRootView();
                final String activityName = activity.getClass().getSimpleName();
                ViewProcessor processor = new ViewProcessor(
                        rootView, activityName, listenerSet, uiThreadHandler);
                this.viewProcessors.add(processor);
            }
        }
    }

    protected static class ViewProcessor implements ViewTreeObserver.OnGlobalLayoutListener,
            ViewTreeObserver.OnScrollChangedListener, Runnable {
        private WeakReference<View> rootView;
        private final Handler handler;
        private final String activityName;
        private HashSet<String> listenerSet;
        private HashMap<String, WeakReference<View>> viewMap;
        public static volatile Set<String> loadedKeySet = new HashSet<>();
        private static volatile float displayDensity = -1;
        private final String viewPlaceholder = "{\"classname\": \"placeholder\", \"id\": 1}";

        public ViewProcessor(View rootView,
                           String activityName, HashSet<String> listenerSet,
                           Handler handler) {
            this.rootView = new WeakReference<>(rootView);
            this.handler = handler;
            this.activityName = activityName;
            this.listenerSet = listenerSet;
            this.viewMap = new HashMap<>();
            if(displayDensity < 0) {
                Context context = rootView.getContext();
                DisplayMetrics displayMetrics = context.getResources().getDisplayMetrics();
                displayDensity = displayMetrics.density;
            }

            this.handler.postDelayed(this, 200);
        }

        @Override
        public void run() {
            final String appId = FacebookSdk.getApplicationId();
            RemoteConfig remoteConfig = RemoteConfigManager.getRemoteConfigWithoutQuery(appId);
            if (remoteConfig == null || !remoteConfig.getEnableButtonIndexing()) {
                return;
            }
            process();
        }

        @Override
        public void onGlobalLayout() {
            process();
        }

        @Override
        public void onScrollChanged() {
            process();
        }

        private void process() {
            View view = this.rootView.get();
            if (view != null) {
                attachListeners(view);
            }
        }

        public void attachListeners(final View rootView) {
            JSONObject tree = getClickableElementsOfView(rootView, -1, this.activityName, false);
            if (tree != null) {
                ButtonIndexingLogger.logAllIndexing(tree, this.activityName);
            }
            for (HashMap.Entry<String, WeakReference<View>> entry : this.viewMap.entrySet()) {
                String mapKey = entry.getKey();
                View view = entry.getValue().get();
                if (view == null) {
                    continue;
                }
                if (!(view instanceof AdapterView)) {
                    attachOnClickListener(view, mapKey);
                } else if (view instanceof ListView) {
                    attachOnItemClickListener((ListView) view, mapKey);
                }
            }
        }

        @Nullable
        public JSONObject getClickableElementsOfView(
                View view, int index, String mapKey, boolean isAncestorClickable) {
            mapKey += "." + String.valueOf(index);
            if (view == null) {
                return null;
            }
            JSONObject res = new JSONObject();
            try {
                boolean isClickable = view.isClickable() || view instanceof Button;
                if (isClickable) {
                    this.viewMap.put(mapKey, new WeakReference<>(view));
                }
                if ((view instanceof TextView || view instanceof ImageView) &&
                        (isAncestorClickable || isClickable)) {
                    if (loadedKeySet.contains(mapKey)) {
                        return null;
                    }
                    loadedKeySet.add(mapKey);
                    ViewHierarchy.updateBasicInfoOfView(view, res);
                    ViewHierarchy.updateAppearanceOfView(view, res, displayDensity);
                    return res;
                }

                JSONArray childviews = new JSONArray();
                if (view instanceof ViewGroup) {
                    ViewGroup viewGroup = (ViewGroup)view;
                    int count = viewGroup.getChildCount();
                    int idx = 0;
                    for (int i = 0; i< count; i++) {
                        View child = viewGroup.getChildAt(i);
                        if (child.getVisibility() == View.VISIBLE) {
                            JSONObject childview = getClickableElementsOfView(
                                    child, idx++, mapKey, isAncestorClickable || isClickable);
                            if (childview != null) {
                                childviews.put(childview);
                            } else {
                                childviews.put(new JSONObject(this.viewPlaceholder));
                            }
                        }
                    }
                }
                if (childviews.length() > 0) {
                    ViewHierarchy.updateBasicInfoOfView(view, res);
                    res.put("childviews", childviews);
                    return res;
                }
            } catch (JSONException e) {
                Utility.logd(TAG, e);
            }
            return null;
        }

        private void attachOnClickListener(final View view, final String mapKey) {
            try {
                if (view == null) {
                    return;
                }
                View.OnClickListener existingListener =
                        ViewHierarchy.getExistingOnClickListener(view);
                boolean isButtonIndexingEventListener = existingListener
                    instanceof ButtonIndexingEventListener.ButtonIndexingOnClickListener;
                boolean listenerSupportButtonIndexing = isButtonIndexingEventListener &&
                                ((ButtonIndexingEventListener.ButtonIndexingOnClickListener)
                                        existingListener).getSupportButtonIndexing();
                if (!this.listenerSet.contains(mapKey) && !listenerSupportButtonIndexing) {
                    View.OnClickListener listener = ButtonIndexingEventListener.
                            getOnClickListener(view, mapKey);
                    view.setOnClickListener(listener);
                    this.listenerSet.add(mapKey);
                }
            } catch (Exception e) {
                Utility.logd(TAG, e);
            }
        }

        private void attachOnItemClickListener(final AdapterView view, final String mapKey) {
            try {
                if (view == null) {
                    return;
                }
                AdapterView.OnItemClickListener existingListener =
                    view.getOnItemClickListener();
                boolean isButtonIndexingEventListener = existingListener
                    instanceof ButtonIndexingEventListener.ButtonIndexingOnItemClickListener;
                boolean listenerSupportButtonIndexing = isButtonIndexingEventListener &&
                    ((ButtonIndexingEventListener.ButtonIndexingOnItemClickListener)
                        existingListener).getSupportButtonIndexing();
                if (!this.listenerSet.contains(mapKey) && !listenerSupportButtonIndexing) {
                    AdapterView.OnItemClickListener listener = ButtonIndexingEventListener.
                        getOnItemClickListener(view, mapKey);
                    view.setOnItemClickListener(listener);
                    this.listenerSet.add(mapKey);
                }
            } catch (Exception e) {
                Utility.logd(TAG, e);
            }
        }
    }
}
