/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.internal;

import android.content.Context;
import com.facebook.LoggingBehavior;
import com.facebook.Settings;
import com.facebook.internal.Logger;
import com.facebook.internal.Utility;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.InvalidParameterException;
import java.util.Date;
import java.util.PriorityQueue;
import java.util.concurrent.atomic.AtomicLong;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;

public final class FileLruCache {
    static final String TAG = FileLruCache.class.getSimpleName();
    private static final String HEADER_CACHEKEY_KEY = "key";
    private static final String HEADER_CACHE_CONTENT_TAG_KEY = "tag";
    private static final AtomicLong bufferIndex = new AtomicLong();
    private final String tag;
    private final Limits limits;
    private final File directory;
    private boolean isTrimPending;
    private boolean isTrimInProgress;
    private final Object lock;
    private AtomicLong lastClearCacheTime = new AtomicLong(0L);

    public FileLruCache(Context context, String tag, Limits limits) {
        this.tag = tag;
        this.limits = limits;
        this.directory = new File(context.getCacheDir(), tag);
        this.lock = new Object();
        if (this.directory.mkdirs() || this.directory.isDirectory()) {
            BufferFile.deleteAll(this.directory);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    long sizeInBytesForTest() {
        Object object = this.lock;
        synchronized (object) {
            while (this.isTrimPending || this.isTrimInProgress) {
                try {
                    this.lock.wait();
                }
                catch (InterruptedException e) {}
            }
        }
        File[] files = this.directory.listFiles();
        long total = 0L;
        if (files != null) {
            for (File file : files) {
                total += file.length();
            }
        }
        return total;
    }

    public InputStream get(String key) throws IOException {
        return this.get(key, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public InputStream get(String key, String contentTag) throws IOException {
        File file = new File(this.directory, Utility.md5hash(key));
        FileInputStream input = null;
        try {
            input = new FileInputStream(file);
        }
        catch (IOException e) {
            return null;
        }
        BufferedInputStream buffered = new BufferedInputStream(input, 8192);
        boolean success = false;
        try {
            JSONObject header = StreamHeader.readHeader(buffered);
            if (header == null) {
                InputStream inputStream = null;
                return inputStream;
            }
            String foundKey = header.optString(HEADER_CACHEKEY_KEY);
            if (foundKey == null || !foundKey.equals(key)) {
                InputStream inputStream = null;
                return inputStream;
            }
            String headerContentTag = header.optString(HEADER_CACHE_CONTENT_TAG_KEY, null);
            if (contentTag == null && headerContentTag != null || contentTag != null && !contentTag.equals(headerContentTag)) {
                InputStream inputStream = null;
                return inputStream;
            }
            long accessTime = new Date().getTime();
            Logger.log(LoggingBehavior.CACHE, TAG, "Setting lastModified to " + Long.valueOf(accessTime) + " for " + file.getName());
            file.setLastModified(accessTime);
            success = true;
            BufferedInputStream bufferedInputStream = buffered;
            return bufferedInputStream;
        }
        finally {
            if (!success) {
                buffered.close();
            }
        }
    }

    OutputStream openPutStream(String key) throws IOException {
        return this.openPutStream(key, null);
    }

    public OutputStream openPutStream(final String key, String contentTag) throws IOException {
        final File buffer = BufferFile.newFile(this.directory);
        buffer.delete();
        if (!buffer.createNewFile()) {
            throw new IOException("Could not create file at " + buffer.getAbsolutePath());
        }
        FileOutputStream file = null;
        try {
            file = new FileOutputStream(buffer);
        }
        catch (FileNotFoundException e) {
            Logger.log(LoggingBehavior.CACHE, 5, TAG, "Error creating buffer output stream: " + e);
            throw new IOException(e.getMessage());
        }
        final long bufferFileCreateTime = System.currentTimeMillis();
        StreamCloseCallback renameToTargetCallback = new StreamCloseCallback(){

            @Override
            public void onClose() {
                if (bufferFileCreateTime < FileLruCache.this.lastClearCacheTime.get()) {
                    buffer.delete();
                } else {
                    FileLruCache.this.renameToTargetAndTrim(key, buffer);
                }
            }
        };
        CloseCallbackOutputStream cleanup = new CloseCallbackOutputStream(file, renameToTargetCallback);
        BufferedOutputStream buffered = new BufferedOutputStream(cleanup, 8192);
        boolean success = false;
        try {
            JSONObject header = new JSONObject();
            header.put(HEADER_CACHEKEY_KEY, (Object)key);
            if (!Utility.isNullOrEmpty(contentTag)) {
                header.put(HEADER_CACHE_CONTENT_TAG_KEY, (Object)contentTag);
            }
            StreamHeader.writeHeader(buffered, header);
            success = true;
            BufferedOutputStream bufferedOutputStream = buffered;
            return bufferedOutputStream;
        }
        catch (JSONException e) {
            Logger.log(LoggingBehavior.CACHE, 5, TAG, "Error creating JSON header for cache file: " + (Object)((Object)e));
            throw new IOException(e.getMessage());
        }
        finally {
            if (!success) {
                buffered.close();
            }
        }
    }

    public void clearCache() {
        final File[] filesToDelete = this.directory.listFiles(BufferFile.excludeBufferFiles());
        this.lastClearCacheTime.set(System.currentTimeMillis());
        if (filesToDelete != null) {
            Settings.getExecutor().execute(new Runnable(){

                @Override
                public void run() {
                    for (File file : filesToDelete) {
                        file.delete();
                    }
                }
            });
        }
    }

    private void renameToTargetAndTrim(String key, File buffer) {
        File target = new File(this.directory, Utility.md5hash(key));
        if (!buffer.renameTo(target)) {
            buffer.delete();
        }
        this.postTrim();
    }

    public InputStream interceptAndPut(String key, InputStream input) throws IOException {
        OutputStream output = this.openPutStream(key);
        return new CopyingInputStream(input, output);
    }

    public String toString() {
        return "{FileLruCache: tag:" + this.tag + " file:" + this.directory.getName() + "}";
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void postTrim() {
        Object object = this.lock;
        synchronized (object) {
            if (!this.isTrimPending) {
                this.isTrimPending = true;
                Settings.getExecutor().execute(new Runnable(){

                    @Override
                    public void run() {
                        FileLruCache.this.trim();
                    }
                });
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void trim() {
        Object object = this.lock;
        synchronized (object) {
            this.isTrimPending = false;
            this.isTrimInProgress = true;
        }
        try {
            Logger.log(LoggingBehavior.CACHE, TAG, "trim started");
            PriorityQueue<ModifiedFile> heap = new PriorityQueue<ModifiedFile>();
            long size = 0L;
            long count = 0L;
            File[] filesToTrim = this.directory.listFiles(BufferFile.excludeBufferFiles());
            if (filesToTrim != null) {
                for (File file : filesToTrim) {
                    ModifiedFile modified = new ModifiedFile(file);
                    heap.add(modified);
                    Logger.log(LoggingBehavior.CACHE, TAG, "  trim considering time=" + Long.valueOf(modified.getModified()) + " name=" + modified.getFile().getName());
                    size += file.length();
                    ++count;
                }
            }
            while (size > (long)this.limits.getByteCount() || count > (long)this.limits.getFileCount()) {
                File file = ((ModifiedFile)heap.remove()).getFile();
                Logger.log(LoggingBehavior.CACHE, TAG, "  trim removing " + file.getName());
                size -= file.length();
                --count;
                file.delete();
            }
        }
        finally {
            object = this.lock;
            synchronized (object) {
                this.isTrimInProgress = false;
                this.lock.notifyAll();
            }
        }
    }

    private static interface StreamCloseCallback {
        public void onClose();
    }

    private static final class ModifiedFile
    implements Comparable<ModifiedFile> {
        private static final int HASH_SEED = 29;
        private static final int HASH_MULTIPLIER = 37;
        private final File file;
        private final long modified;

        ModifiedFile(File file) {
            this.file = file;
            this.modified = file.lastModified();
        }

        File getFile() {
            return this.file;
        }

        long getModified() {
            return this.modified;
        }

        @Override
        public int compareTo(ModifiedFile another) {
            if (this.getModified() < another.getModified()) {
                return -1;
            }
            if (this.getModified() > another.getModified()) {
                return 1;
            }
            return this.getFile().compareTo(another.getFile());
        }

        public boolean equals(Object another) {
            return another instanceof ModifiedFile && this.compareTo((ModifiedFile)another) == 0;
        }

        public int hashCode() {
            int result = 29;
            result = result * 37 + this.file.hashCode();
            result = result * 37 + (int)(this.modified % Integer.MAX_VALUE);
            return result;
        }
    }

    public static final class Limits {
        private int byteCount = 0x100000;
        private int fileCount = 1024;

        int getByteCount() {
            return this.byteCount;
        }

        int getFileCount() {
            return this.fileCount;
        }

        void setByteCount(int n) {
            if (n < 0) {
                throw new InvalidParameterException("Cache byte-count limit must be >= 0");
            }
            this.byteCount = n;
        }

        void setFileCount(int n) {
            if (n < 0) {
                throw new InvalidParameterException("Cache file count limit must be >= 0");
            }
            this.fileCount = n;
        }
    }

    private static final class CopyingInputStream
    extends InputStream {
        final InputStream input;
        final OutputStream output;

        CopyingInputStream(InputStream input, OutputStream output) {
            this.input = input;
            this.output = output;
        }

        @Override
        public int available() throws IOException {
            return this.input.available();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void close() throws IOException {
            try {
                this.input.close();
            }
            finally {
                this.output.close();
            }
        }

        @Override
        public void mark(int readlimit) {
            throw new UnsupportedOperationException();
        }

        @Override
        public boolean markSupported() {
            return false;
        }

        @Override
        public int read(byte[] buffer) throws IOException {
            int count = this.input.read(buffer);
            if (count > 0) {
                this.output.write(buffer, 0, count);
            }
            return count;
        }

        @Override
        public int read() throws IOException {
            int b = this.input.read();
            if (b >= 0) {
                this.output.write(b);
            }
            return b;
        }

        @Override
        public int read(byte[] buffer, int offset, int length) throws IOException {
            int count = this.input.read(buffer, offset, length);
            if (count > 0) {
                this.output.write(buffer, offset, count);
            }
            return count;
        }

        @Override
        public synchronized void reset() {
            throw new UnsupportedOperationException();
        }

        @Override
        public long skip(long byteCount) throws IOException {
            long total;
            int count;
            byte[] buffer = new byte[1024];
            for (total = 0L; total < byteCount; total += (long)count) {
                count = this.read(buffer, 0, (int)Math.min(byteCount - total, (long)buffer.length));
                if (count >= 0) continue;
                return total;
            }
            return total;
        }
    }

    private static class CloseCallbackOutputStream
    extends OutputStream {
        final OutputStream innerStream;
        final StreamCloseCallback callback;

        CloseCallbackOutputStream(OutputStream innerStream, StreamCloseCallback callback) {
            this.innerStream = innerStream;
            this.callback = callback;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void close() throws IOException {
            try {
                this.innerStream.close();
            }
            finally {
                this.callback.onClose();
            }
        }

        @Override
        public void flush() throws IOException {
            this.innerStream.flush();
        }

        @Override
        public void write(byte[] buffer, int offset, int count) throws IOException {
            this.innerStream.write(buffer, offset, count);
        }

        @Override
        public void write(byte[] buffer) throws IOException {
            this.innerStream.write(buffer);
        }

        @Override
        public void write(int oneByte) throws IOException {
            this.innerStream.write(oneByte);
        }
    }

    private static final class StreamHeader {
        private static final int HEADER_VERSION = 0;

        private StreamHeader() {
        }

        static void writeHeader(OutputStream stream, JSONObject header) throws IOException {
            String headerString = header.toString();
            byte[] headerBytes = headerString.getBytes();
            stream.write(0);
            stream.write(headerBytes.length >> 16 & 0xFF);
            stream.write(headerBytes.length >> 8 & 0xFF);
            stream.write(headerBytes.length >> 0 & 0xFF);
            stream.write(headerBytes);
        }

        static JSONObject readHeader(InputStream stream) throws IOException {
            int readCount;
            int version = stream.read();
            if (version != 0) {
                return null;
            }
            int headerSize = 0;
            for (int i = 0; i < 3; ++i) {
                int b = stream.read();
                if (b == -1) {
                    Logger.log(LoggingBehavior.CACHE, TAG, "readHeader: stream.read returned -1 while reading header size");
                    return null;
                }
                headerSize <<= 8;
                headerSize += b & 0xFF;
            }
            byte[] headerBytes = new byte[headerSize];
            for (int count = 0; count < headerBytes.length; count += readCount) {
                readCount = stream.read(headerBytes, count, headerBytes.length - count);
                if (readCount >= 1) continue;
                Logger.log(LoggingBehavior.CACHE, TAG, "readHeader: stream.read stopped at " + Integer.valueOf(count) + " when expected " + headerBytes.length);
                return null;
            }
            String headerString = new String(headerBytes);
            JSONObject header = null;
            JSONTokener tokener = new JSONTokener(headerString);
            try {
                Object parsed = tokener.nextValue();
                if (!(parsed instanceof JSONObject)) {
                    Logger.log(LoggingBehavior.CACHE, TAG, "readHeader: expected JSONObject, got " + parsed.getClass().getCanonicalName());
                    return null;
                }
                header = (JSONObject)parsed;
            }
            catch (JSONException e) {
                throw new IOException(e.getMessage());
            }
            return header;
        }
    }

    private static class BufferFile {
        private static final String FILE_NAME_PREFIX = "buffer";
        private static final FilenameFilter filterExcludeBufferFiles = new FilenameFilter(){

            @Override
            public boolean accept(File dir, String filename) {
                return !filename.startsWith(BufferFile.FILE_NAME_PREFIX);
            }
        };
        private static final FilenameFilter filterExcludeNonBufferFiles = new FilenameFilter(){

            @Override
            public boolean accept(File dir, String filename) {
                return filename.startsWith(BufferFile.FILE_NAME_PREFIX);
            }
        };

        private BufferFile() {
        }

        static void deleteAll(File root) {
            File[] filesToDelete = root.listFiles(BufferFile.excludeNonBufferFiles());
            if (filesToDelete != null) {
                for (File file : filesToDelete) {
                    file.delete();
                }
            }
        }

        static FilenameFilter excludeBufferFiles() {
            return filterExcludeBufferFiles;
        }

        static FilenameFilter excludeNonBufferFiles() {
            return filterExcludeNonBufferFiles;
        }

        static File newFile(File root) {
            String name = FILE_NAME_PREFIX + Long.valueOf(bufferIndex.incrementAndGet()).toString();
            return new File(root, name);
        }
    }
}

