/*
 *  Copyright 2017 Expedia, Inc.
 *
 *       Licensed under the Apache License, Version 2.0 (the "License");
 *       you may not use this file except in compliance with the License.
 *      You may obtain a copy of the License at
 *
 *           http://www.apache.org/licenses/LICENSE-2.0
 *
 *       Unless required by applicable law or agreed to in writing, software
 *       distributed under the License is distributed on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *       See the License for the specific language governing permissions and
 *       limitations under the License.
 */

package com.expedia.www.haystack.trace.reader.readers.transformers

import com.expedia.open.tracing.Span
import com.expedia.www.haystack.trace.reader.readers.utils.{MutableSpanForest, SpanTree, SpanUtils}


/**
  * If the root span is missing within a trace, create a pseudo root span to wrap all the spans.
  *
  * ** [[com.expedia.www.haystack.trace.reader.readers.validators.RootValidator]] and [[com.expedia.www.haystack.trace.reader.readers.validators.ParentIdValidator]] must be turned off for this to take into effect. **
  * ** Should place first in the POST transformers sequence of the configuration, as the other transformers may depend on or use the generated root during their transformation. **
  */
object OrphanedTraceTransformerConstants {
  val AUTO_GEN_REASON = "Missing root span"
}

class OrphanedTraceTransformer extends SpanTreeTransformer {

  override def transform(forest: MutableSpanForest): MutableSpanForest = {
    val orphanedTrees = forest.orphanedTrees()
    if (orphanedTrees.isEmpty) {
      forest
    } else if (multipleOrphans(orphanedTrees)) {
      forest.updateUnderlyingSpans(Seq.empty)
    } else {
      val rootSpan = generateRootSpan(forest.getUnderlyingSpans)
      forest.addNewRoot(rootSpan)
    }
  }

  def multipleOrphans(orphanedTrees: Seq[SpanTree]): Boolean = {
    val orphanedParents = orphanedTrees.groupBy(_.span.getParentSpanId)
    if (orphanedParents.size != 1) return true

    // we may now have multiple orphaned trees but each tree's root span has the same parentId
    // if this parentId is same as traceId, then we will not call them as multipleOrphans as
    // we will build an autogenerated span as their parent
    val orphanedSpan = orphanedParents.head._2.head.span
    orphanedSpan.getParentSpanId != orphanedSpan.getTraceId
  }

  def generateRootSpan(spans: Seq[Span]): Span = {
    SpanUtils.createAutoGeneratedRootSpan(spans, OrphanedTraceTransformerConstants.AUTO_GEN_REASON, spans.head.getTraceId).build()
  }
}
