package com.eway.payment.sdk.presentation;


import java.io.IOException;
import java.util.ArrayList;

import io.reactivex.Flowable;
import retrofit2.Call;
import retrofit2.Response;
import com.eway.payment.sdk.data.beans.NVPair;
import com.eway.payment.sdk.data.beans.Transaction;
import com.eway.payment.sdk.data.entities.EncryptItemsResponse;
import com.eway.payment.sdk.data.entities.SubmitPayResponse;
import com.eway.payment.sdk.data.entities.UserMessageResponse;
import com.eway.payment.sdk.data.net.RapidRestAdapter;
import com.eway.payment.sdk.domain.usecases.Impl.AndroidPayUseCase;
import com.eway.payment.sdk.domain.usecases.Impl.EncryptValuesUseCase;
import com.eway.payment.sdk.domain.usecases.Impl.SubmitPaymentUseCase;
import com.eway.payment.sdk.domain.usecases.Impl.AsynUserMessageUseCase;
import com.eway.payment.sdk.domain.usecases.Impl.UserMessageUseCase;
import com.eway.payment.sdk.domain.usecases.Impl.rxEncryptValuesUseCase;
import com.eway.payment.sdk.domain.usecases.Impl.rxSubmitPaymentUseCase;
import com.eway.payment.sdk.domain.usecases.Impl.rxUserMessageUseCase;


/**
 * Created by alexanderparra on 9/11/16.
 */

public class RapidAPI {


    /**
     * This method set the endpoint according to Rapid environment {RapidEnvironment.PRODUCTION, RapidEnvironment.SANDBOX, RapidEnvironment.DEVELOPMENT}
     *
     * @param rapidEndpoint receive the environment endpoint
     */
    public static void setRapidEndpoint(String rapidEndpoint) { RapidRestAdapter.RapidEndpoint =rapidEndpoint; }


    /**
     * This method set the public key
     *
     * @param publicAPIKey
     */

    public static void setPublicAPIKey(String publicAPIKey) { RapidRestAdapter.PublicAPIKey =publicAPIKey; }

    /**
     *
     * This method submitPayment makes a purchase o refund transaction synchronously
     *
     * @param transaction
     * @return
     * @throws IOException
     */


    public static Response<SubmitPayResponse> submitPayment(Transaction transaction) throws IOException {

        SubmitPaymentUseCase submitPaymentUseCase = new SubmitPaymentUseCase(transaction);

        return submitPaymentUseCase.execute();

    }

    /**
     *
     * This method asyncSubmitPayment makes a purchase o refund transaction asynchronously
     *
     *
     * @param transaction
     * @return
     */

    public static Call<SubmitPayResponse> asyncSubmitPayment(Transaction transaction){

        SubmitPaymentUseCase submitPaymentUseCase = new SubmitPaymentUseCase(transaction);

        return submitPaymentUseCase.asyncExecute();
    }

    /**
     * This method submitPayment makes a purchase o refund transaction using rxjava. The version before used to use Observables
     * .However this implementation does not support back pressure. If you dont know about this there are two very useful articles
     *
     * https://github.com/ReactiveX/RxJava/wiki/What's-different-in-2.0
     *
     * https://realm.io/news/gotocph-jake-wharton-exploring-rxjava2-android/
     *
     * @param transaction
     * @return a Flowable in the version 1.2 and 1.2.1. it was returning an observable.In this version it is returning a Flowable.
     *
     * For further inforamtion about rxjava 2.0 please check the two articles above.
     *
     */

    public static Flowable<SubmitPayResponse> rxSubmitPayment(Transaction transaction){

        rxSubmitPaymentUseCase rxSubmitPayUseCase = new rxSubmitPaymentUseCase(transaction);

        return rxSubmitPayUseCase.rxExecute();

    }

    /**
     * This method  request  two encrypt NVPair values and response it synchronously
     *
     * @param values
     * @return Response EncryptItemsResponse
     * @throws IOException
     */

    public static Response<EncryptItemsResponse> encryptValues(ArrayList<NVPair> values) throws IOException {

        EncryptValuesUseCase encryptValuesUseCase = new EncryptValuesUseCase();

        EncryptValuesUseCase.setValues(values);

        return encryptValuesUseCase.execute();

    }

    /**
     *This method  request  two encrypt NVPair values and response it asynchronously
     *
     * @param values
     * @return
     * @throws IOException
     */

    public static Call<EncryptItemsResponse> asyncEncryptValues(ArrayList<NVPair> values) throws IOException {

        EncryptValuesUseCase encryptValuesUseCase = new EncryptValuesUseCase();

        EncryptValuesUseCase.setValues(values);

        return encryptValuesUseCase.asyncExecute();


    }

    /**
     * This method  request  two encrypt NVPair values. The response of this method is a flowable.For further information please check the articles above
     *
     * @param values
     * @return
     */

    public static Flowable<EncryptItemsResponse> rxEncryptValues(ArrayList<NVPair> values) {

        rxEncryptValuesUseCase encryptValuesUseCase = new rxEncryptValuesUseCase();

        rxEncryptValuesUseCase.setValues(values);

        return encryptValuesUseCase.rxExecute();

    }

    /**
     *  This method response an UserMessageResponse object base on error codes. This methods make a synchronous call
     *
     * @param language
     * @param errorCodes
     * @return
     */

    public static UserMessageResponse userMessage(String language, String errorCodes){

        UserMessageUseCase userMessageUseCase = new UserMessageUseCase(language,errorCodes);

        return userMessageUseCase.execute();

    }

    /**
     * This method response an UserMessageResponse object base on error codes. This method makes an asynchronous call.
     *
     * @param Language
     * @param ErrorCodes
     * @param callbackUserMessage
     * @return
     */

    public static AsynUserMessageUseCase.CallbackUserMessage asynUserMessage(String Language, String ErrorCodes, AsynUserMessageUseCase.CallbackUserMessage callbackUserMessage){

        final AsynUserMessageUseCase asynUserMessageUseCase = new AsynUserMessageUseCase(Language,ErrorCodes,callbackUserMessage);

        asynUserMessageUseCase.asyncExecute();

        return callbackUserMessage;

    }

    /**
     *
     * This method response an UserMessageResponse object base on error codes. This method receive a flowable call.
     *
     * @param language
     * @param ErrorCodes
     * @return
     */

    public static Flowable<UserMessageResponse> rxUserMessage(String language, String ErrorCodes) {

        rxUserMessageUseCase rxuserMessageUseCase = new rxUserMessageUseCase(language,ErrorCodes);

        return rxuserMessageUseCase.rxExecute();
    }


    /**
     * This methos submit a transaction through Android Pay. This method receive a synchronous response.
     *
     * @param transaction
     * @return
     * @throws IOException
     */

    public static Response<SubmitPayResponse> synAndroidPay(Transaction transaction) throws IOException {

        AndroidPayUseCase androidPayUseCase = new AndroidPayUseCase(transaction);

        return androidPayUseCase.execute();
    }

}
