/*
 * Copyright (c) 2014 by EagleXad
 * Team: EagleXad
 * Create: 2014-08-29
 */

package com.eaglexad.lib.http.tool;

import android.text.TextUtils;

import com.eaglexad.lib.http.entry.ExDownload;
import com.eaglexad.lib.http.entry.ExRequest;
import com.eaglexad.lib.http.entry.ExResponse;
import com.eaglexad.lib.http.exception.ErrorConnection;
import com.eaglexad.lib.http.exception.ExHttpException;
import com.eaglexad.lib.http.ible.IExHttpStack;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLDecoder;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

/**
 * @author Aloneter
 * @ClassName: ExHttpStackDownload
 * @Description:
 */
public class ExHttpStackDownload implements IExHttpStack {

    private ExRequest mRequest;
    private String mUrl;
    private ExDownload mDownload;
    private Map<String, String> mAdditionalHeaders;
    private HttpURLConnection mConnection;

    private File mFile;
    private InputStream mInputStream;
    private FileOutputStream mOutputStream;

    @Override
    public ExResponse performRequest(ExRequest request, Map<String, String> additionalHeaders) throws ExHttpException {

        mRequest = request;
        mUrl = request.url;
        mDownload = request.getDownload();
        mAdditionalHeaders = additionalHeaders;

        int responseCode;

        ExResponse.Builder builder = new ExResponse.Builder(request.what);
        builder.setReuest(request);

        try {

            prepare();
            openConnection();
            initHeader();

            responseCode = getResponseCode();

            if (responseCode == HttpURLConnection.HTTP_OK || responseCode == HttpURLConnection.HTTP_PARTIAL) {
                mDownload.totalSize = getContentLength();
                mInputStream = mConnection.getInputStream();
                if (mInputStream != null && mDownload.totalSize > 0) {
                    mOutputStream = new FileOutputStream(mFile, responseCode == HttpURLConnection.HTTP_PARTIAL);

                    byte[] buffer = new byte[1024];
                    int readSize;

                    int progress = (int) (mDownload.currSize * 100 / mDownload.totalSize);
                    mDownload.currProgress = progress > 100 ? 100 : progress;

                    while ((readSize = mInputStream.read(buffer)) > 0) {
                        mOutputStream.write(buffer, 0, readSize);
                        mDownload.currSize += readSize;

                        progress = (int) (mDownload.currSize * 100 / mDownload.totalSize);
                        if (mDownload.currProgress != progress && progress <= 100) {
                            mDownload.currProgress = progress > 100 ? 100 : progress;
                        }
                    }
                    if (mDownload.currProgress == 100) {
                        String destFileName = mDownload.saveFilePath.replace(ExDownload.CONSTANT_DEFAULT_EXT, mDownload.ext);

                        if (renameFile(mDownload.saveFilePath, destFileName)) {
                            mDownload.saveFilePath = destFileName;
                        }
                    }
                }
            } else if (responseCode == HttpURLConnection.HTTP_MOVED_PERM || responseCode == HttpURLConnection.HTTP_MOVED_TEMP) {

                return performRequest(request, additionalHeaders);
            }

            builder.setContentLength(getContentLength());
            builder.setLastModified(getLastModified());
            builder.setExpiration(getExpiration());
            builder.setETag(getETag());
            builder.setResponseMessage(getResponseMessage());
            builder.setResponseCode(responseCode);
            builder.setResponseHeaders(getResponseHeaders());

        } catch (IOException e) {
            throw new ErrorConnection(e.getMessage(), e);
        } finally {
            close();
        }

        return builder.builder();
    }

    @Override
    public void close() {

        if (mInputStream != null) {
            ExHttpUtils.closeQuietly(mInputStream);
        }
        if (mOutputStream != null) {
            ExHttpUtils.closeQuietly(mOutputStream);
        }
        if (mConnection != null) {
            mConnection.disconnect();
        }
    }

    private void prepare() throws IOException {

        mFile = new File(mDownload.saveFilePath);

        if (!mFile.exists()) {
            mFile.createNewFile();
        } else {
            if (mDownload.isOverride) {
                mFile.delete();
            }
        }

        mDownload.currSize = mFile.length();
    }

    private void openConnection() throws IOException {

        URL url = new URL(mUrl);

        mConnection = (HttpURLConnection) url.openConnection();

        mConnection.setReadTimeout(mRequest.connectTimeout);
        mConnection.setConnectTimeout(mRequest.connectTimeout);
        mConnection.setRequestProperty("Connection", "Keep-Alive");
        mConnection.setRequestProperty("Content-Type", "application/octet-stream");
        mConnection.setInstanceFollowRedirects(true);

        if (mDownload.currSize > 0 && isSupportRange()) {
            mConnection.setRequestProperty("RANGE", "bytes=" + mDownload.currSize + "-");
        }
    }

    private void initHeader() {

        List<ExRequest.Header> headers = mRequest.headers;

        if (headers != null) {
            for (ExRequest.Header header : headers) {
                String name = header.key;
                String value = header.getValueStr();

                if (!ExHttpUtils.isEmpty(name) && !ExHttpUtils.isEmpty(value)) {
                    if (header.setHeader) {
                        mConnection.setRequestProperty(name, value);
                    } else {
                        mConnection.addRequestProperty(name, value);
                    }
                }
            }
        }
        if (mAdditionalHeaders != null && mAdditionalHeaders.size() > 0) {
            for (String key : mAdditionalHeaders.keySet()) {
                String value = mAdditionalHeaders.get(key);

                if (!ExHttpUtils.isEmpty(value)) {
                    mConnection.addRequestProperty(key, value);
                }
            }
        }
    }

    private long getContentLength() {

        long result = 0;

        if (mConnection != null) {
            try {
                result = mConnection.getContentLength();
            } catch (Throwable ex) {
            }
            if (result < 1) {
                try {
                    result = mInputStream.available();
                } catch (Throwable ignored) {
                }
            }
        } else {
            try {
                result = mInputStream.available();
            } catch (Throwable ignored) {
            }
        }

        return result;
    }

    private long getExpiration() {

        if (mConnection == null) {
            return -1L;
        }

        long expiration = -1L;

        // from max-age
        String cacheControl = mConnection.getHeaderField("Cache-Control");

        if (!ExHttpUtils.isEmpty(cacheControl)) {
            StringTokenizer tok = new StringTokenizer(cacheControl, ",");

            while (tok.hasMoreTokens()) {
                String token = tok.nextToken().trim().toLowerCase();

                if (token.startsWith("max-age")) {
                    int eqIdx = token.indexOf('=');

                    if (eqIdx > 0) {
                        try {
                            String value = token.substring(eqIdx + 1).trim();
                            long seconds = Long.parseLong(value);

                            if (seconds > 0L) {
                                expiration = System.currentTimeMillis() + seconds * 1000L;
                            }
                        } catch (Throwable ex) {
                        }
                    }
                    break;
                }
            }
        }

        // from expires
        if (expiration <= 0L) {
            expiration = mConnection.getExpiration();
        }
        if (expiration <= 0) {
            expiration = System.currentTimeMillis() + mRequest.cacheMaxAge;
        }
        if (expiration <= 0) {
            expiration = Long.MAX_VALUE;
        }

        return expiration;
    }

    private long getLastModified() {

        return mConnection == null ? System.currentTimeMillis() : mConnection.getHeaderFieldDate("Last-Modified", System.currentTimeMillis());
    }

    private String getETag() {

        return mConnection == null ? null : mConnection.getHeaderField("ETag");
    }

    private Map<String, List<String>> getResponseHeaders() {

        return mConnection == null ? null : mConnection.getHeaderFields();
    }

    private int getResponseCode() throws IOException {

        return mConnection == null ? -1 : mConnection.getResponseCode();
    }

    private String getResponseMessage() throws IOException {

        return mConnection == null ? null : URLDecoder.decode(mConnection.getResponseMessage(), mRequest.charset);
    }

    private String getResponseFileName() {

        String disposition = mConnection.getHeaderField("Content-Disposition");
        if (!TextUtils.isEmpty(disposition)) {
            int startIndex = disposition.indexOf("filename=");
            if (startIndex > 0) {
                startIndex += 9; // "filename=".length()
                int endIndex = disposition.indexOf(";", startIndex);
                if (endIndex < 0) {
                    endIndex = disposition.length();
                }
                if (endIndex > startIndex) {
                    try {
                        return URLDecoder.decode(disposition.substring(startIndex, endIndex), mRequest.charset);
                    } catch (UnsupportedEncodingException ex) {
                    }
                }
            }
        }
        return null;
    }

    private boolean isSupportRange() {

        String ranges = mConnection.getHeaderField("Accept-Ranges");

        if (ranges != null) {
            return ranges.contains("bytes");
        }
        ranges = mConnection.getHeaderField("Content-Range");

        return ranges != null && ranges.contains("bytes");
    }

    private boolean renameFile(String srcFilePath, String destFilePath) {

        File srcFile = new File(srcFilePath);
        File destFile = new File(destFilePath);

        return srcFile.renameTo(destFile);
    }

}
