/*
 * Copyright (c) 2014 by EagleXad
 * Team: EagleXad
 * Create: 2014-08-29
 */

package com.eaglexad.lib.http;

import android.content.Context;

import com.eaglexad.lib.http.entry.ExDownload;
import com.eaglexad.lib.http.entry.ExError;
import com.eaglexad.lib.http.entry.ExRequest;
import com.eaglexad.lib.http.entry.ExResponse;
import com.eaglexad.lib.http.ible.IExCache;
import com.eaglexad.lib.http.ible.IExConvert;
import com.eaglexad.lib.http.ible.IExHttpMethod;
import com.eaglexad.lib.http.ible.IExHttpStack;
import com.eaglexad.lib.http.ible.IExNetwork;
import com.eaglexad.lib.http.ible.IExRedirect;
import com.eaglexad.lib.http.tool.ExHttpQueue;
import com.eaglexad.lib.http.tool.ExHttpStackDownload;
import com.eaglexad.lib.http.tool.ExHttpStackRequest;
import com.eaglexad.lib.http.tool.ExHttpUtils;
import com.eaglexad.lib.http.tool.ExNetwork;

import java.io.File;
import java.net.Proxy;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.net.ssl.SSLSocketFactory;

/**
 * @author Aloneter
 * @ClassName: ExHttp
 * @Description:
 */
public abstract class ExHttp implements IExHttpMethod {

    public static final String TAG = ExHttp.class.getName();

    public static final int DEFAULT_REQUEST_TIME = 30000;
    public static final int DEFAULT_CACHE_TIME = -1;

    public static ExHttpQueue newRequestQueue() {

        return newRequestQueue(null);
    }

    public static ExHttpQueue newRequestQueue(IExHttpStack httpStack) {

        if (httpStack == null) {
            httpStack = new ExHttpStackRequest();
        }

        IExNetwork network = new ExNetwork(httpStack);

        ExHttpQueue queue = new ExHttpQueue(network);
        queue.start();

        return queue;
    }

    public static ExHttpQueue newDownloadQueue() {

        return newDownloadQueue(null);
    }

    public static ExHttpQueue newDownloadQueue(IExHttpStack httpStack) {

        if (httpStack == null) {
            httpStack = new ExHttpStackDownload();
        }

        IExNetwork network = new ExNetwork(httpStack);

        ExHttpQueue queue = new ExHttpQueue(network);
        queue.start();

        return queue;
    }

    public static ExHttpQueue newUploadQueue() {

        return newUploadQueue(null);
    }

    public static ExHttpQueue newUploadQueue(IExHttpStack httpStack) {

        if (httpStack == null) {
            httpStack = new ExHttpStackRequest();
        }

        IExNetwork network = new ExNetwork(httpStack);

        ExHttpQueue queue = new ExHttpQueue(network, ExHttpQueue.DEFAULT_NETWORK_THREAD_SINGLE_POOL_SIZE);
        queue.start();

        return queue;
    }

    public static ExHttp newInstance(Context context) {

        return new ExHttp(context) {

            @Override
            protected boolean onInterceptRequest(ExRequest request) {
                return false;
            }

            @Override
            protected void onRecordSuccess(ExResponse response) {

            }

            @Override
            protected void onRecordError(ExError error, int status, String message) {

            }
        };
    }

    public static ExHttp newInstanceRequest(Context context) {

        ExHttp http = newInstance(context);

        http.initQueueRequest(null);

        return http;
    }

    public static ExHttp newInstanceUpload(Context context) {

        ExHttp http = newInstance(context);

        http.initQueueUpload(null);

        return http;
    }

    public static ExHttp newInstanceDownload(Context context) {

        ExHttp http = newInstance(context);

        http.initQueueDownload(null);

        return http;
    }

    @Override
    public ExRequest request(int what, String url, Method method, Object remark, RequestCallback callback) {

        return request(builder(what, url, method, null, null, null, CacheType.NONE, remark, callback));
    }

    @Override
    public ExRequest request(int what, String url, Method method, Map<String, Object> bodyParams, Object remark, RequestCallback callback) {

        return request(builder(what, url, method, null, bodyParams, null, CacheType.NONE, remark, callback));
    }

    @Override
    public ExRequest request(int what, String url, Method method, Map<String, Object> bodyParams, CacheType cacheType, Object remark, RequestCallback callback) {

        return request(builder(what, url, method, null, bodyParams, null, cacheType, remark, callback));
    }

    @Override
    public ExRequest request(int what, String url, Method method, Map<String, String> queryParams, Map<String, Object> bodyParams, Object remark, RequestCallback callback) {

        return request(builder(what, url, method, queryParams, bodyParams, null, CacheType.NONE, remark, callback));
    }

    @Override
    public ExRequest request(int what, String url, Method method, Map<String, String> queryParams, Map<String, Object> bodyParams, CacheType cacheType, Object remark, RequestCallback callback) {

        return request(builder(what, url, method, queryParams, bodyParams, null, cacheType, remark, callback));
    }

    @Override
    public ExRequest request(int what, String url, Method method, String bodyContent, Object remark, RequestCallback callback) {

        return request(builder(what, url, method, null, null, bodyContent, CacheType.NONE, remark, callback));
    }

    @Override
    public ExRequest request(int what, String url, Method method, String bodyContent, CacheType cacheType, Object remark, RequestCallback callback) {

        return request(builder(what, url, method, null, null, bodyContent, cacheType, remark, callback));
    }

    @Override
    public ExRequest request(int what, String url, Method method, Map<String, String> queryParams, String bodyContent, Object remark, RequestCallback callback) {

        return request(builder(what, url, method, queryParams, null, bodyContent, CacheType.NONE, remark, callback));
    }

    @Override
    public ExRequest request(int what, String url, Method method, Map<String, String> queryParams, String bodyContent, CacheType cacheType, Object remark, RequestCallback callback) {

        return request(builder(what, url, method, queryParams, null, bodyContent, cacheType, remark, callback));
    }

    @Override
    public ExRequest request(int what, String url, Method method, Map<String, String> queryParams, Map<String, Object> bodyParams, String bodyContent, CacheType cacheType, Object remark, RequestCallback callback) {

        return request(builder(what, url, method, queryParams, bodyParams, bodyContent, cacheType, remark, callback));
    }

    @Override
    public ExRequest execute(int what, String url, Method method, Object remark, RequestCallback callback) {

        return execute(builder(what, url, method, null, null, null, CacheType.NONE, remark, callback));
    }

    @Override
    public ExRequest execute(int what, String url, Method method, Map<String, Object> bodyParams, Object remark, RequestCallback callback) {

        return execute(builder(what, url, method, null, bodyParams, null, CacheType.NONE, remark, callback));
    }

    @Override
    public ExRequest execute(int what, String url, Method method, Map<String, Object> bodyParams, CacheType cacheType, Object remark, RequestCallback callback) {

        return execute(builder(what, url, method, null, bodyParams, null, cacheType, remark, callback));
    }

    @Override
    public ExRequest execute(int what, String url, Method method, Map<String, String> queryParams, Map<String, Object> bodyParams, Object remark, RequestCallback callback) {

        return execute(builder(what, url, method, queryParams, bodyParams, null, CacheType.NONE, remark, callback));
    }

    @Override
    public ExRequest execute(int what, String url, Method method, Map<String, String> queryParams, Map<String, Object> bodyParams, CacheType cacheType, Object remark, RequestCallback callback) {

        return execute(builder(what, url, method, queryParams, bodyParams, null, cacheType, remark, callback));
    }

    @Override
    public ExRequest execute(int what, String url, Method method, String bodyContent, Object remark, RequestCallback callback) {

        return execute(builder(what, url, method, null, null, bodyContent, CacheType.NONE, remark, callback));
    }

    @Override
    public ExRequest execute(int what, String url, Method method, String bodyContent, CacheType cacheType, Object remark, RequestCallback callback) {

        return execute(builder(what, url, method, null, null, bodyContent, cacheType, remark, callback));
    }

    @Override
    public ExRequest execute(int what, String url, Method method, Map<String, String> queryParams, String bodyContent, Object remark, RequestCallback callback) {

        return execute(builder(what, url, method, queryParams, null, bodyContent, CacheType.NONE, remark, callback));
    }

    @Override
    public ExRequest execute(int what, String url, Method method, Map<String, String> queryParams, String bodyContent, CacheType cacheType, Object remark, RequestCallback callback) {

        return execute(builder(what, url, method, queryParams, null, bodyContent, cacheType, remark, callback));
    }

    @Override
    public ExRequest execute(int what, String url, Method method, Map<String, String> queryParams, Map<String, Object> bodyParams, String bodyContent, CacheType cacheType, Object remark, RequestCallback callback) {

        return execute(builder(what, url, method, queryParams, bodyParams, bodyContent, cacheType, remark, callback));
    }

    // use

    @Override
    public ExRequest download(int what, String url, Object remark, DownloadCallback callback) {

        return download(what, url, null, ExDownload.CONSTANT_DEFAULT_NAME, ExDownload.CONSTANT_DEFAULT_PATH, ExDownload.CONSTANT_DEFAULT_EXT, remark, callback);
    }

    @Override
    public ExRequest download(int what, String url, Map<String, String> queryParams, Object remark, DownloadCallback callback) {

        return download(what, url, queryParams, ExDownload.CONSTANT_DEFAULT_NAME, ExDownload.CONSTANT_DEFAULT_PATH, ExDownload.CONSTANT_DEFAULT_EXT, remark, callback);
    }

    @Override
    public ExRequest download(int what, String url, File file, Object remark, DownloadCallback callback) {

        return download(what, url, null, file, remark, callback);
    }

    @Override
    public ExRequest download(int what, String url, Map<String, String> queryParams, File file, Object remark, DownloadCallback callback) {

        return download(what, url, queryParams, ExDownload.getName(file), ExDownload.getSaveFilePath(file), ExDownload.getExt(file), remark, callback);
    }

    @Override
    public ExRequest download(int what, String url, String name, String saveFilePath, String ext, Object remark, DownloadCallback callback) {

        return download(what, url, null, name, saveFilePath, ext, remark, callback);
    }

    @Override
    public ExRequest download(int what, String url, Map<String, String> queryParams, String name, String saveFilePath, String ext, Object remark, DownloadCallback callback) {

        ExDownload.Builder builder = new ExDownload.Builder(name, name, saveFilePath, ext);

        return download(builder(what, url, Method.POST, queryParams, builder.builder(), remark, callback));
    }

    @Override
    public ExRequest upload(int what, String url, Map<String, File> files, Object remark, UploadCallback callback) {

        return upload(what, url, null, files, remark, callback);
    }

    @Override
    public ExRequest upload(int what, String url, Map<String, String> queryParams, Map<String, File> files, Object remark, UploadCallback callback) {

        Map<String, List<File>> result = new HashMap<>();

        for (String key : files.keySet()) {
            File file = files.get(key);
            if (!ExHttpUtils.isEmpty(file)) {
                List<File> list = new ArrayList<>();

                list.add(file);

                result.put(key, list);
            }
        }

        return upload(builder(what, url, Method.POST, queryParams, result, remark, callback));
    }

    @Override
    public ExRequest upload(int what, String url, File file, Object remark, UploadCallback callback) {

        return upload(what, url, null, file, remark, callback);
    }

    @Override
    public ExRequest upload(int what, String url, Map<String, String> queryParams, File file, Object remark, UploadCallback callback) {

        Map<String, List<File>> result = new HashMap<>();

        if (!ExHttpUtils.isEmpty(file)) {
            List<File> list = new ArrayList<>();

            list.add(file);

            result.put(file.getName(), list);
        }

        return upload(builder(what, url, Method.POST, queryParams, result, remark, callback));
    }

    @Override
    public ExRequest upload(int what, String url, String fileName, List<File> files, Object remark, UploadCallback callback) {

        return upload(what, url, null, fileName, files, remark, callback);
    }

    @Override
    public ExRequest upload(int what, String url, Map<String, String> queryParams, String fileName, List<File> files, Object remark, UploadCallback callback) {

        Map<String, List<File>> result = new HashMap<>();

        for (File file : files) {
            if (!ExHttpUtils.isEmpty(file)) {
                List<File> list = new ArrayList<>();

                list.add(file);

                result.put(fileName, list);
            }
        }

        return upload(builder(what, url, Method.POST, queryParams, result, remark, callback));
    }

    private static Context mContext;

    public static Context getContext() {

        return mContext;
    }

    private ExHttpQueue mHttpQueue;
    private ExHttpQueue mHttpDownloadQueue;
    private ExHttpQueue mHttpUploadQueue;

    public ExHttp(Context context) {
        this(context, null);
    }

    public ExHttp(Context context, IExHttpStack httpStack) {

        mContext = context;

        if (!ExHttpUtils.isEmpty(httpStack)) {
            initQueueRequest(httpStack);
            initQueueUpload(httpStack);
            initQueueDownload(httpStack);
        }
    }

    public void initQueueRequest(IExHttpStack httpStack) {

        mHttpQueue = newRequestQueue(httpStack);
    }

    public void initQueueUpload(IExHttpStack httpStack) {

        mHttpUploadQueue = newRequestQueue(httpStack);
    }

    public void initQueueDownload(IExHttpStack httpStack) {

        mHttpDownloadQueue = newDownloadQueue(httpStack);
    }

    public ExRequest.Builder builder(int what, String url, Method method, Map<String, String> queryParams, Map<String, Object> bodyParams, String bodyContent, CacheType cacheType, Object remark, RequestCallback callback) {

        ExRequest.Builder builder = new ExRequest.Builder(what, reWriteURL(url), method, remark, callback);

        builder.setCacheType(cacheType);

        if (!ExHttpUtils.isEmpty(queryParams)) {
            for (String key : queryParams.keySet()) {
                builder.addQueryParam(key, queryParams.get(key));
            }
        }
        if (!ExHttpUtils.isEmpty(bodyParams)) {
            for (String key : bodyParams.keySet()) {
                builder.addBodyParam(key, bodyParams.get(key));
            }
        }
        if (!ExHttpUtils.isEmpty(bodyContent)) {
            builder.setBodyContent(bodyContent);
        }

        return builder;
    }

    public ExRequest.Builder builder(int what, String url, Method method, Map<String, String> queryParams, ExDownload download, Object remark, DownloadCallback callback) {

        ExRequest.Builder builder = new ExRequest.Builder(what, reWriteURL(url), method, remark, callback);

        if (!ExHttpUtils.isEmpty(queryParams)) {
            for (String key : queryParams.keySet()) {
                builder.addQueryParam(key, queryParams.get(key));
            }
        }

        ExRequest request = builder.build();

        request.setDownload(download);

        return builder;
    }

    public ExRequest.Builder builder(int what, String url, Method method, Map<String, String> queryParams, Map<String, List<File>> files, Object remark, UploadCallback callback) {

        ExRequest.Builder builder = new ExRequest.Builder(what, reWriteURL(url), method, remark, callback);

        if (!ExHttpUtils.isEmpty(queryParams)) {
            for (String key : queryParams.keySet()) {
                builder.addQueryParam(key, queryParams.get(key));
            }
        }
        if (!ExHttpUtils.isEmpty(files)) {
            for (String key : files.keySet()) {
                List<File> list = files.get(key);
                if (!ExHttpUtils.isEmpty(list)) {
                    for (File file : list) {
                        builder.addFileParam(key, file);
                    }
                }
            }
        }

        return builder;
    }

    public ExRequest request(ExRequest.Builder builder) {

        if (ExHttpUtils.isEmpty(mHttpQueue)) {
            throw new NullPointerException("request http queue is null object,queue is not initialized");
        }

        ExRequest request = prepare(builder);

        if (!request.isIntercept && !request.isCancel) {
            mHttpQueue.add(request);
        }

        return request;
    }

    public ExRequest execute(ExRequest.Builder builder) {

        if (ExHttpUtils.isEmpty(mHttpQueue)) {
            throw new NullPointerException("request http queue is null object,queue is not initialized");
        }

        ExRequest request = prepare(builder);

        if (!request.isIntercept && !request.isCancel) {
            mHttpQueue.execute(request);
        }

        return request;
    }

    public ExRequest download(ExRequest.Builder builder) {

        if (ExHttpUtils.isEmpty(mHttpDownloadQueue)) {
            throw new NullPointerException("download http queue is null object,queue is not initialized");
        }

        ExRequest request = prepare(builder);

        if (!request.isIntercept && !request.isCancel) {
            mHttpDownloadQueue.add(request);
        }

        return request;
    }

    public ExRequest upload(ExRequest.Builder builder) {

        if (ExHttpUtils.isEmpty(mHttpUploadQueue)) {
            throw new NullPointerException("upload http queue is null object,http queue not to init");
        }

        ExRequest request = prepare(builder);

        if (!request.isIntercept && !request.isCancel) {
            mHttpUploadQueue.add(request);
        }

        return request;
    }

    private ExRequest prepare(ExRequest.Builder builder) {

        builder.setProxy(initProxy());
        builder.setSSLSocketFactory(initSSLSocketFactory());
        builder.setHostNameVerify(initHostnameVerifier());
        builder.setConnectTimeout(initRequestTime());
        builder.setCacheTimeout(initCacheTime());

        builder.setRuleOfCache(ruleOfCacheIEx());
        builder.setRuleOfConvert(ruleOfConvertIEx());
        builder.setRuleOfRedirect(ruleOfRedirectIEx());

        Map<String, String> header = initRequestHeader();

        if (!ExHttpUtils.isEmpty(header)) {
            for (String key : header.keySet()) {
                builder.addHeader(key, header.get(key));
            }
        }

        ExRequest request = builder.build();

        request.isIntercept = onInterceptRequest(request);

        if (request.shouldCache) {
            request.cacheKey = ruleOfCacheKey(request);
            request.cacheGroupKey = ruleOfCacheGroupKey(request);
        }

        return request;
    }

    protected int initRequestTime() {

        return DEFAULT_REQUEST_TIME;
    }

    protected int initCacheTime() {

        return DEFAULT_CACHE_TIME;
    }

    protected SSLSocketFactory initSSLSocketFactory() {

        // Client should authenticate itself with the valid certificate to Server.
//        	InputStream clientStream = VolleySampleApplication.getContext().getResources().openRawResource(R.raw.production_test_client);
//        	char[] password = "XXXXXXXXXXXXX".toCharArray();
//
//	        KeyStore keyStore = KeyStore.getInstance("PKCS12");
//	        keyStore.load(clientStream, password);
//
//            KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
//            keyManagerFactory.init(keyStore, password);
//
//
//	        // Client should also add the CA certificate obtained from server and create TrustManager from it for the client to validate the
//            // identity of the server.
//	        KeyStore trustStore  = KeyStore.getInstance( "BKS");
//	        InputStream inputStream = null;
//	        inputStream = VolleySampleApplication.getContext().getResources().openRawResource(R.raw.production_test_ca);
//
//	        try {
//	            trustStore.load(inputStream, "XXXXXXXX".toCharArray());
//	        } catch (Exception e) {
//	            e.printStackTrace();
//	        } finally {
//	            try { inputStream.close(); } catch (Exception ignore) {}
//	        }
//
//            String tmfAlgorithm = TrustManagerFactory.getDefaultAlgorithm();
//            TrustManagerFactory tmf = TrustManagerFactory.getInstance(tmfAlgorithm);
//            tmf.init(trustStore);
//
//            // Create an SSLContext that uses our TrustManager & Keystore
//            SSLContext context = SSLContext.getInstance("TLS");
//            context.init(keyManagerFactory.getKeyManagers(), tmf.getTrustManagers(), null);
//            String tmfAlgorithm = TrustManagerFactory.getDefaultAlgorithm();
//            TrustManagerFactory tmf = TrustManagerFactory.getInstance(tmfAlgorithm);
//            tmf.init(trustStore);
//
//            // Create an SSLContext that uses our TrustManager & Keystore
//            SSLContext context = SSLContext.getInstance("TLS");
//            context.init(null, tmf.getTrustManagers(), null);

        return null;
    }

    protected boolean initHostnameVerifier() {

        return true;
    }

    protected Proxy initProxy() {

        return null;
    }

    protected Map<String, String> initRequestHeader() {

        return null;
    }

    protected String reWriteURL(String url) {

        return url;
    }

    protected String ruleOfCacheKey(ExRequest request) {

        return request.url;
    }

    protected String ruleOfCacheGroupKey(ExRequest request) {

        return request.url;
    }

    public IExConvert ruleOfConvertIEx() {

        return null;
    }

    public IExCache ruleOfCacheIEx() {

        return null;
    }

    public IExRedirect ruleOfRedirectIEx() {

        return null;
    }

    protected abstract boolean onInterceptRequest(ExRequest request);

    protected abstract void onRecordSuccess(ExResponse response);

    protected abstract void onRecordError(ExError error, int status, String message);

    public interface RequestCallback {

        void onSuccess(ExResponse response);

        void onError(ExError error, int status, String message);

    }

    public interface UploadCallback extends RequestCallback {

        void onInitialize(long length);

        void onProgress(long progress);

    }

    public interface DownloadCallback extends RequestCallback {

        void onInitialize(long length);

        void onProgress(long progress);

    }

    public enum CacheType {

        NONE,
        REQUEST_BEFORE,
        REQUEST_AFTER;

    }

    public enum Priority {

        UI_TOP,
        UI_NORMAL,
        UI_LOW,
        DEFAULT,
        BG_TOP,
        BG_NORMAL,
        BG_LOW;

    }

    public enum Method {

        GET("GET"),
        POST("POST"),
        PUT("PUT"),
        PATCH("PATCH"),
        HEAD("HEAD"),
        MOVE("MOVE"),
        COPY("COPY"),
        DELETE("DELETE"),
        OPTIONS("OPTIONS"),
        TRACE("TRACE"),
        CONNECT("CONNECT");

        private final String value;

        Method(String value) {
            this.value = value;
        }

        @Override
        public String toString() {
            return this.value;
        }

        public static boolean permitsRetry(Method method) {
            return method == GET;
        }

        public static boolean permitsCache(Method method) {
            return method == GET || method == POST;
        }

        public static boolean permitsRequestBody(Method method) {
            return method == POST
                    || method == PUT
                    || method == PATCH
                    || method == DELETE;
        }

    }

}
