/*
 * Copyright 2015 Dynatrace
 *
 * Licensed under the Dynatrace SaaS terms of service (the "License");
 * You may obtain a copy of the License at
 *
 *      https://ruxit.com/eula/saas/#terms-of-service
 */
package com.dynatrace.tools.android;

import static com.google.common.base.Preconditions.checkNotNull;

import java.io.BufferedOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.security.PrivateKey;
import java.security.cert.X509Certificate;
import java.util.Map;
import java.util.Properties;
import java.util.function.Function;

import org.gradle.api.GradleException;
import org.gradle.api.Project;
import org.gradle.api.file.FileTree;
import org.gradle.api.internal.ConventionTask;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.InputFile;
import org.gradle.api.tasks.InputFiles;
import org.gradle.api.tasks.Internal;
import org.gradle.api.tasks.Optional;
import org.gradle.api.tasks.OutputFile;
import org.gradle.api.tasks.TaskAction;

import com.android.build.gradle.tasks.PackageApplication;
import com.android.builder.core.AndroidBuilder;
import com.android.builder.model.SigningConfig;
import com.android.ide.common.signing.CertificateInfo;
import com.android.ide.common.signing.KeystoreHelper;

/**
 * Task to modify the APK file.
 */
public class AutoInstrumentTask extends ConventionTask {
	/**
	 * APK file to modify
	 */
	private File apkFile;

	/**
	 * Application id in dynatrace
	 */
	private String applicationId;

	/**
	 * Environment id in dynatrace
	 */
	private String environmentId;

	/**
	 * Cluster to report data to
	 */
	private String cluster;

	/**
	 * Environment id in dynatrace
	 */
	private String startupPath;

	/**
	 * Additional agent properties
	 */
	private Map<String, String> agentProperties;

	private SigningConfig signingConfig;

	private File apkitDir;

	/**
	 * The instrumented APK
	 */
	private File outputFile;

	private AndroidPluginVersion androidPluginVersion = AndroidPluginVersion.VERSION_1_5;
	private PackageApplication packageTask;

	@TaskAction
	public void instrument() {
		boolean isAppMon = getEnvironmentId() == null;

		Properties tmpProperties = new Properties();
		tmpProperties.putAll(getAgentProperties());

		if (getApplicationId() != null) {
			tmpProperties.put("DTXApplicationID", getApplicationId());
		}

		if (!isAppMon) {
			if (getEnvironmentId() != null) {
				tmpProperties.put("DTXAgentEnvironment", getEnvironmentId());
			}
			if (getCluster() != null) {
				tmpProperties.put("DTXClusterURL", getCluster());
			}
		} else {
			if (getStartupPath() != null) {
				tmpProperties.put("DTXAgentStartupPath", getStartupPath());
			}
		}

		final File propertyFile = new File(getTemporaryDir(), "instrument.properties");
		try (BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(propertyFile))) {
			tmpProperties.store(out, "Autogenerated properties");
		} catch (IOException ex) {
			throw new GradleException("Exception writing instrument.properties", ex);
		}

		final String executableName = System.getProperty("os.name").toLowerCase().contains("windows") ?
				"instrument.cmd" :
				"instrument.sh";

		final File instrumentExecutable = getProject().fileTree(getApkitDir())
				.filter(file -> executableName.equalsIgnoreCase(file.getName())).getSingleFile();

		instrumentExecutable.setExecutable(true);

		if (getApkFile() == null || !getApkFile().getName().endsWith(".apk")) {
			throw new GradleException("Task input contains an invalid apk file: " + getApkFile());
		}

		String dirName = getApkFile().getName().substring(0, getApkFile().getName().lastIndexOf(".apk"));
		File instrumentDist = new File(getApkFile().getParentFile(), dirName + "/dist");
		File instrumentDirectory = new File(getApkFile().getParentFile(), dirName);
		File instrumentedAPK = new File(instrumentDist, getApkFile().getName());

		// clean up instrumentation files
		if (instrumentDirectory.exists()) {
			getProject().delete(instrumentDirectory);
		}

		getProject().exec(execSpec -> {
			execSpec.setExecutable(instrumentExecutable);
			execSpec.setWorkingDir((Object) getTemporaryDir());
			execSpec.args("apk=" + getApkFile().getAbsolutePath(), "prop=" + propertyFile.getAbsolutePath());
		});

		if (!instrumentedAPK.exists()) {
			throw new IllegalStateException("auto-instrumentation failed");
		}

		try {

			AndroidBuilder androidBuilder;
			String methodBuilderClass = androidPluginVersion.isOlderThan(AndroidPluginVersion.VERSION_3_1) ?
					"com.android.build.gradle.internal.tasks.BaseTask" :
					"com.android.build.gradle.internal.tasks.AndroidBuilderTask";

			Method methodBuilder = Class.forName(methodBuilderClass).getDeclaredMethod("getBuilder");
			methodBuilder.setAccessible(true);
			androidBuilder = (AndroidBuilder) methodBuilder.invoke(packageTask);


			if (androidPluginVersion == AndroidPluginVersion.VERSION_1_5) {
				Method methodSignApk = AndroidBuilder.class
						.getDeclaredMethod("signApk", File.class, SigningConfig.class, File.class);
				methodSignApk.setAccessible(true);
				methodSignApk.invoke(androidBuilder, instrumentedAPK, getSigningConfig(), getOutputFile());
			} else {
				String mainFileName = getApkFile().getAbsolutePath();
				String origFileName = mainFileName.substring(0, mainFileName.length() - 4) + "_uninstrumented.apk";
				getApkFile().renameTo(new File(origFileName));

				Class classPackageAndroidArtifact = Class.forName("com.android.build.gradle.tasks.PackageAndroidArtifact");

				Method methodMinSdk = classPackageAndroidArtifact.getDeclaredMethod("getMinSdkVersion");
				methodMinSdk.setAccessible(true);
				int minSdkVersion = (int) methodMinSdk.invoke(packageTask);

				Method methodDebugBuild = classPackageAndroidArtifact.getDeclaredMethod("getDebugBuild");
				methodDebugBuild.setAccessible(true);
				boolean debugBuild = (boolean) methodDebugBuild.invoke(packageTask);

				Method methodCreatedBy = AndroidBuilder.class.getDeclaredMethod("getCreatedBy");
				methodCreatedBy.setAccessible(true);
				String createdBy = (String) methodCreatedBy.invoke(androidBuilder);

				PrivateKey key;
				X509Certificate certificate;
				boolean v1SigningEnabled;
				boolean v2SigningEnabled;

				if (signingConfig != null && signingConfig.isSigningReady()) {
					CertificateInfo certificateInfo =
							KeystoreHelper.getCertificateInfo(
									signingConfig.getStoreType(),
									checkNotNull(signingConfig.getStoreFile()),
									checkNotNull(signingConfig.getStorePassword()),
									checkNotNull(signingConfig.getKeyPassword()),
									checkNotNull(signingConfig.getKeyAlias()));
					key = certificateInfo.getKey();
					certificate = certificateInfo.getCertificate();

					Method methodIsV1 = SigningConfig.class.getMethod("isV1SigningEnabled");
					methodIsV1.setAccessible(true);
					Method methodIsV2 = SigningConfig.class.getMethod("isV2SigningEnabled");
					methodIsV2.setAccessible(true);

					v1SigningEnabled = (boolean) methodIsV1.invoke(signingConfig);
					v2SigningEnabled = (boolean) methodIsV2.invoke(signingConfig);
				} else {
					key = null;
					certificate = null;
					v1SigningEnabled = false;
					v2SigningEnabled = false;
				}

				String packageName = androidPluginVersion == AndroidPluginVersion.VERSION_2_2 ?
						"com.android.builder.packaging." :
						"com.android.apkzlib.zfile.";
				Class classNativeLibrariesPackagingMode = Class.forName(packageName + "NativeLibrariesPackagingMode");
				Class classCreationData = Class.forName(packageName + "ApkCreatorFactory$CreationData");
				Constructor constructor = classCreationData
						.getDeclaredConstructor(File.class, PrivateKey.class, X509Certificate.class, boolean.class, boolean.class,
								String.class, String.class, int.class, classNativeLibrariesPackagingMode,
								java.util.function.Predicate.class);

				Class classPackagingUtils = Class.forName("com.android.builder.packaging.PackagingUtils");
				Method methodNativeLibraries = classPackagingUtils
						.getDeclaredMethod("getNativeLibrariesLibrariesPackagingMode", File.class);
				methodNativeLibraries.setAccessible(true);

				Object nativeLibrariesPackagingMode;
				Object noCompressPredicate;
				if (androidPluginVersion == AndroidPluginVersion.VERSION_3_0
						|| androidPluginVersion == AndroidPluginVersion.VERSION_3_1) {
					//TODO split feature not supported: see ONE-9418
					// we are currently using default values
					nativeLibrariesPackagingMode = classNativeLibrariesPackagingMode.getEnumConstants()[0];
					noCompressPredicate = (java.util.function.Predicate<String>) s -> false;
				} else {
					Field fieldManifest = classPackageAndroidArtifact.getDeclaredField("manifest");
					fieldManifest.setAccessible(true);
					File manifest = (File) fieldManifest.get(packageTask);
					nativeLibrariesPackagingMode = methodNativeLibraries.invoke(null, manifest);

					Method methodGetNoCompressPredicate = classPackageAndroidArtifact.getDeclaredMethod("getNoCompressPredicate");
					methodGetNoCompressPredicate.setAccessible(true);
					if (androidPluginVersion == AndroidPluginVersion.VERSION_2_3) {
						noCompressPredicate = methodGetNoCompressPredicate.invoke(packageTask);
					} else {
						noCompressPredicate = ((java.util.function.Predicate<String>) ((com.google.common.base.Predicate<String>) methodGetNoCompressPredicate
								.invoke(packageTask))::apply);
					}
				}

				Object creationData = constructor.newInstance(
						getOutputFile(),
						key,
						certificate,
						v1SigningEnabled,
						v2SigningEnabled,
						null, // BuiltBy
						createdBy,
						minSdkVersion,
						nativeLibrariesPackagingMode,
						noCompressPredicate);

				Class classApkCreatorFactories = Class.forName("com.android.build.gradle.internal.packaging.ApkCreatorFactories");
				Method methodFromProjectProperties = classApkCreatorFactories
						.getDeclaredMethod("fromProjectProperties", Project.class, boolean.class);
				methodFromProjectProperties.setAccessible(true);
				Object factory = methodFromProjectProperties.invoke(null, getProject(), debugBuild);

				Class classApkCreatorFactory = Class.forName(packageName + "ApkCreatorFactory");
				Method methodMake = classApkCreatorFactory.getDeclaredMethod("make", classCreationData);
				methodMake.setAccessible(true);

				try (Closeable creator = (Closeable) methodMake.invoke(factory, creationData)) {
					Method methodWriteZip = creator.getClass()
							.getDeclaredMethod("writeZip", File.class, Function.class, java.util.function.Predicate.class);
					methodWriteZip.setAccessible(true);
					methodWriteZip.invoke(creator, instrumentedAPK, null, null);
				}

				getProject().copy(copy -> {
					copy.from(getOutputFile());
					copy.into(getApkFile().getParentFile());
				});
			}
		} catch (Exception ex) {
			throw new GradleException("Failed to sign apk", ex);
		}
	}

	@InputFile
	public File getApkFile() {
		return apkFile;
	}

	public void setApkFile(File apkFile) {
		this.apkFile = apkFile;
	}

	@Input
	@Optional
	public String getApplicationId() {
		return applicationId;
	}

	public void setApplicationId(String applicationId) {
		this.applicationId = applicationId;
	}

	@Input
	@Optional
	public String getEnvironmentId() {
		return environmentId;
	}

	public void setEnvironmentId(String environmentId) {
		this.environmentId = environmentId;
	}

	@Input
	@Optional
	public String getCluster() {
		return cluster;
	}

	public void setCluster(String cluster) {
		this.cluster = cluster;
	}

	@Input
	@Optional
	public String getStartupPath() {
		return startupPath;
	}

	public void setStartupPath(String startupPath) {
		this.startupPath = startupPath;
	}

	@Input
	public Map<String, String> getAgentProperties() {
		return agentProperties;
	}

	public void setAgentProperties(Map<String, String> agentProperties) {
		this.agentProperties = agentProperties;
	}

	public SigningConfig getSigningConfig() {
		return signingConfig;
	}

	public void setSigningConfig(SigningConfig signingConfig) {
		this.signingConfig = signingConfig;
	}

	@Internal
	//@InputDirectory // after dropping gradle 2.x support
	public File getApkitDir() {
		return apkitDir;
	}

	// only used for gradle 2.x compatible up-to-date check of directory contents
	// method can be removed and replaced with @InputDirectory on the plain getter after dropping 2.x support
	@Optional
	@InputFiles
	FileTree getApkitFiles() {
		if (apkitDir != null) {
			return getProject().fileTree(apkitDir);
		}
		return null;
	}

	public void setApkitDir(File apkitDir) {
		this.apkitDir = apkitDir;
	}

	@OutputFile
	public File getOutputFile() {
		return outputFile;
	}

	public void setOutputFile(File outputFile) {
		this.outputFile = outputFile;
	}

	@Input
	public AndroidPluginVersion getAndroidPluginVersion() {
		return androidPluginVersion;
	}

	public void setAndroidPluginVersion(AndroidPluginVersion androidPluginVersion) {
		this.androidPluginVersion = androidPluginVersion;
	}

	public PackageApplication getPackageTask() {
		return packageTask;
	}

	public void setPackageTask(PackageApplication packageTask) {
		this.packageTask = packageTask;
	}
}
