package com.drand.bottomnavigation;
/*
 * Bottom navigation library for Android
 * Guidelines and reference: 
 * https://www.google.com/design/spec/components/bottom-navigation.html
 * 
 * Copyright (c) 2016 Alberto Dallaporta (https://github.com/39otrebla).
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import android.content.Context;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.graphics.Typeface;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.GradientDrawable;
import android.support.annotation.DrawableRes;
import android.support.annotation.IntDef;
import android.support.annotation.MenuRes;
import android.support.annotation.StringRes;
import android.support.v4.view.ViewCompat;
import android.support.v7.widget.AppCompatImageView;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;

public class BottomNavigationLayout extends ViewGroup {
    private static final String TAG = "BottomNavigationLayout";

    /**********************************************************
     * ENV MODE (future implementation)
     * ********************************************************/
    public static final int ENV_STANDARD = 0;
    public static final int ENV_REACT_NATIVE = 1;
    @IntDef({ENV_STANDARD, ENV_REACT_NATIVE})
    @Retention(RetentionPolicy.SOURCE)
    public @interface Env {}

    /**********************************************************
     * TAB MODE (future implementation)
     * ********************************************************/
    private static final int MODE_DEFAULT = 0;
    public static final int MODE_FIXED = 1;
    public static final int MODE_SHIFTING = 2;
    public static final int MODE_FORCE_FIXED = 3;

    @IntDef({MODE_DEFAULT, MODE_FIXED, MODE_SHIFTING, MODE_FORCE_FIXED})
    @Retention(RetentionPolicy.SOURCE)
    public @interface Mode {}

    /**********************************************************
     * TAB STATE (future implementation)
     * ********************************************************/
    /**
     * Indicates settled state. No animation is in progress.
     */
    public static final int STATE_IDLE = 0;
    /**
     * Indicates that a tab is currently being selected by the user.
     */
    public static final int STATE_SELECTING = 1;
    /**
     * Indicates that a tab is in the process of settling to a final state.
     */
    public static final int STATE_SETTLING = 2;

    @IntDef({STATE_IDLE, STATE_SELECTING, STATE_SETTLING})
    @Retention(RetentionPolicy.SOURCE)
    private @interface State {}

    /**********************************************************
     * CONSTANTS
     * ********************************************************/
    private static final int CONTENT_INDEX = 0;
    private static final int TABBAR_INDEX = 1;
    private static final int INJECTED_CONTENT_INDEX = 0;
    private static final int SHADOW_INDEX = 1;
    /**********************************************************
     * MATERIAL DESIGN SPECS
     * google.com/design/spec/components/bottom-navigation.html
     * ********************************************************/
    public static final int MIN_ITEMS = 3;
    public static final int MAX_ITEMS = 5;
    private static final int ELEVATION = 8; // dp
    private static final int BAR_HEIGHT = 56; // dp
    private static final int MAX_TAB_WIDTH_FIXED = 168; // dp
    private static final int MIN_TAB_WIDTH_FIXED = 80; // dp
    private static final int MAX_TAB_WIDTH_SHIFTING = 168; // dp
    private static final int MIN_TAB_WIDTH_SHIFTING = 96; // dp
    private static final int PADDING_TOP_ACTIVE_FIXED = 6; // dp
    private static final int PADDING_TOP_INACTIVE_FIXED = 8; // dp
    private static final int PADDING_TOP_INACTIVE_SHIFTING = 16; // dp
    private static final int PADDING_BOTTOM = 10; // dp
    private static final int PADDING_BOTTOM_SHIFTING_INACTIVE = 10; // dp
    private static final int PADDING_HORIZONTAL = 12; // dp
    private static final int PADDING_BOTTOM_ICON_SHIFTING_ACTIVE = 6; // dp
    // let devs override these values because on certain languages
    // some labels are very long (f.e. "Impostazioni" in italian)
    private int FONT_SIZE_ACTIVE_FIXED = 14; // sp
    private int FONT_SIZE_INACTIVE_FIXED = 12; // sp
    private int FONT_SIZE_SHIFTING = 14; // sp
    // let devs override (and even hide) the shadow
    private int SHADOW_HEIGHT = 2; // dp

    private Context mContext;
    @Env
    private int mEnv = ENV_REACT_NATIVE;
    @Mode
    private int mMode = MODE_DEFAULT;
    private int mSelected = 0;
    private int mFontStyle = Typeface.NORMAL;
    private float mDensity;
    private float mElevation;
    private boolean mInLayout = false;
    private boolean mFirstLayout = true;
    private String mFontFamily = "sans-serif-condensed";
    private String mBackgroundColor;
    private String mActiveTabTintColor;
    private String mInactiveTabTintColor;
    private List<Tab> mTabs;
    private List<View> mContent;
    private ImageView mShadowView;
    private LinearLayout mTabsContainer;
    private FrameLayout mContentContainer;
    private FrameLayout mTabsOuterContainer;
    private BottomNavigationListener mListener;

    /**********************************************************
     * EVENTS LISTENERS
     * ********************************************************/
    public interface BottomNavigationListener {
        void onTabSelected(int position);
        void onTabReSelected(int position);
    }

    public static abstract class SimpleBottomNavigationListener implements BottomNavigationListener {
        @Override
        public void onTabReSelected(int position) {}

        @Override
        public void onTabSelected(int position) {}
    }

    /**********************************************************
     * CONSTRUCTORS
     * ********************************************************/
    public BottomNavigationLayout(Context context) {
        this(context, null, 0);
    }

    public BottomNavigationLayout(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public BottomNavigationLayout(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);

        mContext = context;
        mDensity = getResources().getDisplayMetrics().density;
        mElevation = mDensity * ELEVATION;

        setClipToPadding(false);
        setFocusableInTouchMode(true);
        ViewCompat.setElevation(this, mElevation);

        LayoutParams lp = new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);

        mContentContainer = new FrameLayout(getContext());
        mContentContainer.setLayoutParams(lp);
        mContentContainer.setBackgroundColor(Color.TRANSPARENT);

        mTabsOuterContainer = new FrameLayout(getContext());
        mTabsOuterContainer.setLayoutParams(lp);
        mTabsOuterContainer.setBackgroundColor(Color.TRANSPARENT);

        mTabsContainer = new LinearLayout(context);
        LayoutParams lpTabbar = new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);
        mTabsContainer.setLayoutParams(lpTabbar);
        mTabsContainer.setOrientation(LinearLayout.HORIZONTAL);
        mTabsContainer.setGravity(Gravity.CENTER_HORIZONTAL);

        mShadowView = new ImageView(context);
        FrameLayout.LayoutParams lpShadow = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, dpToPixels(SHADOW_HEIGHT));
        lpShadow.gravity = Gravity.BOTTOM;
        mShadowView.setLayoutParams(lpShadow);
        mShadowView.setImageDrawable(getShadowDrawable());

        mTabs = new ArrayList<>();
        mContent = new ArrayList<>();

        addView(mContentContainer, CONTENT_INDEX);
        addView(mTabsOuterContainer, TABBAR_INDEX);
    }

    /**********************************************************
     * ACTUAL IMPLEMENTATION
     * ********************************************************/

    public void createViewGroup() {

        if (mTabs.size() > MAX_ITEMS) {
            throw new IllegalArgumentException("BottomNavigationLayout can have at most " + MAX_ITEMS + " tabs");
        }

        if (mContent.size() != mTabs.size() && mEnv != ENV_REACT_NATIVE) {
            int tabs = mTabs.size();
            int content = mContent.size();

            throw new IllegalArgumentException("The number of tabs (" + tabs + ") does not match the number of content views (" + content + ")");
        }

        if (mBackgroundColor != null) mTabsContainer.setBackgroundColor(Color.parseColor(mBackgroundColor));

        // we can do the width calcs only here,
        // when the total number of tabs is known
        final int mostLikelyWidth = getResources().getDisplayMetrics().widthPixels / mTabs.size();

        if (mMode == MODE_DEFAULT) {
            mMode = mTabs.size() <= MIN_ITEMS ? MODE_FIXED : MODE_SHIFTING;
        }

        for (int i = 0; i < mTabs.size(); i++) {
            int width = 0;
            if (mMode != MODE_SHIFTING) {
                width = Math.max(dpToPixels(MIN_TAB_WIDTH_FIXED), mostLikelyWidth);
                width = Math.min(dpToPixels(MAX_TAB_WIDTH_FIXED), mostLikelyWidth);
            } else {
                width = Math.max(dpToPixels(MIN_TAB_WIDTH_SHIFTING), mostLikelyWidth);
                width = Math.min(dpToPixels(MAX_TAB_WIDTH_SHIFTING), mostLikelyWidth);
            }

            mTabs.get(i).setWidth(width);

            if (i == mSelected) mTabs.get(i).select();
            else mTabs.get(i).deselect();

            mTabsContainer.addView(mTabs.get(i));
        }

        mTabsOuterContainer.addView(mTabsContainer, 0);

        if (mEnv == ENV_STANDARD) {
            mContentContainer.addView(mContent.get(mSelected), INJECTED_CONTENT_INDEX);
            mContentContainer.addView(mShadowView, SHADOW_INDEX);
        }
    }

    public void addItem(String title, String icon, final int index) {

        if (mEnv != ENV_REACT_NATIVE) throw new IllegalArgumentException("addItem(String, String, int) must be used only with ReactNative");

        Tab tab = new Tab(mContext, title, icon, index);
        tab.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View view) {
                selectTab(index);
            }
        });
        mTabs.add(index, tab);
    }

    public void addItem(@StringRes int title, @DrawableRes int icon, final int index) {
        Tab tab = new Tab(mContext, title, icon, index);
        tab.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View view) {
                selectTab(index);
            }
        });
        mTabs.add(index, tab);
    }

    public void addItem(String title, @DrawableRes int icon, final int index) {
        Tab tab = new Tab(mContext, title, icon, index);
        tab.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View view) {
                selectTab(index);
            }
        });
        mTabs.add(index, tab);
    }

    public void addContent(View view, int index) {
        mContent.add(index, view);
    }

    public void addReactNativeContent(View view) {
        if (mContentContainer.getChildCount() > 0) mContentContainer.removeAllViews();

        mContentContainer.addView(view, INJECTED_CONTENT_INDEX);
        mContentContainer.addView(mShadowView, SHADOW_INDEX);
    }

    public void hideShadow() {
        if (mContentContainer.getChildAt(SHADOW_INDEX) != null) {
            mContentContainer.getChildAt(SHADOW_INDEX).setVisibility(GONE);
        }
    }

    public void showShadow() {
        if (mContentContainer.getChildAt(SHADOW_INDEX) != null) {
            mContentContainer.getChildAt(SHADOW_INDEX).setVisibility(VISIBLE);
        }
    }

    private void selectTab(int index) {
        if (index != mSelected) {
            // deselect currently selected tab
            mTabs.get(mSelected).deselect();
            // select new tab
            mSelected = index;
            mTabs.get(index).select();

            if (mEnv != ENV_REACT_NATIVE) {
                // update content
                mContentContainer.removeViewAt(INJECTED_CONTENT_INDEX);
                mContentContainer.addView(mContent.get(index), INJECTED_CONTENT_INDEX);
            }

            dispatchOnTabSelected(index);
        } else {
            dispatchOnTabReSelected(index);
        }
    }

    /**********************************************************
     * GETTERS
     * ********************************************************/

    /**********************************************************
     * SETTERS
     * ********************************************************/

    public void setEnv(int env) {
        mEnv = env;
    }

    public void setBackgroundColor(String backgroundColor) {
        mBackgroundColor = backgroundColor;
    }

    public void setMode(int mode) {
        mMode = mode;
    }

    public void setActiveTabTintColor(String activeTabTintColor) {
        mActiveTabTintColor = activeTabTintColor;
    }

    public void setInactiveTabTintColor(String inactiveTabTintColor) {
        mInactiveTabTintColor = inactiveTabTintColor;
    }

    public void setFontSizeActiveTabFixed(int size) {
        FONT_SIZE_ACTIVE_FIXED = size;
    }

    public void setFontSizeInActiveTabFixed(int size) {
        FONT_SIZE_INACTIVE_FIXED = size;
    }

    public void setFontSizeTabShifting(int size) {
        FONT_SIZE_SHIFTING = size;
    }

    public void setFontFamily(String font) {
        mFontFamily = font;

        if (mTabs.size() > 0) {
            for (int i = 0; i < mTabs.size(); i++) {
                mTabs.get(i).getTitleView().setTypeface(
                        Typeface.create(font, mFontStyle)
                );
            }
        }
    }

    public void setFontStyle(String style) {
        switch (style) {
            case "bold": mFontStyle = Typeface.BOLD; break;
            case "italic": mFontStyle = Typeface.ITALIC; break;
            case "bold_italic": mFontStyle = Typeface.BOLD_ITALIC; break;
            default: mFontStyle = Typeface.NORMAL;
        }

        if (mTabs.size() > 0) {
            for (int i = 0; i < mTabs.size(); i++) {
                mTabs.get(i).getTitleView().setTypeface(
                        Typeface.create(mFontFamily, mFontStyle)
                );
            }
        }
    }

    public void setShadowHeight(int height) {
        SHADOW_HEIGHT = height;
        FrameLayout.LayoutParams lpShadow = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, dpToPixels(height));
        lpShadow.gravity = Gravity.BOTTOM;
        mShadowView.setLayoutParams(lpShadow);
    }

    public void setBottomNavigationListener(BottomNavigationListener listener) {
        mListener = listener;
    }

    /**********************************************************
     * EVENT DISPATCHERS
     * ********************************************************/
    void dispatchOnTabSelected(int index) {
        if (mListener != null) {
            mListener.onTabSelected(index);
        }
    }

    void dispatchOnTabReSelected(int index) {
        if (mListener != null) {
            mListener.onTabReSelected(index);
        }
    }

    /**********************************************************
     * OVERRIDDEN METHODS
     * ********************************************************/
    @Override
    public void addView(View child) {
        if (getChildCount() < 2) super.addView(child);
        else logSkippedAddView();
    }

    @Override
    public void addView(View child, int index) {
        if (getChildCount() < 2) super.addView(child, index);
        else logSkippedAddView();
    }

    @Override
    public void addView(View child, int index, LayoutParams params) {
        if (getChildCount() < 2) super.addView(child, index, params);
        else logSkippedAddView();
    }

    @Override
    public void addView(View child, LayoutParams params) {
        if (getChildCount() < 2) super.addView(child, params);
        else logSkippedAddView();
    }

    @Override
    public void addView(View child, int width, int height) {
        if (getChildCount() < 2) super.addView(child, width, height);
        else logSkippedAddView();
    }

    @Override
    protected boolean checkLayoutParams(ViewGroup.LayoutParams p) {
        return p instanceof LayoutParams && super.checkLayoutParams(p);
    }

    @Override
    protected LayoutParams generateDefaultLayoutParams() {
        return new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);
    }

    @Override
    public LayoutParams generateLayoutParams(ViewGroup.LayoutParams p) {
        return p instanceof LayoutParams
                ? new LayoutParams((LayoutParams) p)
                : p instanceof ViewGroup.MarginLayoutParams
                ? new LayoutParams((MarginLayoutParams) p)
                : new LayoutParams(p);
    }

    @Override
    public ViewGroup.LayoutParams generateLayoutParams(AttributeSet attrs) {
        return new LayoutParams(getContext(), attrs);
    }

    @Override
    protected void onLayout(boolean changed, int l, int t, int r, int b) {
        mInLayout = true;

        final View content = getChildAt(CONTENT_INDEX);
        final View tabbar = getChildAt(TABBAR_INDEX);

        if (content.getVisibility() != GONE) {
            content.layout(0, 0, content.getMeasuredWidth(), content.getMeasuredHeight());
        }

        if (tabbar.getVisibility() != GONE) {
            tabbar.layout(0, content.getMeasuredHeight(), tabbar.getMeasuredWidth(), b - t);
        }

        mInLayout = false;
        mFirstLayout = false;
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        int widthMode = MeasureSpec.getMode(widthMeasureSpec);
        int heightMode = MeasureSpec.getMode(heightMeasureSpec);
        int widthSize = MeasureSpec.getSize(widthMeasureSpec);
        int heightSize = MeasureSpec.getSize(heightMeasureSpec);

        // borrowed from DrawerLayout
        // (https://android.googlesource.com/platform/frameworks/support/+/refs/heads/master/v4/java/android/support/v4/widget/DrawerLayout.java)
        if (widthMode != MeasureSpec.EXACTLY || heightMode != MeasureSpec.EXACTLY) {
            if (isInEditMode()) {
                // Don't crash the layout editor. Consume all of the space if specified
                // or pick a magic number from thin air otherwise.
                // TODO Better communication with tools of this bogus state.
                // It will crash on a real device.
                if (widthMode == MeasureSpec.AT_MOST) {
                    widthMode = MeasureSpec.EXACTLY;
                } else if (widthMode == MeasureSpec.UNSPECIFIED) {
                    widthMode = MeasureSpec.EXACTLY;
                    widthSize = 300;
                }
                if (heightMode == MeasureSpec.AT_MOST) {
                    heightMode = MeasureSpec.EXACTLY;
                }
                else if (heightMode == MeasureSpec.UNSPECIFIED) {
                    heightMode = MeasureSpec.EXACTLY;
                    heightSize = 300;
                }
            } else {
                throw new IllegalArgumentException(
                        "DrawerLayout must be measured with MeasureSpec.EXACTLY.");
            }
        }

        setMeasuredDimension(widthSize, heightSize);

        final View content = getChildAt(CONTENT_INDEX);
        final View tabbar = getChildAt(TABBAR_INDEX);

        content.measure(
                MeasureSpec.makeMeasureSpec(widthSize, MeasureSpec.EXACTLY),
                MeasureSpec.makeMeasureSpec(heightSize - dpToPixels(BAR_HEIGHT), MeasureSpec.EXACTLY)
        );

        tabbar.measure(
                MeasureSpec.makeMeasureSpec(widthSize, MeasureSpec.EXACTLY),
                MeasureSpec.makeMeasureSpec(dpToPixels(BAR_HEIGHT), MeasureSpec.EXACTLY)
        );
    }

    @Override
    public void requestLayout() {
        if (!mInLayout) super.requestLayout();
    }

    /**********************************************************
     * UTILS
     * ********************************************************/
    private static void logSkippedAddView() {
        Log.w(TAG, TAG + " already has 2 children, addView() is a no-op");
    }

    private int dpToPixels(int dp) {
        return (int)(dp * mDensity + 0.5f);
    }

    private Drawable getShadowDrawable() {
        GradientDrawable gd = new GradientDrawable(
                GradientDrawable.Orientation.TOP_BOTTOM, // orientation
                new int[] {0x0D000000, 0x00000000}
        );
        gd.setGradientType(GradientDrawable.LINEAR_GRADIENT);
        return gd;
    }

    /**********************************************************
     * INNER CLASSES
     * ********************************************************/
    private class Tab extends FrameLayout {
        private static final int ICON_INDEX = 0;
        private static final int TITLE_INDEX = 1;

        private String mTitle;
        private int mIconRes;
        private TextView mTitleView;
        private AppCompatImageView mDrawable;

        private int position = -1;

        private Tab(Context context, String title, String iconName, int index) {
            super(context);

            mTitle = title;
            mIconRes = context.getResources().getIdentifier(iconName, "drawable", context.getPackageName());
            position = index;
            init(context);
        }

        private Tab(Context context, String title, @DrawableRes int iconRes, int index) {
            super(context);

            mTitle = title;
            mIconRes = iconRes;
            position = index;
            init(context);
        }

        private Tab(Context context, @StringRes int titleRes, @DrawableRes int iconRes, int index) {
            super(context);

            mTitle = context.getString(titleRes);
            mIconRes = iconRes;
            position = index;
            init(context);
        }

        private void init(Context context) {

            setLayoutParams(new LinearLayout.LayoutParams(LinearLayout.LayoutParams.WRAP_CONTENT, dpToPixels(BAR_HEIGHT)));
            setClickable(true);

            mTitleView = new TextView(context);
            mDrawable = new AppCompatImageView(context);

            FrameLayout.LayoutParams lpTitle = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.WRAP_CONTENT, FrameLayout.LayoutParams.WRAP_CONTENT);
            FrameLayout.LayoutParams lpIcon = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.WRAP_CONTENT, FrameLayout.LayoutParams.WRAP_CONTENT);

            // layout
            lpTitle.gravity = Gravity.BOTTOM | Gravity.CENTER_HORIZONTAL;
            mTitleView.setLayoutParams(lpTitle);
            lpIcon.gravity = Gravity.TOP | Gravity.CENTER_HORIZONTAL;
            mDrawable.setLayoutParams(lpIcon);

            // style
            mTitleView.setTypeface(Typeface.create(mFontFamily, Typeface.NORMAL));

            // content
            mTitleView.setText(mTitle);
            mTitleView.setSingleLine(true);
            mTitleView.setMaxLines(1);

            if (mIconRes != 0) mDrawable.setImageResource(mIconRes);

            addView(mDrawable, ICON_INDEX);
            addView(mTitleView, TITLE_INDEX);
        }

        private void select() {
            if (mMode == MODE_SHIFTING) {
                if (getChildCount() == 1) addView(mTitleView, TITLE_INDEX);
                setPadding(dpToPixels(PADDING_HORIZONTAL), dpToPixels(PADDING_TOP_ACTIVE_FIXED), dpToPixels(PADDING_HORIZONTAL), dpToPixels(PADDING_BOTTOM));
                mDrawable.setPadding(0, 0, 0, dpToPixels(PADDING_BOTTOM_ICON_SHIFTING_ACTIVE));
            } else {
                setPadding(dpToPixels(PADDING_HORIZONTAL), dpToPixels(PADDING_TOP_ACTIVE_FIXED), dpToPixels(PADDING_HORIZONTAL), dpToPixels(PADDING_BOTTOM));
            }
            setTintActive();
            setFontSize(true);

            if (mEnv == ENV_REACT_NATIVE) mTabsContainer.requestLayout();
        }

        private void deselect() {
            if (mMode == MODE_SHIFTING) {
                removeViewAt(TITLE_INDEX);
                setPadding(dpToPixels(PADDING_HORIZONTAL), dpToPixels(PADDING_TOP_INACTIVE_SHIFTING), dpToPixels(PADDING_HORIZONTAL), dpToPixels(PADDING_BOTTOM_SHIFTING_INACTIVE));
                mDrawable.setPadding(0, 0, 0, 0);
            } else {
                setPadding(dpToPixels(PADDING_HORIZONTAL), dpToPixels(PADDING_TOP_INACTIVE_FIXED), dpToPixels(PADDING_HORIZONTAL), dpToPixels(PADDING_BOTTOM));
            }
            setTintInactive();
            setFontSize(false);

            if (mEnv == ENV_REACT_NATIVE) mTabsContainer.requestLayout();
        }

        private void setWidth(int width) {
            setLayoutParams(new LinearLayout.LayoutParams(width, dpToPixels(BAR_HEIGHT)));
        }

        private void setFontSize(boolean isSelected) {
            if (mMode != MODE_SHIFTING) {
                if (isSelected) mTitleView.setTextSize(FONT_SIZE_ACTIVE_FIXED);
                else mTitleView.setTextSize(FONT_SIZE_INACTIVE_FIXED);
            } else {
                mTitleView.setTextSize(FONT_SIZE_SHIFTING);
            }
        }

        private TextView getTitleView() {
            return mTitleView;
        }

        private void setTintActive() {
            if (mActiveTabTintColor != null) {
                mTitleView.setTextColor(Color.parseColor(mActiveTabTintColor));
                mDrawable.setColorFilter(Color.parseColor(mActiveTabTintColor));
            }
        }

        private void setTintInactive() {
            if (mInactiveTabTintColor != null) {
                mTitleView.setTextColor(Color.parseColor(mInactiveTabTintColor));
                mDrawable.setColorFilter(Color.parseColor(mInactiveTabTintColor));
            }
        }

        private int getPosition() {
            return position;
        }

        private Drawable getIcon() {
            return mDrawable.getDrawable();
        }

        private int getIconRes() {
            return mIconRes;
        }

        private String getTitle() {
            return mTitle;
        }
    }
}
