package com.devicenative.dna.network;

import android.content.Context;

import com.devicenative.dna.db.DNADatabaseQueue;
import com.google.android.gms.ads.identifier.AdvertisingIdClient;

import com.devicenative.dna.db.DNAShortcutRecord;
import com.devicenative.dna.utils.DNAConstants;
import com.devicenative.dna.db.DNAAdRecord;
import com.devicenative.dna.db.DNADatabaseInterface;
import com.devicenative.dna.db.DNAEventRecord;
import com.devicenative.dna.utils.DNALogger;
import com.devicenative.dna.utils.DNAPreferences;
import com.devicenative.dna.utils.DNAStatsLogger;
import com.devicenative.dna.utils.DNAUtils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

public class DNANetworkRequestSync extends DNANetworkRequest {

    long startTime;

    /**
     * <p>Create an instance of {@link DNANetworkRequestSync} to synchronize settings and ads.</p>
     *
     * @param context     Current context
     */
    public DNANetworkRequestSync(Context context) {
        super(context, DNAConstants.RequestPath.SYNC);
        try {
            setPost(new JSONObject());
        } catch (JSONException ex) {
            DNALogger.e("DNANetworkRequestSync: " + ex.getMessage());
        }
    }

    @Override
    public void onPreExecute() {
        super.onPreExecute();
        startTime = System.currentTimeMillis();
        JSONObject postBody = new JSONObject();
        try {
            // put all the keys
            postBody.put(DNAConstants.RequestParams.DEVICE_KEY, prefHelper_.getDeviceKey());
            postBody.put(DNAConstants.RequestParams.APP_VERSION, prefHelper_.getAppVersion());
            postBody.put(DNAConstants.RequestParams.APP_PACKAGE_NAME, prefHelper_.getAppPackageName());
            postBody.put(DNAConstants.RequestParams.SDK_VERSION, DNAConstants.DNA_SDK_VERSION);
            postBody.put(DNAConstants.RequestParams.OS_VERSION, prefHelper_.getAndroidVersion());
            postBody.put(DNAConstants.RequestParams.PLATFORM, prefHelper_.getDeviceModel());
            postBody.put(DNAConstants.RequestParams.CARRIER, prefHelper_.getCarrierValue());
            postBody.put(DNAConstants.RequestParams.LAST_SYNC_TIME, prefHelper_.getLastApiSyncTime());
            postBody.put(DNAConstants.RequestParams.BUILD_TIME, prefHelper_.getDeviceStartTime());
            postBody.put(DNAConstants.RequestParams.CPU_COUNT, prefHelper_.getCPUCount());
            postBody.put(DNAConstants.RequestParams.MAX_MEMORY, prefHelper_.getMaxMemory());
            postBody.put(DNAConstants.RequestParams.SESSION_ID, prefHelper_.getSessionID());
            postBody.put(DNAConstants.RequestParams.CODE_VERSION_KEY, prefHelper_.getCodeVersionID());
            postBody.put(DNAConstants.RequestParams.ADS_VERSION_KEY, prefHelper_.getAdBundleID());
            postBody.put(DNAConstants.RequestParams.METADATA_VERSION_KEY, prefHelper_.getMetadataVersionID());
            postBody.put(DNAConstants.RequestParams.BLACKLIST_VERSION_KEY, prefHelper_.getBlacklistVersionID());
            if (prefHelper_.getConfigCountryOverride() != null && !prefHelper_.getConfigCountryOverride().isEmpty()) {
                postBody.put(DNAConstants.RequestParams.COUNTRY_OVERRIDE, prefHelper_.getConfigCountryOverride());
            }
            postBody.put(DNAConstants.RequestParams.CONFIG_GAID_DISABLED, prefHelper_.getConfigGAIDDisabled());
            postBody.put(DNAConstants.RequestParams.CONFIG_USAGE_DISABLED, prefHelper_.getConfigAppUsageDisabled());
            postBody.put(DNAConstants.RequestParams.CONFIG_ADS_DISABLED, prefHelper_.getConfigAdsDisabled());
            postBody.put(DNAConstants.RequestParams.CONFIG_SEARCH_ADS_DISABLED, prefHelper_.getSearchAdsDisabled());
            postBody.put(DNAConstants.RequestParams.CONFIG_SEARCH_INSTALL_DISABLED, prefHelper_.getSearchInstallAdsDisabled());
            postBody.put(DNAConstants.RequestParams.CONFIG_REC_ADS_DISABLED, prefHelper_.getRecomAdsDisabled());
            postBody.put(DNAConstants.RequestParams.CONFIG_REC_INSTALL_ADS_DISABLED, prefHelper_.getRecomInstallAdsDisabled());

            DNALogger.i("DNANetworkRequestSync: onPreExecute: " + postBody);

            setPost(postBody);
        } catch (JSONException ex) {
            DNALogger.e("DNANetworkRequestSync: " + ex.getMessage());
        }
    }

    @Override
    public void onRequestSucceeded(DNANetworkResponse resp) {
        DNADatabaseQueue.enqueue(context_.get(), () -> processSyncResponse(resp));
    }

    private void processSyncResponse(DNANetworkResponse resp) {
        try {
            JSONObject respJson = resp.getBody();

            String currCodeVersion = prefHelper_.getCodeVersionID();
            String currAdBundleID = prefHelper_.getAdBundleID();
            int numAdsAdded = 0;
            int numAdsUpdated = 0;
            int numAdsDeleted = 0;
            int numShortcutsAdded = 0;
            int numShortcutsDeleted = 0;

            // check if the config key is present, and populate the config params if so
            if (respJson.has(DNAConstants.SyncParams.CONFIG)) {
                JSONObject config = respJson.getJSONObject(DNAConstants.SyncParams.CONFIG);

                if (config.has(DNAConstants.SyncParams.REFRESH_INTERVAL) && !config.isNull(DNAConstants.SyncParams.REFRESH_INTERVAL)) {
                    prefHelper_.setRefreshInterval(config.getInt(DNAConstants.SyncParams.REFRESH_INTERVAL));
                }

                if (config.has(DNAConstants.SyncParams.LOCAL_REFRESH_INTERVAL) && !config.isNull(DNAConstants.SyncParams.LOCAL_REFRESH_INTERVAL)) {
                    prefHelper_.setLocalRefreshInterval(config.getInt(DNAConstants.SyncParams.LOCAL_REFRESH_INTERVAL));
                }

                if (config.has(DNAConstants.SyncParams.API_URL) && !config.isNull(DNAConstants.SyncParams.API_URL)) {
                    prefHelper_.setAPIUrl(config.getString(DNAConstants.SyncParams.API_URL));
                }

                if (config.has(DNAConstants.SyncParams.CONNECT_TIMEOUT) && !config.isNull(DNAConstants.SyncParams.CONNECT_TIMEOUT)) {
                    prefHelper_.setConnectTimeout(config.getInt(DNAConstants.SyncParams.CONNECT_TIMEOUT));
                }

                if (config.has(DNAConstants.SyncParams.NETWORK_TIMEOUT) && !config.isNull(DNAConstants.SyncParams.NETWORK_TIMEOUT)) {
                    prefHelper_.setTimeout(config.getInt(DNAConstants.SyncParams.NETWORK_TIMEOUT));
                }

                if (config.has(DNAConstants.SyncParams.RETRY_INTERVAL) && !config.isNull(DNAConstants.SyncParams.RETRY_INTERVAL)) {
                    prefHelper_.setRetryInterval(config.getInt(DNAConstants.SyncParams.RETRY_INTERVAL));
                }

                if (config.has(DNAConstants.SyncParams.RETRY_COUNT) && !config.isNull(DNAConstants.SyncParams.RETRY_COUNT)) {
                    prefHelper_.setRetryCount(config.getInt(DNAConstants.SyncParams.RETRY_COUNT));
                }

                if (config.has(DNAConstants.SyncParams.VTA_ENABLED) && !config.isNull(DNAConstants.SyncParams.VTA_ENABLED)) {
                    prefHelper_.setVTAEnabled(config.getBoolean(DNAConstants.SyncParams.VTA_ENABLED));
                }

                if (config.has(DNAConstants.SyncParams.INSTALL_ADS_ENABLED) && !config.isNull(DNAConstants.SyncParams.INSTALL_ADS_ENABLED)) {
                    prefHelper_.setInstallAdsEnabled(config.getBoolean(DNAConstants.SyncParams.INSTALL_ADS_ENABLED));
                }

                if (config.has(DNAConstants.SyncParams.SEARCH_ADS_ENABLED) && !config.isNull(DNAConstants.SyncParams.SEARCH_ADS_ENABLED)) {
                    prefHelper_.setSearchAdsDisabled(!config.getBoolean(DNAConstants.SyncParams.SEARCH_ADS_ENABLED));
                }

                if (config.has(DNAConstants.SyncParams.RECOMMENDATION_ADS_ENABLED) && !config.isNull(DNAConstants.SyncParams.RECOMMENDATION_ADS_ENABLED)) {
                    prefHelper_.setRecomAdsDisabled(!config.getBoolean(DNAConstants.SyncParams.RECOMMENDATION_ADS_ENABLED));
                }

                if (config.has(DNAConstants.SyncParams.SEARCH_INSTALL_ADS_ENABLED) && !config.isNull(DNAConstants.SyncParams.SEARCH_INSTALL_ADS_ENABLED)) {
                    prefHelper_.setSearchInstallAdsDisabled(!config.getBoolean(DNAConstants.SyncParams.SEARCH_INSTALL_ADS_ENABLED));
                }

                if (config.has(DNAConstants.SyncParams.RECOMMENDATION_INSTALL_ADS_ENABLED) && !config.isNull(DNAConstants.SyncParams.RECOMMENDATION_INSTALL_ADS_ENABLED)) {
                    prefHelper_.setRecomInstallAdsDisabled(!config.getBoolean(DNAConstants.SyncParams.RECOMMENDATION_INSTALL_ADS_ENABLED));
                }

                if (config.has(DNAConstants.SyncParams.WEB_ADS_ENABLED) && !config.isNull(DNAConstants.SyncParams.WEB_ADS_ENABLED)) {
                    prefHelper_.setWebAdsEnabled(config.getBoolean(DNAConstants.SyncParams.WEB_ADS_ENABLED));
                }

                if (config.has(DNAConstants.SyncParams.ORGANIC_ADS_ENABLED) && !config.isNull(DNAConstants.SyncParams.ORGANIC_ADS_ENABLED)) {
                    prefHelper_.setOrganicAdsEnabled(config.getBoolean(DNAConstants.SyncParams.ORGANIC_ADS_ENABLED));
                }

                if (config.has(DNAConstants.SyncParams.ONLY_ORGANIC_ADS) && !config.isNull(DNAConstants.SyncParams.ONLY_ORGANIC_ADS)) {
                    prefHelper_.setOrganicAdsOnly(config.getBoolean(DNAConstants.SyncParams.ONLY_ORGANIC_ADS));
                }

                if (config.has(DNAConstants.SyncParams.ORGANIC_ADS_BOOST_ENABLED) && !config.isNull(DNAConstants.SyncParams.ORGANIC_ADS_BOOST_ENABLED)) {
                    prefHelper_.setPaidOrganicBoostEnabled(config.getBoolean(DNAConstants.SyncParams.ORGANIC_ADS_BOOST_ENABLED));
                }

                if (config.has(DNAConstants.SyncParams.IMPRESSION_BLACKOUT_WINDOW) && !config.isNull(DNAConstants.SyncParams.IMPRESSION_BLACKOUT_WINDOW)) {
                    prefHelper_.setImpressionBlackoutWindow(config.getInt(DNAConstants.SyncParams.IMPRESSION_BLACKOUT_WINDOW));
                }

                if (config.has(DNAConstants.SyncParams.AD_CACHE_TIME) && !config.isNull(DNAConstants.SyncParams.AD_CACHE_TIME)) {
                    prefHelper_.setAdCacheTimeout(config.getInt(DNAConstants.SyncParams.AD_CACHE_TIME));
                }

                if (config.has(DNAConstants.SyncParams.REPLACE_GOOGLE_PLAY_URL) && !config.isNull(DNAConstants.SyncParams.REPLACE_GOOGLE_PLAY_URL)) {
                    prefHelper_.setReplaceGooglePlayUrl(config.getBoolean(DNAConstants.SyncParams.REPLACE_GOOGLE_PLAY_URL));
                }

                if (config.has(DNAConstants.SyncParams.PLAY_STORE_REPLACEMENT_URL) && !config.isNull(DNAConstants.SyncParams.PLAY_STORE_REPLACEMENT_URL)) {
                    prefHelper_.setGooglePlayUrlReplacement(config.getString(DNAConstants.SyncParams.PLAY_STORE_REPLACEMENT_URL));
                }

                if (config.has(DNAConstants.SyncParams.ADID_USAGE_ENABLED) && !config.isNull(DNAConstants.SyncParams.ADID_USAGE_ENABLED)) {
                    prefHelper_.setAdidUsageEnabled(config.getBoolean(DNAConstants.SyncParams.ADID_USAGE_ENABLED));
                }

                if (config.has(DNAConstants.SyncParams.APP_USAGE_DISABLED) && !config.isNull(DNAConstants.SyncParams.APP_USAGE_DISABLED)) {
                    prefHelper_.setAppUsageDisabled(config.getBoolean(DNAConstants.SyncParams.APP_USAGE_DISABLED));
                }

                if (config.has(DNAConstants.SyncParams.AUTO_DEEP_LINK_ENABLED) && !config.isNull(DNAConstants.SyncParams.AUTO_DEEP_LINK_ENABLED)) {
                    prefHelper_.setAutoDeepLinkEnabled(config.getBoolean(DNAConstants.SyncParams.AUTO_DEEP_LINK_ENABLED));
                }

                if (prefHelper_.getAdidUsageEnabled() && prefHelper_.getAdid().isEmpty() && !prefHelper_.getConfigGAIDDisabled()) {
                    ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor();
                    scheduler.schedule(() -> {
                        try {
                            AdvertisingIdClient.Info adInfo = AdvertisingIdClient.getAdvertisingIdInfo(context_.get().getApplicationContext());
                            String adId = adInfo.getId();

                            if (adId != null && !adId.equals("00000000-0000-0000-0000-000000000000")) {
                                DNALogger.i("getAdvertisingId: direct Ad ID: " + adId);
                                prefHelper_.setAdid(adId);
                            } else {
                                Class<?> advertisingIdClientClass = Class.forName("com.google.android.gms.ads.identifier.AdvertisingIdClient");
                                Class<?> adInfoClass = Class.forName("com.google.android.gms.ads.identifier.AdvertisingIdClient$Info");
                                Object advertisingIdClient = advertisingIdClientClass.getMethod("getAdvertisingIdInfo", Context.class).invoke(null, context_.get());

                                adId = (String) adInfoClass.getMethod("getId").invoke(advertisingIdClient);
                                if (adId != null && !adId.equals("00000000-0000-0000-0000-000000000000")) {
                                    DNALogger.i("getAdvertisingId: reflection Ad ID: " + adId);
                                    prefHelper_.setAdid(adId);
                                }
                            }
                        } catch (Exception e) {
                            DNALogger.i("getAdvertisingId: Failed to access Ad ID: " + e.getMessage());
                        }
                    }, 5, TimeUnit.SECONDS);
                }
            }

            if (respJson.has(DNAConstants.SyncParams.CODE) &&
                    respJson.get(DNAConstants.SyncParams.CODE) instanceof String &&
                    !respJson.getString(DNAConstants.SyncParams.CODE).isEmpty() &&
                    !respJson.getString(DNAConstants.SyncParams.CODE).equals("null")) {
                prefHelper_.setAdSelectionCode(respJson.getString(DNAConstants.SyncParams.CODE));
            }

            if (respJson.has(DNAConstants.RequestParams.CODE_VERSION_KEY) &&
                    respJson.get(DNAConstants.RequestParams.CODE_VERSION_KEY) instanceof String &&
                    !respJson.getString(DNAConstants.RequestParams.CODE_VERSION_KEY).isEmpty() &&
                    !respJson.getString(DNAConstants.RequestParams.CODE_VERSION_KEY).equals("null")) {
                prefHelper_.setCodeVersionID(respJson.getString(DNAConstants.RequestParams.CODE_VERSION_KEY));
            }

            if (respJson.has(DNAConstants.SyncParams.ADS) && respJson.get(DNAConstants.SyncParams.ADS) instanceof JSONArray) {
                JSONArray adsList = respJson.getJSONArray(DNAConstants.SyncParams.ADS);
                List<DNAAdRecord> adRecords = new ArrayList<>();
                for (int i = 0; i < adsList.length(); i++) {
                    String adString = adsList.getString(i);
                    adRecords.add(new DNAAdRecord(new JSONObject(adString)));
                }
                DNADatabaseInterface.insertOrUpdateAdRecords(context_.get(), adRecords);
                numAdsAdded = adRecords.size();
            }

            if (respJson.has(DNAConstants.SyncParams.UPDATE_ADS) && respJson.get(DNAConstants.SyncParams.UPDATE_ADS) instanceof JSONArray) {
                JSONArray adsList = respJson.getJSONArray(DNAConstants.SyncParams.UPDATE_ADS);
                DNADatabaseInterface.partialUpdateAdRecordsFromJSON(context_.get(), adsList);
                numAdsUpdated = adsList.length();
            }

            if (respJson.has(DNAConstants.SyncParams.DELETE_ADS) && respJson.get(DNAConstants.SyncParams.DELETE_ADS) instanceof JSONArray) {
                JSONArray deleteAdsList = respJson.getJSONArray(DNAConstants.SyncParams.DELETE_ADS);
                List<String> deleteAds = DNAUtils.convertJSONArrayToStrings(deleteAdsList);
                DNADatabaseInterface.deleteAdRecords(context_.get(), deleteAds);
                numAdsDeleted = deleteAds.size();
            }

            if (respJson.has(DNAConstants.RequestParams.ADS_VERSION_KEY) &&
                    respJson.get(DNAConstants.RequestParams.ADS_VERSION_KEY) instanceof String &&
                    !respJson.getString(DNAConstants.RequestParams.ADS_VERSION_KEY).isEmpty() &&
                    !respJson.getString(DNAConstants.RequestParams.ADS_VERSION_KEY).equals("null")) {
                prefHelper_.setAdBundleID(respJson.getString(DNAConstants.RequestParams.ADS_VERSION_KEY));
            }

            if (respJson.has(DNAConstants.RequestParams.METADATA_VERSION_KEY) &&
                    respJson.get(DNAConstants.RequestParams.METADATA_VERSION_KEY) instanceof String &&
                    !respJson.getString(DNAConstants.RequestParams.METADATA_VERSION_KEY).isEmpty() &&
                    !respJson.getString(DNAConstants.RequestParams.METADATA_VERSION_KEY).equals("null")) {
                prefHelper_.setMetadataVersionID(respJson.getString(DNAConstants.RequestParams.METADATA_VERSION_KEY));
            }

            if (respJson.has(DNAConstants.RequestParams.BLACKLIST_VERSION_KEY) &&
                    respJson.get(DNAConstants.RequestParams.BLACKLIST_VERSION_KEY) instanceof String &&
                    !respJson.getString(DNAConstants.RequestParams.BLACKLIST_VERSION_KEY).isEmpty() &&
                    !respJson.getString(DNAConstants.RequestParams.BLACKLIST_VERSION_KEY).equals("null")) {
                prefHelper_.setBlacklistVersionID(respJson.getString(DNAConstants.RequestParams.BLACKLIST_VERSION_KEY));
            }

            if (respJson.has(DNAConstants.SyncParams.BLACKLIST_APPS) && respJson.get(DNAConstants.SyncParams.BLACKLIST_APPS) instanceof JSONArray) {
                JSONArray blacklistedPackages = respJson.getJSONArray(DNAConstants.SyncParams.BLACKLIST_APPS);
                List<String> blacklistPs = DNAUtils.convertJSONArrayToStrings(blacklistedPackages);
                prefHelper_.setBlacklistPackages(blacklistPs);
            }

            if (respJson.has(DNAConstants.SyncParams.DELETE_SHORTCUTS) && respJson.get(DNAConstants.SyncParams.DELETE_SHORTCUTS) instanceof JSONArray) {
//                DNALogger.i("DNANetworkRequestSync: DELETE_SHORTCUTS: " + respJson.getJSONArray(DNAConstants.SyncParams.DELETE_SHORTCUTS));
                JSONArray deleteShortcutList = respJson.getJSONArray(DNAConstants.SyncParams.DELETE_SHORTCUTS);
                List<String> deleteShortcuts = DNAUtils.convertJSONArrayToStrings(deleteShortcutList);
                DNADatabaseInterface.deleteShortcutRecords(context_.get(), deleteShortcuts);
                numShortcutsDeleted = deleteShortcuts.size();
            }

            if (respJson.has(DNAConstants.SyncParams.SHORTCUTS) && respJson.get(DNAConstants.SyncParams.SHORTCUTS) instanceof JSONArray) {
//                DNALogger.i("DNANetworkRequestSync: SHORTCUTS: " + respJson.getJSONArray(DNAConstants.SyncParams.SHORTCUTS));
                JSONArray sList = respJson.getJSONArray(DNAConstants.SyncParams.SHORTCUTS);
                List<DNAShortcutRecord> shortcutsList = new ArrayList<>();
                Set<String> affectedPackages = new HashSet<>();

                for (int i = 0; i < sList.length(); i++) {
                    JSONObject shortcutObject = sList.getJSONObject(i);
                    DNALogger.i("DNANetworkRequestSync: SHORTCUT: " + shortcutObject);
                    DNAShortcutRecord shortcutRecord = new DNAShortcutRecord(shortcutObject);
                    shortcutsList.add(shortcutRecord);
                    affectedPackages.add(shortcutRecord.packageName);
                }

                for (String packageName : affectedPackages) {
                    DNADatabaseInterface.removeShortcutsFromServer(context_.get(), packageName);
                }

                DNADatabaseInterface.insertOrUpdateShortcutRecords(context_.get(), shortcutsList);
                numShortcutsAdded = shortcutsList.size();
            }

            if (respJson.has(DNAConstants.SyncParams.ALIAS_LIST) && respJson.get(DNAConstants.SyncParams.ALIAS_LIST) instanceof JSONObject) {
                DNALogger.i("DNANetworkRequestSync: ALIAS_LIST: " + respJson.getJSONObject(DNAConstants.SyncParams.ALIAS_LIST));
                JSONObject aliasList = respJson.getJSONObject(DNAConstants.SyncParams.ALIAS_LIST);
                DNADatabaseInterface.updateAliasesForApps(context_.get(), aliasList);
            }

            prefHelper_.setLastApiSyncTime(System.currentTimeMillis());

            JSONObject statsMetadata = new JSONObject();
            try {
                statsMetadata.put("sId", prefHelper_.getSessionID());
                statsMetadata.put("status", 200);
                statsMetadata.put("numAdsAdded", numAdsAdded);
                statsMetadata.put("numAdsUpdated", numAdsUpdated);
                statsMetadata.put("numAdsDeleted", numAdsDeleted);
                statsMetadata.put("numShortcutsAdded", numShortcutsAdded);
                statsMetadata.put("numShortcutsDeleted", numShortcutsDeleted);
                statsMetadata.put("currCodeVersion", currCodeVersion);
                statsMetadata.put("currAdBundleID", currAdBundleID);
                statsMetadata.put("currMetadataVersion", prefHelper_.getMetadataVersionID());
                statsMetadata.put("currBlacklistVersion", prefHelper_.getBlacklistVersionID());
                statsMetadata.put("newCodeVersion", prefHelper_.getCodeVersionID());
                statsMetadata.put("newAdBundleID", prefHelper_.getAdBundleID());
            } catch (Exception je) {
                DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
            }
            long endTime = System.currentTimeMillis();
            DNAStatsLogger.logInternal(context_.get(), "Server Sync", (int)(endTime - startTime), statsMetadata);
        } catch (Exception ex) {
            DNALogger.e("DNANetworkRequestSync: " + ex.getMessage());
        }
    }

    @Override
    public void handleFailure(int statusCode, String causeMsg) {
        DNALogger.e("DNANetworkRequestSync: handleFailure: " + statusCode + " : " + causeMsg);
        DNAPreferences prefs = DNAPreferences.getInstance(context_.get());
        JSONObject statsMetadata = new JSONObject();
        try {
            statsMetadata.put("sId", prefs.getSessionID());
            statsMetadata.put("loc", "DNANetworkRequestSync.handleFailure");
            statsMetadata.put("m", causeMsg);
        } catch (Exception je) {
            DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
        }
        DNAStatsLogger.logInternal(context_.get(), "Error", statusCode, statsMetadata);
    }

    @Override
    public boolean isGetRequest() {
        return true;
    }

}
