package com.devicenative.dna.network;

import android.content.Context;

import com.devicenative.dna.utils.DNAConstants;
import com.devicenative.dna.utils.DNAError;
import com.devicenative.dna.utils.DNALogger;
import com.devicenative.dna.utils.DNAPreferences;

import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;

public abstract class DNANetworkRequest {

    private JSONObject params_;

    final WeakReference<Context> context_;

    final String requestPath_;

    final DNAPreferences prefHelper_;

    /**
     * <p>Creates an instance of ServerRequest.</p>
     *
     * @param context     Application context.
     * @param requestPath Path to server for this request.
     */
    public DNANetworkRequest(Context context, String requestPath) {
        context_ = new WeakReference<>(context);
        requestPath_ = requestPath;
        prefHelper_ = DNAPreferences.getInstance(context);
        params_ = new JSONObject();
    }

    /**
     * <p>Called when execution of this request to server succeeds. Child class should implement
     * its own logic for handling the post request execution.</p>
     *
     * @param response A {@link DNANetworkResponse} object containing server response for this request.
     */
    public abstract void onRequestSucceeded(DNANetworkResponse response);

    /**
     * <p>Called when there is an error on executing this request. Child class should handle the failure
     * accordingly.</p>
     *
     * @param statusCode A {@link Integer} value specifying http return code or any specific error defined in {@link DNAError}.
     * @param causeMsg   A {@link String} value specifying cause for the error if any.
     */
    public abstract void handleFailure(int statusCode, String causeMsg);

    /**
     * Specify whether the request is a GET or POST. Child class has to implement accordingly.
     *
     * @return A {@link Boolean} value specifying if this request is a GET or not.
     */
    public abstract boolean isGetRequest();

    /**
     * <p>Provides the complete url for executing this request. URl consist of API base url and request
     * path. Child class need to extend this method if they need to add specific items to the url </p>
     *
     * @return A url for executing this request against the server.
     */
    public String getRequestUrl() {
        return prefHelper_.getAPIUrl() + requestPath_;
    }

    /**
     * <p>Sets a {@link JSONObject} containing the post data supplied with the current request.</p>
     *
     * @param post A {@link JSONObject} containing the post data supplied with the current request
     *             as key-value pairs.
     */
    protected void setPost(JSONObject post) throws JSONException {
        params_ = post;
    }

    /**
     * <p>Gets a {@link JSONObject} containing the post data supplied with the current request as
     * key-value pairs.</p>
     *
     * @return A {@link JSONObject} containing the post data supplied with the current request
     * as key-value pairs.
     */
    public JSONObject getPost() {
        return params_;
    }


    /**
     * Returns a JsonObject with the parameters that needed to be set with the get request.
     *
     * @return A {@link JSONObject} representation of get request parameters.
     */
    public JSONObject getGetParams() {
        return params_;
    }

    /**
     * Called on UI thread just before executing a request. Do any final updates to the request here.
     * Also attaches any required URL query parameters based on the request type.
     */
    public void onPreExecute() {
        DNALogger.i("onPreExecute called for " + this.getClass().getSimpleName());
    }
}
