package com.devicenative.dna.datasource.loader;

import android.app.usage.UsageStats;
import android.app.usage.UsageStatsManager;
import android.content.Context;
import android.os.RemoteException;

import com.devicenative.dna.db.DNAAppRecord;
import com.devicenative.dna.db.DNADatabaseInterface;
import com.devicenative.dna.db.DNADatabaseQueue;
import com.devicenative.dna.utils.DNALogger;
import com.devicenative.dna.utils.DNAPreferences;
import com.devicenative.dna.utils.DNAStatsLogger;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Calendar;
import java.util.Map;
import java.util.Objects;

public class DNAUsageStatsLoader extends DNALoader {

    public DNAUsageStatsLoader(Context context) {
        super(context);
    }

    @Override
    public void run() {
        DNADatabaseQueue.enqueue(context.get(), () -> {
            syncUsageStats();
        });
    }

    private void syncUsageStats() {
        Context ctx = context.get();
        DNAPreferences prefs = DNAPreferences.getInstance(ctx);
        boolean hasInstallDateBeenRead = prefs.getFirstUsageRead();

        UsageStatsManager usageStatsManager = (UsageStatsManager) ctx.getSystemService(Context.USAGE_STATS_SERVICE);
        if (usageStatsManager == null) { prefs.setUsageStatsEnabled(false); runnableFinished(); return; }

        long time7D = System.currentTimeMillis() - (1000 * 60 * 60 * 24 * 7);
        long time3D = System.currentTimeMillis() - (1000 * 60 * 60 * 24 * 3);
        long time1D = System.currentTimeMillis() - (1000 * 60 * 60 * 24);

        List<UsageStats> queryUsageStats = null;
        try {
            queryUsageStats = usageStatsManager.queryUsageStats(UsageStatsManager.INTERVAL_DAILY, time7D,
                    System.currentTimeMillis());
        } catch (RuntimeException e) {
            DNALogger.e("DNAUsageStatsLoader: queryUsageStats failed: " + e.getMessage());
        }

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.YEAR, -2);

        List<DNAAppRecord> apps = DNADatabaseInterface.getAllAppRecords(ctx);

        Map<String, UsageStats> longTermStatsMap = new HashMap<>();
        try {
            longTermStatsMap = usageStatsManager.queryAndAggregateUsageStats(cal.getTimeInMillis(), System.currentTimeMillis());
        } catch (RuntimeException e) {
            DNALogger.e("DNAUsageStatsLoader: queryAndAggregateUsageStats failed: " + e.getMessage());
        }

        if (queryUsageStats != null && !queryUsageStats.isEmpty() && !prefs.getAppUsageDisabled() && !prefs.getConfigAppUsageDisabled()) {
            prefs.setUsageStatsEnabled(true);

            Map<String, List<UsageStats>> statsMap = new HashMap<>();
            for (UsageStats stats : queryUsageStats) {
                String packageName = stats.getPackageName();
                if (statsMap.containsKey(packageName)) {
                    statsMap.get(packageName).add(stats);
                } else {
                    List<UsageStats> statsList = new ArrayList<>();
                    statsList.add(stats);
                    statsMap.put(packageName, statsList);
                }
            }

            for (DNAAppRecord app : apps) {
                if (statsMap.containsKey(app.packageName)) {
                    List<UsageStats> statsList = statsMap.get(app.packageName);
                    long lastUsedAt = 0;
                    long lastTimeInForeground = 0;
                    long timeUsedInLast7D = 0;
                    long timeUsedInLast3D = 0;
                    long timeUsedInLast1D = 0;
                    for (UsageStats stats : statsList) {
                        if (stats.getLastTimeUsed() > lastUsedAt) {
                            lastUsedAt = stats.getLastTimeUsed();
                        }
                        if (stats.getTotalTimeInForeground() > 0 && stats.getLastTimeUsed() > lastTimeInForeground) {
                            lastTimeInForeground = stats.getLastTimeUsed();
                        }
                        timeUsedInLast7D += stats.getTotalTimeInForeground() / 1000;
                        if (stats.getLastTimeStamp() > time3D) {
                            timeUsedInLast3D += stats.getTotalTimeInForeground() / 1000;
                        }
                        if (stats.getLastTimeStamp() > time1D) {
                            timeUsedInLast1D += stats.getTotalTimeInForeground() / 1000;
                        }
                    }
                    app.lastUsedAt = lastUsedAt;
                    app.lastTimeInForeground = lastTimeInForeground;
                    app.timeUsedInLast7D = timeUsedInLast7D;
                    app.timeUsedInLast3D = timeUsedInLast3D;
                    app.timeUsedInLast1D = timeUsedInLast1D;

                    if (longTermStatsMap.containsKey(app.packageName)) {
                        app.timeUsedInLast3Y = Objects.requireNonNull(longTermStatsMap.get(app.packageName)).getTotalTimeInForeground() / 1000;
                    }
                } else {
                    app.timeUsedInLast7D = 0;
                    app.timeUsedInLast3D = 0;
                    app.timeUsedInLast1D = 0;
                }
            }
            if (!hasInstallDateBeenRead) {
                prefs.setFirstUsageRead(true);
            }

            // update all apps
            DNADatabaseInterface.insertOrUpdateAppRecords(ctx, apps);

            prefs.setLastUsageSyncTime(System.currentTimeMillis());
        } else {
            prefs.setUsageStatsEnabled(false);
        }

        runnableFinished();
    }
}
