package com.devicenative.dna.datasource.loader;

import static android.content.pm.LauncherApps.ShortcutQuery.FLAG_MATCH_DYNAMIC;
import static android.content.pm.LauncherApps.ShortcutQuery.FLAG_MATCH_MANIFEST;
import static android.content.pm.LauncherApps.ShortcutQuery.FLAG_MATCH_PINNED;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.LauncherApps;
import android.content.pm.PackageManager;
import android.content.pm.ShortcutInfo;
import android.os.Build;
import android.os.UserManager;
import android.text.TextUtils;

import com.devicenative.dna.db.DNADatabaseInterface;
import com.devicenative.dna.db.DNAShortcutRecord;
import com.devicenative.dna.utils.DNAConstants;
import com.devicenative.dna.utils.DNALogger;
import com.devicenative.dna.utils.DNAPreferences;
import com.devicenative.dna.utils.DNAStatsLogger;

import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

public class DNAShortcutLoader extends DNALoader {
    public DNAShortcutLoader(Context context) {
        super(context);
    }

    @Override
    public void run() {
        long start = System.currentTimeMillis();
        Context ctx = context.get();

        DNAPreferences prefs = DNAPreferences.getInstance(ctx);

        List<DNAShortcutRecord> shortcutRecords = new ArrayList<>();
        List<ShortcutInfo> shortcutInfoList = new ArrayList<>();
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            try {
                UserManager manager = (UserManager) ctx.getSystemService(Context.USER_SERVICE);
                LauncherApps launcherApps = (LauncherApps) ctx.getSystemService(Context.LAUNCHER_APPS_SERVICE);

                if (launcherApps.hasShortcutHostPermission()) {
                    LauncherApps.ShortcutQuery shortcutQuery = new LauncherApps.ShortcutQuery();
                    shortcutQuery.setQueryFlags(FLAG_MATCH_DYNAMIC | FLAG_MATCH_MANIFEST | FLAG_MATCH_PINNED);

                    for (android.os.UserHandle profile : manager.getUserProfiles()) {
                        if (manager.isUserUnlocked(profile)) {
                            List<ShortcutInfo> shortcutsRaw = launcherApps.getShortcuts(shortcutQuery, profile);
                            if (shortcutsRaw != null) {
                                shortcutInfoList.addAll(shortcutsRaw);
                            }
                        }
                    }
                }
            } catch (SecurityException e) {
                DNALogger.e("DNAShortcutLoader: Unable to get all shortcuts: " + e.getMessage());
            }

            PackageManager packageManager = ctx.getPackageManager();

            for (ShortcutInfo sInfo : shortcutInfoList) {
                if (Thread.currentThread().isInterrupted()) {
                    return;
                }

                DNAShortcutRecord shortcutRecord = new DNAShortcutRecord();
                shortcutRecord.intentUri = sInfo.getId();
                shortcutRecord.packageName = sInfo.getPackage();
                shortcutRecord.source = sInfo.isDynamic() ? 1 : 0;
                shortcutRecord.rank = sInfo.getRank();
                shortcutRecord.updatedAt = sInfo.getLastChangedTimestamp();

                String appName = "";
                try {
                    ApplicationInfo info = packageManager.getApplicationInfo(shortcutRecord.packageName, PackageManager.GET_META_DATA);
                    appName = packageManager.getApplicationLabel(info).toString();
                } catch (PackageManager.NameNotFoundException e) {
                    DNALogger.e("DNAShortcutLoader: Unable to get name of app for shortcut with package name: " + shortcutRecord.packageName);
                }

                if (sInfo.getShortLabel() != null) {
                    if (!TextUtils.isEmpty(appName)) {
                        shortcutRecord.name = appName + ": " + sInfo.getShortLabel().toString();
                    } else {
                        shortcutRecord.name = sInfo.getShortLabel().toString();
                    }
                }

                if (sInfo.getLongLabel() != null) {
                    shortcutRecord.description = sInfo.getLongLabel().toString();
                }

                if (shortcutRecord.name == null || shortcutRecord.name.isEmpty()) {
                    if (!TextUtils.isEmpty(appName) && !TextUtils.isEmpty(shortcutRecord.description)) {
                        shortcutRecord.name = appName + ": " + shortcutRecord.description;
                        shortcutRecord.description = null;
                    } else if (!TextUtils.isEmpty(shortcutRecord.description)) {
                        shortcutRecord.name = shortcutRecord.description;
                        shortcutRecord.description = null;
                    }
                }

                if (shortcutRecord.name == null || shortcutRecord.name.isEmpty()) {
                    continue;
                }

                shortcutRecords.add(shortcutRecord);
            }

            DNADatabaseInterface.insertOrUpdateShortcutRecords(ctx, shortcutRecords);
        }

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.O || shortcutRecords.size() > 0) {
            prefs.setShortcutsLoaded();
        }

        runnableFinished();
    }
}
